import torch, math
import torch.nn as nn
import torch.nn.functional as F
from einops import rearrange, repeat
from functools import partial
from typing import Optional, Callable, Any
from collections import OrderedDict
from mamba_ssm.ops.selective_scan_interface import selective_scan_fn
from timm.models.layers import DropPath, trunc_normal_, to_2tuple
DropPath.__repr__ = lambda self: f"timm.DropPath({self.drop_prob})"


class cwFModule(nn.Module):
    def __init__(self, feature_channels, output_dim):
        super(cwFModule, self).__init__()
        self.feature_channels = feature_channels
        self.intermediate_dim = feature_channels  # intermediate_dim is set to the feature_channels
        self.output_dim = output_dim

        # Layers for the transformation and fusion process
        self.conv3x3 = nn.Conv2d(feature_channels * 2, self.intermediate_dim, kernel_size=3, padding=1)
        self.conv1x1 = nn.Conv2d(self.intermediate_dim, output_dim, kernel_size=1)
        self.sigmoid = nn.Sigmoid()
        self.global_avg_pool = nn.AdaptiveAvgPool2d(1)

    def forward(self, e1, y1):
        # Concatenate transformer features and CNN features
        concat_features = torch.cat((e1, y1), dim=1)  # Assuming both e1 and y1 are of shape (N, C, H, W)

        # Apply 3x3 convolution
        g = self.conv3x3(concat_features)

        # Apply 1x1 convolution
        g = self.conv1x1(g)

        # Apply Global Average Pooling
        gap = self.global_avg_pool(g)

        # Apply sigmoid activation
        m = self.sigmoid(gap)

        # Perform element-wise multiplication
        t_weighted = e1 * m
        f_weighted = y1 * m

        # Combine the weighted features
        z_cwf = t_weighted + f_weighted

        return z_cwf

class MPCA(nn.Module):
    def __init__(self, in_channels):
        super(MPCA, self).__init__()
        self.in_channels = in_channels
        self.global_avg_pool = nn.AdaptiveAvgPool2d(1)
        self.conv1 = nn.Conv2d(in_channels, in_channels, kernel_size=1)
        self.conv_merge = nn.Conv2d(in_channels * 2, in_channels * 2, kernel_size=1)
        self.conv_split = nn.Conv2d(in_channels * 2, in_channels * 2, kernel_size=1)
        self.sigmoid = nn.Sigmoid()

    def forward(self, F1, F2):
        F1_prime = self.global_avg_pool(F1)
        F1_prime = self.conv1(F1_prime)
        F2_prime = self.global_avg_pool(F2)
        F2_prime = self.conv1(F2_prime)
        F_concat = torch.cat((F1_prime, F2_prime), dim=1)
        A = self.conv_merge(F_concat)
        A = self.sigmoid(A)
        A1, A2 = torch.split(A, self.in_channels, dim=1)
        F1_hat = F1 * A1
        F2_hat = F2 * A2
        F_out = F1_hat + F2_hat
        return F_out


class SS2D(nn.Module):
    def __init__(
        self,
        d_model,
        d_state=16,
        d_conv=3,
        ssm_ratio=2,
        dt_rank="auto",        
        # ======================
        dropout=0.,
        conv_bias=True,
        bias=False,
        dtype=None,
        # ======================
        dt_min=0.001,
        dt_max=0.1,
        dt_init="random",
        dt_scale=1.0,
        dt_init_floor=1e-4,
        # ======================
        shared_ssm=False,
        softmax_version=False,
        # ======================
        **kwargs,
    ):
        factory_kwargs = {"device": None, "dtype": dtype}
        super().__init__()
        self.softmax_version = softmax_version
        self.d_model = d_model
        self.d_state = math.ceil(self.d_model / 6) if d_state == "auto" else d_state # 20240109
        self.d_conv = d_conv
        self.expand = ssm_ratio
        self.d_inner = int(self.expand * self.d_model)
        self.dt_rank = math.ceil(self.d_model / 16) if dt_rank == "auto" else dt_rank
        self.K = 4 if not shared_ssm else 1

        self.in_proj = nn.Linear(self.d_model, self.d_inner * 2, bias=bias, **factory_kwargs)
        self.conv2d = nn.Conv2d(
            in_channels=self.d_inner,
            out_channels=self.d_inner,
            groups=self.d_inner,
            bias=conv_bias,
            kernel_size=d_conv,
            padding=(d_conv - 1) // 2,
            **factory_kwargs,
        )
        self.act = nn.SiLU()

        self.x_proj = [
            nn.Linear(self.d_inner, (self.dt_rank + self.d_state * 2), bias=False, **factory_kwargs)
            for _ in range(self.K)
        ]
        self.x_proj_weight = nn.Parameter(torch.stack([t.weight for t in self.x_proj], dim=0)) # (K, N, inner)
        del self.x_proj

        self.dt_projs = [
            self.dt_init(self.dt_rank, self.d_inner, dt_scale, dt_init, dt_min, dt_max, dt_init_floor, **factory_kwargs)
            for _ in range(self.K)
        ]
        self.dt_projs_weight = nn.Parameter(torch.stack([t.weight for t in self.dt_projs], dim=0)) # (K, inner, rank)
        self.dt_projs_bias = nn.Parameter(torch.stack([t.bias for t in self.dt_projs], dim=0)) # (K * inner)
        del self.dt_projs
        
        self.A_logs = self.A_log_init(self.d_state, self.d_inner, copies=self.K, merge=True) # (K * D, N)
        self.Ds = self.D_init(self.d_inner, copies=self.K, merge=True) # (K * D)

        if not self.softmax_version:
            self.out_norm = nn.LayerNorm(self.d_inner)
        self.out_proj = nn.Linear(self.d_inner, self.d_model, bias=bias, **factory_kwargs)
        self.dropout = nn.Dropout(dropout) if dropout > 0. else None
        self.freq_module = FrequencyDomainModule(d_model)

        # self.cwf = cwFModule(2*d_model, 2*d_model)
        self.mpca = MPCA(2*d_model)
    @staticmethod
    def dt_init(dt_rank, d_inner, dt_scale=1.0, dt_init="random", dt_min=0.001, dt_max=0.1, dt_init_floor=1e-4, **factory_kwargs):
        dt_proj = nn.Linear(dt_rank, d_inner, bias=True, **factory_kwargs)

        # Initialize special dt projection to preserve variance at initialization
        dt_init_std = dt_rank**-0.5 * dt_scale
        if dt_init == "constant":
            nn.init.constant_(dt_proj.weight, dt_init_std)
        elif dt_init == "random":
            nn.init.uniform_(dt_proj.weight, -dt_init_std, dt_init_std)
        else:
            raise NotImplementedError

        # Initialize dt bias so that F.softplus(dt_bias) is between dt_min and dt_max
        dt = torch.exp(
            torch.rand(d_inner, **factory_kwargs) * (math.log(dt_max) - math.log(dt_min))
            + math.log(dt_min)
        ).clamp(min=dt_init_floor)
        # Inverse of softplus: https://github.com/pytorch/pytorch/issues/72759
        inv_dt = dt + torch.log(-torch.expm1(-dt))
        with torch.no_grad():
            dt_proj.bias.copy_(inv_dt)
        # Our initialization would set all Linear.bias to zero, need to mark this one as _no_reinit
        dt_proj.bias._no_reinit = True
        
        return dt_proj

    @staticmethod
    def A_log_init(d_state, d_inner, copies=-1, device=None, merge=True):
        # S4D real initialization
        A = repeat(
            torch.arange(1, d_state + 1, dtype=torch.float32, device=device),
            "n -> d n",
            d=d_inner,
        ).contiguous()
        A_log = torch.log(A)  # Keep A_log in fp32
        if copies > 0:
            A_log = repeat(A_log, "d n -> r d n", r=copies)
            if merge:
                A_log = A_log.flatten(0, 1)
        A_log = nn.Parameter(A_log)
        A_log._no_weight_decay = True
        return A_log

    @staticmethod
    def D_init(d_inner, copies=-1, device=None, merge=True):
        # D "skip" parameter
        D = torch.ones(d_inner, device=device)
        if copies > 0:
            D = repeat(D, "n1 -> r n1", r=copies)
            if merge:
                D = D.flatten(0, 1)
        D = nn.Parameter(D)  # Keep in fp32
        D._no_weight_decay = True
        return D

    def forward_corev0(self, x: torch.Tensor):
        self.selective_scan = selective_scan_fn

        B, C, H, W = x.shape
        L = H * W
        K = 4

        x_hwwh = torch.stack([x.view(B, -1, L), torch.transpose(x, dim0=2, dim1=3).contiguous().view(B, -1, L)], dim=1).view(B, 2, -1, L)
        xs = torch.cat([x_hwwh, torch.flip(x_hwwh, dims=[-1])], dim=1) # (b, k, d, l)

        x_dbl = torch.einsum("b k d l, k c d -> b k c l", xs, self.x_proj_weight)
        # x_dbl = x_dbl + self.x_proj_bias.view(1, K, -1, 1)
        dts, Bs, Cs = torch.split(x_dbl, [self.dt_rank, self.d_state, self.d_state], dim=2)
        dts = torch.einsum("b k r l, k d r -> b k d l", dts, self.dt_projs_weight)

        xs = xs.float().view(B, -1, L) # (b, k * d, l)
        dts = dts.contiguous().float().view(B, -1, L) # (b, k * d, l)
        Bs = Bs.float() # (b, k, d_state, l)
        Cs = Cs.float() # (b, k, d_state, l)
        
        As = -torch.exp(self.A_logs.float()) # (k * d, d_state)
        Ds = self.Ds.float() # (k * d)
        dt_projs_bias = self.dt_projs_bias.float().view(-1) # (k * d)

        out_y = self.selective_scan(
            xs, dts, 
            As, Bs, Cs, Ds, z=None,
            delta_bias=dt_projs_bias,
            delta_softplus=True,
            return_last_state=False,
        ).view(B, K, -1, L)
        assert out_y.dtype == torch.float

        inv_y = torch.flip(out_y[:, 2:4], dims=[-1]).view(B, 2, -1, L)
        wh_y = torch.transpose(out_y[:, 1].view(B, -1, W, H), dim0=2, dim1=3).contiguous().view(B, -1, L)
        invwh_y = torch.transpose(inv_y[:, 1].view(B, -1, W, H), dim0=2, dim1=3).contiguous().view(B, -1, L)
        y = out_y[:, 0] + inv_y[:, 0] + wh_y + invwh_y
        y = torch.transpose(y, dim0=1, dim1=2).contiguous().view(B, H, W, -1)
        y = self.out_norm(y)

        return y
    
    forward_core = forward_corev0
    def forward(self, x: torch.Tensor, **kwargs):          #64
        xz = self.in_proj(x)      #c*4            #256
        x, z = xz.chunk(2, dim=-1) # (b, h, w, d)  #128    
        x = x.permute(0, 3, 1, 2).contiguous()   #(shape = x*2(C)
        x_fft = self.freq_module(x) 
        x_fft = x_fft.permute(0, 2, 3, 1).contiguous() 
        x = self.act(self.conv2d(x)) # (b, d, h, w)                   #2*128*128*128
        y = self.forward_core(x)
        y = y.permute(0, 3, 1, 2)
        x_fft = x_fft.permute(0, 3, 1, 2)    
         
        y=self.mpca(y,x_fft)          #

        y = y.permute(0, 2, 3, 1)
        y = y * F.silu(z)
        out = self.out_proj(y)   
        if self.dropout is not None:
            out = self.dropout(out)
        return out


class FrequencyDomainModule(nn.Module):
    def __init__(self, input_channels, filter_size=5, scale_factors=[1, 2, 4]):
        super(FrequencyDomainModule, self).__init__()
        self.input_channels = input_channels
        self.filter_size = filter_size
        self.scale_factors = scale_factors
        self.conv = nn.Conv1d(input_channels*2,input_channels*2, kernel_size=filter_size, padding=filter_size // 2, groups=input_channels*2)
        self.norm = nn.LayerNorm(input_channels)
    def forward(self, x):
        B, C, H, W = x.shape
        x = x.view(B * C, H * W)
        x_freq = torch.fft.fft(x, dim=-1) 
        multi_scale_freq = []
        max_size = 0  
        for scale in self.scale_factors:
            scaled_freq = torch.fft.fft(x, dim=-1, n=(H * W) // scale)
            scaled_freq = torch.fft.fftshift(scaled_freq, dim=-1) 
            multi_scale_freq.append(scaled_freq)
            max_size = max(max_size, scaled_freq.shape[-1])
        for i in range(len(multi_scale_freq)):
            pad_size = max_size - multi_scale_freq[i].shape[-1]
            multi_scale_freq[i] = F.pad(multi_scale_freq[i], (0, pad_size))   
        multi_scale_tensor = torch.stack(multi_scale_freq, dim=0)
        x_freq_fusion = multi_scale_tensor.mean(dim=0)
        x_freq_real = x_freq_fusion.real.view(B, C, H * W)
        x_freq_filtered = self.conv(x_freq_real)
        x_freq_filtered = x_freq_filtered.view(B, C, H, W)
        return x_freq_filtered


class Mlp(nn.Module):
    def __init__(self, in_features, hidden_features=None, out_features=None, act_layer=nn.GELU, drop=0.,channels_first=False):
        super().__init__()
        out_features = out_features or in_features
        hidden_features = hidden_features or in_features

        Linear = partial(nn.Conv2d, kernel_size=1, padding=0) if channels_first else nn.Linear
        self.fc1 = Linear(in_features, hidden_features)
        self.act = act_layer()
        self.fc2 = Linear(hidden_features, out_features)
        self.drop = nn.Dropout(drop)

    def forward(self, x):
        x = self.fc1(x)
        x = self.act(x)
        x = self.drop(x)
        x = self.fc2(x)
        x = self.drop(x)
        return x


class STMBlock(nn.Module):
    def __init__(
        self,
        hidden_dim: int = 0,
        drop_path: float = 0,
        norm_layer: Callable[..., torch.nn.Module] = partial(nn.LayerNorm, eps=1e-6),
        attn_drop_rate: float = 0,
        d_state: int = 16,
        dt_rank: Any = "auto",
        ssm_ratio=2.0,
        shared_ssm=False,
        softmax_version=False,
        mlp_ratio=4.0,
        act_layer=nn.GELU,
        drop: float = 0.0,
        **kwargs,
    ):
        super().__init__()
        self.norm = norm_layer(hidden_dim)
        self.op = SS2D(
            d_model=hidden_dim, 
            dropout=attn_drop_rate, 
            d_state=d_state, 
            ssm_ratio=ssm_ratio, 
            dt_rank=dt_rank,
            shared_ssm=shared_ssm,
            softmax_version=softmax_version,
            **kwargs
        )
        self.drop_path = DropPath(drop_path)
        
        self.mlp_branch = mlp_ratio > 0
        if self.mlp_branch:
            self.norm2 = norm_layer(hidden_dim)
            mlp_hidden_dim = int(hidden_dim * mlp_ratio)
            self.mlp = Mlp(in_features=hidden_dim, hidden_features=mlp_hidden_dim, act_layer=act_layer, drop=drop, channels_first=False)


    def forward(self, input: torch.Tensor):
        batch_size, channels, height, width = input.shape
        input_reshaped = input.permute(0, 2, 3, 1)

        x = input_reshaped + self.drop_path(self.op(self.norm(input_reshaped)))
        if self.mlp_branch:
            x = x + self.drop_path(self.mlp(self.norm2(x)))  # FFN
        
        x = x.permute(0, 3, 1, 2)
        return x


class PatchExpand(nn.Module):
    def __init__(self, input_resolution, dim, dim_scale=2, norm_layer=nn.LayerNorm):
        super().__init__()
        self.dim = dim
        self.expand = nn.Linear(dim, 2*dim, bias=False) if dim_scale==2 else nn.Identity()
        self.norm = norm_layer(dim // dim_scale)

    def forward(self, x):
        x = x.permute(0, 2, 3, 1)  # B, C, H, W ==> B, H, W, C
        x = self.expand(x)
        B, H, W, C = x.shape

        x = x.view(B, H, W, C)
        x = rearrange(x, 'b h w (p1 p2 c)-> b (h p1) (w p2) c', p1=2, p2=2, c=C//4)
        x = x.view(B,-1,C//4)
        x = self.norm(x)
        x = x.reshape(B, H*2, W*2, C//4)

        return x

class FinalPatchExpand_X4(nn.Module):
    def __init__(self, input_resolution, dim, dim_scale=4, norm_layer=nn.LayerNorm):
        super().__init__()
        # self.input_resolution = input_resolution
        self.dim = dim
        self.dim_scale = dim_scale
        self.expand = nn.Linear(dim, 16*dim, bias=False)
        self.output_dim = dim 
        self.norm = norm_layer(self.output_dim)

    def forward(self, x):
        # H, W = self.input_resolution
        x = x.permute(0, 2, 3, 1)  # B, C, H, W ==> B, H, W, C
        x = self.expand(x)
        B, H, W, C = x.shape
        
        x = rearrange(x, 'b h w (p1 p2 c)-> b (h p1) (w p2) c', p1=self.dim_scale, p2=self.dim_scale, c=C//(self.dim_scale**2))
        x = x.view(B,-1,self.output_dim)
        x = self.norm(x)
        x = x.reshape(B, H*self.dim_scale, W*self.dim_scale, self.output_dim)

        return x
