import torch
from collections import OrderedDict

class SGD(torch.optim.Optimizer):
    """Modified SGD which allows projection"""

    def __init__(self, params, lr=0.1, momentum=0, dampening=0, nesterov=False,
                 weight_decay=0, weight_decay_ord=2, global_constraint=None, gradient_callback=None):
        momentum = momentum or 0
        dampening = dampening or 0
        weight_decay = weight_decay or 0
        weight_decay_ord = float(weight_decay_ord) if weight_decay_ord is not None else 2
        if not weight_decay_ord >= 1:
            raise ValueError(f"Invalid weight_decay order: {weight_decay_ord}.")
        if lr < 0.0:
            raise ValueError("Invalid learning rate: {}".format(lr))
        if not (0.0 <= momentum <= 1.0):
            raise ValueError("Momentum must be between 0 and 1.")
        if not (0.0 <= dampening <= 1.0):
            raise ValueError("Dampening must be between 0 and 1.")
        if weight_decay < 0.0:
            raise ValueError("Invalid weight_decay value: {}".format(weight_decay))
        if nesterov and (momentum == 0 or dampening != 0):
            raise ValueError("Nesterov momentum requires momentum and zero dampening")

        self.global_constraint = global_constraint  # If not None, this points to the constraint instance
        self.gradient_callback = gradient_callback # If not None, call this function right before the update step

        defaults = dict(lr=lr, momentum=momentum, dampening=dampening,
                        weight_decay=weight_decay, weight_decay_ord=weight_decay_ord, nesterov=nesterov)
        super(SGD, self).__init__(params, defaults)

    def __setstate__(self, state):
        super(SGD, self).__setstate__(state)
        for group in self.param_groups:
            group.setdefault('nesterov', False)

    @torch.no_grad()
    def reset_momentum(self):
        """Resets momentum, typically used directly after pruning"""
        for group in self.param_groups:
            momentum = group['momentum']
            if momentum > 0:
                for p in group['params']:
                    param_state = self.state[p]
                    if 'momentum_buffer' in param_state: del param_state['momentum_buffer']

    @torch.no_grad()
    def step(self, closure=None):
        """Performs a single optimization step.
        Args:
            closure (callable, optional): A closure that reevaluates the model
                and returns the loss.
        """
        loss = None
        if closure is not None:
            with torch.enable_grad():
                loss = closure()

        grad_list = []  # Collect gradients if we need to call the gradient_callback
        param_dict = OrderedDict()
        for i in range(len(self.param_groups)):
            param_dict[i] = []

        # For Linf regularization, we need to compute the maximal element out of all parameters
        for group in self.param_groups:
            if group['weight_decay'] > 0 and group['weight_decay_ord'] == float('inf'):
                group['maxParam'] = max(float(torch.max(torch.abs(p)))
                                        for p in group['params'] if p.grad is not None)

        for idx, group in enumerate(self.param_groups):
            weight_decay = group['weight_decay']
            weight_decay_ord = group['weight_decay_ord']
            momentum = group['momentum']
            dampening = group['dampening']
            nesterov = group['nesterov']

            for p in group['params']:
                if p.grad is None:
                    continue
                d_p = p.grad

                if weight_decay > 0:
                    if weight_decay_ord == 1:
                        # L1 regularization
                        d_p = d_p.add(torch.sign(p), alpha=weight_decay)
                    elif weight_decay_ord == 2:
                        # L2 regularization
                        d_p = d_p.add(p, alpha=weight_decay)
                    elif weight_decay_ord == float('inf'):
                        # Linf regularization
                        maxParam = group['maxParam']
                        d_p = d_p.add(torch.sign(p) * (torch.abs(p) == maxParam), alpha=weight_decay)
                    else:
                        # Arbitrary Lp regularization when p ist not 1, 2 or inf
                        d_p = d_p.add(torch.sign(p) * torch.abs(p).pow(weight_decay_ord - 1), alpha=weight_decay)
                if momentum > 0:
                    param_state = self.state[p]
                    if 'momentum_buffer' not in param_state:
                        buf = param_state['momentum_buffer'] = torch.clone(d_p).detach()
                    else:
                        buf = param_state['momentum_buffer']
                        buf.mul_(momentum).add_(d_p, alpha=1 - dampening)
                    if nesterov:
                        d_p = d_p.add(buf, alpha=momentum)
                    else:
                        d_p = buf
                if self.gradient_callback is not None:
                    grad_list.append(d_p.view(-1))
                    param_dict[idx].append(p)
                else:
                    p.add_(d_p, alpha=-group['lr'])

                    # Project if necessary
                    if not self.global_constraint:  # We have to project afterwards
                        if hasattr(p, 'constraint'):
                            p.copy_(p.constraint.euclidean_project(p))

        if self.gradient_callback is not None:
            d_p = self.gradient_callback(torch.cat(grad_list))
            for idx in param_dict:
                group = self.param_groups[idx]
                for p in param_dict[idx]:
                    numberOfElements = p.numel()
                    p.add_(d_p[:numberOfElements].view(p.shape), alpha=-group['lr'])
                    d_p = d_p[numberOfElements:]
                    # Project if necessary
                    if not self.global_constraint:  # We have to project afterwards
                        if hasattr(p, 'constraint'):
                            p.copy_(p.constraint.euclidean_project(p))

        if self.global_constraint:
            # Project entire gradient vector
            assert len(self.param_groups) == 1, "This does not work for multiple param_groups yet."
            param_dict = [p for p in self.param_groups[0]['params'] if p.grad is not None]
            p_proj = self.global_constraint.euclidean_project(torch.cat([p.view(-1) for p in param_dict]))
            for p in param_dict:
                numberOfElements = p.numel()
                p.copy_(p_proj[:numberOfElements].view(p.shape))
                p_proj = p_proj[numberOfElements:]

        return loss
