import copy
from omegaconf import DictConfig
from pado.core.base.optimizer import PadoOptimizer, PadoOptimizerList

OPTIMIZER_REGISTRY = {}  # {name: optimizer_cls}

__all__ = ["register_optimizer", "build_optimizer", PadoOptimizerList]


def register_optimizer(name: str):
    """Decorator for register optimizer."""

    def register_optimizer_cls(cls):
        if name in OPTIMIZER_REGISTRY:
            raise ValueError(f"Cannot register duplicated optimizer {name}.")

        OPTIMIZER_REGISTRY[name] = cls
        return cls

    return register_optimizer_cls


# ------------------------------------------------------------------------  #
from pado.optim.sgd import SGD  # noqa
from pado.optim.adam import Adam  # noqa
from pado.optim.lamb import Lamb  # noqa


# ------------------------------------------------------------------------  #

def build_optimizer(cfg: DictConfig, params):
    """
    Build a single optimizer.
    """
    if "name" not in cfg:
        raise ValueError("Optimizer config dict should have name.")

    name = cfg["name"]
    cfg = copy.deepcopy(cfg)
    del cfg["name"]

    if name in OPTIMIZER_REGISTRY:
        optimizer_cls = OPTIMIZER_REGISTRY[name]
        optimizer_cls: PadoOptimizer
        optimizer = optimizer_cls.from_config(cfg, params)
    else:
        raise ValueError(f"Optimizer {name} not found.")
    return optimizer
