import math
from omegaconf import DictConfig, OmegaConf

from pado.core.base.optimizer import PadoOptimizer
from pado.core.base.lr_scheduler import PadoScheduler
from pado.optim.lr_scheduler import register_scheduler

__all__ = ["ExponentialLR"]


@register_scheduler("ExponentialLR")
class ExponentialLR(PadoScheduler):

    def __init__(self,
                 optimizer: PadoOptimizer,
                 decay_factor: float = 0.999,
                 warmup_iters: int = 0,
                 keep_iters: int = 0,
                 min_lr: float = 1e-8,
                 mode: str = "min") -> None:
        super().__init__(optimizer, warmup_iters, keep_iters, min_lr, mode)
        if decay_factor >= 1.0:
            raise ValueError(f"ExponentialLR decay_factor {decay_factor} >= 1.0.")
        self.decay_factor = decay_factor

    def state_dict(self) -> dict:
        d = super().state_dict()
        d["decay_factor"] = self.decay_factor
        return d

    def load_state_dict(self, state_dict: dict) -> None:
        super().load_state_dict(state_dict)
        self.decay_factor = state_dict.get("decay_factor", 0.999)

    def _get_lr(self, initial_lr: float, param_group_index=None, **kwargs) -> float:
        if initial_lr <= self.min_lr:
            return initial_lr

        if self.num_iters < self.warmup_iters:
            lr = initial_lr * (self.num_iters + 1) / self.warmup_iters
        elif self.num_iters < self.warmup_iters + self.keep_iters:
            lr = initial_lr
        else:
            curr_iters = self.num_iters - self.warmup_iters - self.keep_iters
            lr = self.min_lr + (initial_lr - self.min_lr) * math.pow(self.decay_factor, curr_iters)
        return lr

    @classmethod
    def from_config(cls, cfg: DictConfig, optimizer: PadoOptimizer) -> "ExponentialLR":
        cfg = OmegaConf.to_container(cfg, resolve=True)
        return cls(optimizer, **cfg)
