import copy
from omegaconf import DictConfig
from pado.core.base.lr_scheduler import PadoScheduler, PadoSchedulerList

SCHEDULER_REGISTRY = {}  # {name: scheduler_cls}

__all__ = ["register_scheduler", "build_scheduler", PadoSchedulerList]


def register_scheduler(name: str):
    """Decorator for register scheduler."""

    def register_scheduler_cls(cls):
        if name in SCHEDULER_REGISTRY:
            raise ValueError(f"Cannot register duplicated scheduler {name}.")

        SCHEDULER_REGISTRY[name] = cls
        return cls

    return register_scheduler_cls


# ------------------------------------------------------------------------  #
from pado.optim.lr_scheduler.constant import ConstantLR  # noqa
from pado.optim.lr_scheduler.cosine import CosineLR  # noqa
from pado.optim.lr_scheduler.exponential import ExponentialLR  # noqa
from pado.optim.lr_scheduler.inv_sqrt import InvSqrtLR  # noqa
from pado.optim.lr_scheduler.reduce_plateau import ReduceLROnPlateau  # noqa
from pado.optim.lr_scheduler.step import StepLR  # noqa


# ------------------------------------------------------------------------  #

def build_scheduler(cfg: DictConfig, optimizer):
    """
    Build a single scheduler.
    """
    if "name" not in cfg:
        raise ValueError("Scheduler config dict should have name.")

    name = cfg["name"]
    cfg = copy.deepcopy(cfg)
    del cfg["name"]

    if name in SCHEDULER_REGISTRY:
        scheduler_cls = SCHEDULER_REGISTRY[name]
        scheduler_cls: PadoScheduler
        scheduler = scheduler_cls.from_config(cfg, optimizer)
    else:
        raise ValueError(f"Scheduler {name} not found.")
    return scheduler
