import math
from omegaconf import DictConfig, OmegaConf

from pado.core.base.optimizer import PadoOptimizer
from pado.core.base.lr_scheduler import PadoScheduler
from pado.optim.lr_scheduler import register_scheduler

__all__ = ["InvSqrtLR"]


@register_scheduler("InvSqrtLR")
class InvSqrtLR(PadoScheduler):

    def __init__(self,
                 optimizer: PadoOptimizer,
                 warmup_iters: int = 0,
                 keep_iters: int = 0,
                 min_lr: float = 1e-8,
                 mode: str = "min") -> None:
        super().__init__(optimizer, warmup_iters, keep_iters, min_lr, mode)

    def _get_lr(self, initial_lr: float, param_group_index=None, **kwargs) -> float:
        if initial_lr <= self.min_lr:
            return initial_lr

        if self.num_iters < self.warmup_iters:
            lr = initial_lr * (self.num_iters + 1) / self.warmup_iters
        elif self.num_iters < self.warmup_iters + self.keep_iters:
            lr = initial_lr
        else:
            # following FairSeq implementation of InverseSquareRootSchedule.
            curr_iters = self.num_iters - self.keep_iters
            scale = float(math.sqrt(self.warmup_iters + 1)) / float(math.sqrt((curr_iters + 1)))
            lr = self.min_lr + (initial_lr - self.min_lr) * min(scale, 1.0)
        return lr

    @classmethod
    def from_config(cls, cfg: DictConfig, optimizer: PadoOptimizer) -> "InvSqrtLR":
        cfg = OmegaConf.to_container(cfg, resolve=True)
        return cls(optimizer, **cfg)

