import os
import warnings
from typing import Dict, List

import neptune.new as neptune
import numpy as np
from acme.utils.loggers import base
from datetime import datetime


def format_key(key: str) -> str:
    """Internal function for formatting keys in Tensorboard format."""
    return key.title().replace("_", "")


class NeptuneLogger(base.Logger):
    def __init__(
        self,
        label: str,
        tag: str,
        name: str = str(datetime.now()),
        exp_params: Dict = {},
        project: str = "project_name",
        # Logging hardware metrics fails with nvidia migs
        capture_hardware_metrics: bool = True,
    ):
        """Initialise a logger for logging experiment metrics
            to Neptune.

        Args:
            label: identifier indicating what process the logger was created for.
                eg. executor, evaluator or trainer.
            project: Namespace and project name to be logged to. This will usually be
                something like `f"NeptuneOrganisation/{your_project_name}"`.
            tag: a tag for separating experiments from eachother. This is useful
                for grouping, aggregating and plotting experiment results later on.
            name: unique ID from logging to Neptune defaults to the current
                date and time.
            exp_params: all parameters of the current experiment.
            capture_hardware_metrics: whether machine hardware metrics should be
                logged to Neptune.
        """

        self._label = label
        self._name = name
        self._exp_params = exp_params
        self._api_token = os.getenv("NEPTUNE_API_TOKEN")
        self._project = project
        self._tag = tag

        self._run = neptune.init(
            name=self._name,
            monitoring_namespace=f"monitoring/{self._label}",
            api_token=self._api_token,
            project=self._project,
            tags=self._tag,
            capture_hardware_metrics=capture_hardware_metrics,
        )
        self._run["params"] = self._exp_params  # type: ignore

    def write(self, values: base.LoggingData) -> None:
        try:
            if isinstance(values, dict):
                for key, value in values.items():
                    is_scalar_array = hasattr(value, "shape") and (
                        value.shape == [1] or value.shape == 1 or value.shape == ()
                    )
                    if np.isscalar(value):
                        self.scalar_summary(key, value)
                    elif is_scalar_array:
                        if hasattr(value, "item"):
                            self.scalar_summary(key, value.item())
                        else:
                            self.scalar_summary(key, value)
                    elif hasattr(value, "shape"):
                        self.histogram_summary(key, value)
                    elif isinstance(value, dict):
                        flatten_dict = self._flatten_dict(
                            parent_key=key, dict_info=value
                        )
                        self.write(flatten_dict)
                    elif isinstance(value, tuple) or isinstance(value, list):
                        for index, elements in enumerate(value):
                            self.write({f"{key}_info_{index}": elements})
                    else:
                        warnings.warn(
                            f"Unable to log: {key}, unknown type: {type(value)}"
                        )
            elif isinstance(values, tuple) or isinstance(value, list):
                for elements in values:
                    self.write(elements)
            else:
                warnings.warn(f"Unable to log: {values}, unknown type: {type(values)}")
        except Exception as ex:
            warnings.warn(
                f"Unable to log: {key}, type: {type(value)} , value: {value}"
                + f" ex: {ex}"
            )

    def scalar_summary(self, key: str, value: float) -> None:
        if self._run:
            self._run[f"{self._label}/{format_key(key)}"].log(value)

    def dict_summary(self, key: str, value: Dict) -> None:
        dict_info = self._flatten_dict(parent_key=key, dict_info=value)
        for k, v in dict_info.items():
            self.scalar_summary(k, v)

    def histogram_summary(self, key: str, value: np.ndarray) -> None:
        return

    # Flatten dict, adapted from
    # https://stackoverflow.com/questions/6027558/flatten-nested-dictionaries-compressing-keys
    # Converts {'agent_0': {'critic_loss': 0.1, 'policy_loss': 0.2},...}
    #   to  {'agent_0_critic_loss':0.1,'agent_0_policy_loss':0.1 ,...}
    def _flatten_dict(
        self, parent_key: str, dict_info: Dict, sep: str = "_"
    ) -> Dict[str, float]:
        items: List = []
        for k, v in dict_info.items():
            k = str(k)
            if parent_key:
                new_key = parent_key + sep + k
            else:
                new_key = k
            if isinstance(v, dict):
                items.extend(
                    self._flatten_dict(parent_key=new_key, dict_info=v, sep=sep).items()
                )
            else:
                items.append((new_key, v))
        return dict(items)

    def close(self) -> None:
        self._run = None

    def stop(self) -> None:
        self._run.stop()
