import copy
import json
import collections
import os
import random
import numpy as np
from tqdm import tqdm
import sqlite3


def load_json(json_file) -> list:
    with open(json_file, 'r', encoding='utf-8') as f:
        ex_list = json.load(f)
    return ex_list

def write_data(lines, output_file):
    with open(output_file, 'w', encoding='utf-8') as f:
        f.write('\n'.join(lines))

def get_db_schema(bench_roots) -> dict:
    db_schema = collections.defaultdict(dict)
    for bench_root in bench_roots:
        tables_json = load_json(bench_root + '/tables.json')
        for table_json in tables_json:
            db_id = table_json['db_id']
            db_schema[db_id] = collections.defaultdict(dict)

            table_id_to_column_ids = collections.defaultdict(list)
            column_id_to_column_name = {}
            column_id_to_table_id = {}
            for column_id, table_column in enumerate(table_json['column_names_original']):
                table_id = table_column[0]
                column_name = table_column[1]
                column_id_to_column_name[column_id] = column_name.replace(" ", "_")
                table_id_to_column_ids[table_id].append(column_id)
                column_id_to_table_id[column_id] = table_id

            column_id_to_column_type = {}
            if len(table_json['column_types']) < len(table_json['column_names_original']):
                table_json['column_types'] = ["text"] + table_json['column_types']
            for column_id, column_type in enumerate(table_json['column_types']):
                column_id_to_column_type[column_id] = column_type

            table_id_to_table_name = {}
            if len(table_json['table_names_original']) == 0:
                table_name_list = [db_id]
            else:
                table_name_list = table_json['table_names_original']
            for table_id, table_name in enumerate(table_name_list):
                table_id_to_table_name[table_id] = table_name.replace(" ", "_")

            primary_keys = table_json['primary_keys']
            foreign_keys = {}
            for column_id, referenced_column_id in table_json['foreign_keys']:
                foreign_keys[column_id] = referenced_column_id

            for table_id in table_id_to_table_name.keys():
                table_name = table_id_to_table_name[table_id]
                for column_id in table_id_to_column_ids[table_id]:
                    column_name = column_id_to_column_name[column_id]
                    column_info = {
                        'type': column_id_to_column_type[column_id],
                        'is_primary_key': column_id in primary_keys,
                        'is_foreign_key': column_id in foreign_keys.keys(),
                    }
                    if column_info['is_foreign_key']:
                        referenced_table = table_id_to_table_name[column_id_to_table_id[foreign_keys[column_id]]]
                        referenced_column = column_id_to_column_name[foreign_keys[column_id]]
                        column_info['reference'] = '.'.join([referenced_table, referenced_column])
                    db_schema[db_id][table_name][column_name] = column_info
    return db_schema

def is_wikisql_table(table_name):
    for s in table_name:
        if s != "-" and not s.isdigit():
            return False
    return True

def generate_template(dataset, db_id, db_schema: dict):
    tables = []
    for table in db_schema[db_id]:
        # if dataset in ['spider', 'sparc', 'cosql', 'fusion']:
        #     table_info = f'{table}: '
        # else:
        #     table_info = ''

        table_info = f'{table}: ' if not is_wikisql_table(table) else ''
        # print(table, table_info)

        columns = []
        for column in db_schema[db_id][table].keys():
            column_info = column
            columns.append(column_info)

        if not is_wikisql_table(table):
            table_info += ', '.join(columns[:10])
        else:
            table_info += ', '.join(columns[:20])

        # print(table_info)
        # exit()

        tables.append(table_info)

    return '; '.join(tables[:10])

def generate_continual_tasks(setting_name, run_id=0, seed=23, combine_K=6, context_num=2,
                             context_threshold=0., mix=True, train_without_context=False, template_idx=1,
                             current_task_context_only=True, share_p=0.):

    dataset = setting_name.split("_")[0]
    oirginal_task_num = len(os.listdir(f'./data/task_splits/{dataset}_context/'))

    random.seed(seed)

    if dataset in ['spider', 'sparc', 'cosql', 'fusion']:
        task_perm = [11, 5, 3, 7, 12, 2, 10, 8, 6, 4, 15, 0, 1, 13, 14, 9]
        task_0 = [task_perm[:combine_K]]
        other_tasks = [[x] for x in task_perm[combine_K:]]
        random.shuffle(other_tasks)
    elif dataset in ['combine1', 'combine2']:
        if dataset == 'combine1':
            task_perm = [3, 1, 0, 4, 5, 9, 2, 10, 7, 8, 6]
        else:
            task_perm = [0, 1, 2, 3, 4, 5, 11, 6, 12, 7, 13, 8, 14, 9, 15, 10]
        task_0 = [task_perm[:combine_K]]
        other_tasks_1 = [[x] for i, x in enumerate(task_perm[combine_K:]) if i % 2 == 0]
        other_tasks_2 = [[x] for i, x in enumerate(task_perm[combine_K:]) if i % 2 == 1]
        random.shuffle(other_tasks_1)
        random.shuffle(other_tasks_2)
        other_tasks = [other_tasks_1[i // 2] if i % 2 == 0 else other_tasks_2[i // 2] for i in range(len(task_perm[combine_K:]))]
    else:
        raise NotImplementedError('No such dataset !')

    task_split = task_0 + other_tasks
    print(task_split)

    assert len(task_split) == oirginal_task_num - combine_K + 1

    id2examples_1 = {}
    id2examples_2 = {}
    id2examples_3 = {}
    for i in range(oirginal_task_num):  # load all training examples
        for example in json.load(open(f'./data/task_splits/{dataset}_context/task_{i}/train.json', 'r')):
            if 'context' in example:
                id2examples_1[example['guid']] = example
            else:
                id2examples_2[example['guid']] = example
            id2examples_3[example['guid']] = example

    if dataset == 'combine1':
        db_schema = get_db_schema(['./data/original_datasets/spider',
                                   './data/original_datasets/wikisql'])
    elif dataset == 'combine2':
        db_schema = get_db_schema(['./data/original_datasets/spider',
                                   './data/original_datasets/cosql_dataset'])
    elif dataset == 'fusion':
        db_schema = get_db_schema(['./data/original_datasets/spider',
                                   './data/original_datasets/cosql_dataset'])
    elif dataset == 'cosql':
        db_schema = get_db_schema(['./data/original_datasets/cosql_dataset'])
    else:
        db_schema = get_db_schema([f'./data/original_datasets/{dataset}'])

    db_templates = {}
    for db_id in db_schema.keys():
        db_templates[db_id] = generate_template(dataset, db_id, db_schema)

    dataset_root = f'./data/seq2seq_data/{dataset}'
    os.makedirs(dataset_root, exist_ok=True)
    new_root = f'{dataset_root}/{setting_name}_perm_{run_id}/'
    os.makedirs(new_root, exist_ok=True)
    print(new_root)

    table_list = []
    for i, task_ids in enumerate(task_split):
        if i == 0: continue
        table_data = []
        for task_id in task_ids:
            bench_root = f'./data/task_splits/{dataset}_context/task_{task_id}'
            table_data += load_json(f'{bench_root}/tables.json')
            table_list.append(table_data)

    def get_raw_datasets(task_ids):
        train_data = []
        dev_data = []
        test_data = []
        table_data = []
        for task_id in task_ids:
            bench_root = f'./data/task_splits/{dataset}_context/task_{task_id}'
            train_data += load_json(f'{bench_root}/train.json')
            dev_data += load_json(f'{bench_root}/dev.json')
            test_data += load_json(f'{bench_root}/test.json')
            table_data += load_json(f'{bench_root}/tables.json')

        raw_datasets = {
            'train': train_data,
            'dev': dev_data,
            'test': test_data,
            'tables': table_data
        }
        return raw_datasets

    def write_seq2seq_dataset(datas, seen_ex_ids, mode, threshold: float=0):
        json_lines = []
        for ex in datas:
            # print(ex.keys())

            if 'context' in ex:
                text = ' | '.join([db_templates[ex['db_id']], ex['context'], ex['question']])
                cand_contexts = [x[0] for x in ex['demonstration_examples']
                                 if x[0] in seen_ex_ids and x[1] >= threshold and x[0] != ex['guid'] and x[0] in id2examples_1 and id2examples_1[x[0]]['interaction_id'] != ex['interaction_id']]
            else:
                text = ' | '.join([db_templates[ex['db_id']], ex['question']])
                cand_contexts = [x[0] for x in ex['demonstration_examples']
                                 if x[0] in seen_ex_ids and x[1] >= threshold and x[0] != ex['guid'] and x[0] in id2examples_2]

            if mode == "train" and train_without_context is True:
                actual_num_context = 0
            else:
                actual_num_context = min(context_num, len(cand_contexts))

            contexts = []
            for c_id in range(actual_num_context):
                # print(cand_contexts[c_id] in id2examples_1, cand_contexts[c_id] in id2examples_2, cand_contexts[c_id] in seen_ex_ids)
                context_ex = id2examples_1[cand_contexts[c_id]] if 'context' in ex else id2examples_2[cand_contexts[c_id]]
                if 'context' in context_ex:
                    contexts.append(' | '.join([context_ex['context'],
                                                context_ex['question'],
                                                wrap_query(context_ex['query'], context_ex)]))
                else:
                    contexts.append(' | '.join([context_ex['question'],
                                                wrap_query(context_ex['query'], context_ex)]))

            template = ' || '.join(contexts + [text])

            sql = ex['query']
            sql = wrap_query(sql, ex)
            sql = sql
            template = template

            json_lines.append(json.dumps({'text': template, 'sql': sql, 'example': ex}))
            if mix is True and mode == "train" and len(contexts) > 0:
                template = text
                json_lines.append(json.dumps({'text': template, 'sql': sql, 'example': ex}))
        return json_lines

    def wrap_query(query, example):
        if dataset == "wikisql" or (dataset == "combine" and example["dataset_name"] == "wikisql"):
            x, y = query.split("FROM")
            query = x + " ".join(y.strip(" ").split(" ")[1:])
        return query

    seen_example_ids = []
    for i, task_ids in enumerate(task_split):

        raw_datasets = get_raw_datasets(task_ids)
        # raw_datasets = simplify_context(raw_datasets)

        new_root_task_i = f'{new_root}/task_{i}/'
        if not os.path.exists(new_root_task_i):
            os.mkdir(new_root_task_i)

        seen_example_ids += [x["guid"] for x in raw_datasets["train"]]
        seen_example_ids_current_task = [x["guid"] for x in raw_datasets['train']]

        # current task
        for mode in ['train', 'dev', 'test']:
            if current_task_context_only or mode == "train":
                json_lines = write_seq2seq_dataset(raw_datasets[mode], set(seen_example_ids_current_task),
                                                   mode=mode, threshold=context_threshold)
            else:
                json_lines = write_seq2seq_dataset(raw_datasets[mode], set(seen_example_ids),
                                                   mode=mode, threshold=context_threshold)
            write_data(json_lines, f'{new_root_task_i}/{mode}_seq2seq.jsonl')
            json.dump(raw_datasets[mode], open(f'{new_root_task_i}/{mode}.json', "w"))
        json.dump(raw_datasets["tables"], open(f'{new_root_task_i}/tables.json', "w"))

def combine_table():
    dbs = {}
    for mode in ['train', 'dev', 'test']:
        tables = json.load(open(f'data/original_datasets/wikisql/{mode}.tables.json'))
        for table in tables:
            if table['db_id'] not in dbs:
                dbs[table['db_id']] = table
    dbs = [v for k, v in dbs.items()]
    json.dump(dbs, open(f'data/original_datasets/wikisql/tables.json', 'w'), indent=4)

def convert_context_name():
    for i in range(0, 11):
        for mode in ['train', 'dev', 'test']:
            datas = json.load(open(f'data/task_splits/combine1_context/task_{i}/{mode}.json'))
            for data in datas:
                if 'contexts' in data:
                    data['demonstration_examples'] = [x for x in data['contexts']]
                    data.pop('contexts')
            json.dump(datas, open(f'data/task_splits/combine1_context/task_{i}/{mode}.json', 'w'), indent=4)



if __name__ == '__main__':

    os.makedirs('./data/seq2seq_data/', exist_ok=True)
    # convert_context_name()
    combine_table()

    generate_continual_tasks("spider_cxt_1_4.0_mix_ctco", run_id=1, seed=23, combine_K=6, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=True)
    generate_continual_tasks("spider_cxt_1_4.0_mix_ctco", run_id=2, seed=17, combine_K=6, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=True)
    generate_continual_tasks("spider_cxt_1_4.0_mix_ctco", run_id=3, seed=19, combine_K=6, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=True)
    generate_continual_tasks("spider_cxt_1_4.0_mix_ctco", run_id=4, seed=29, combine_K=6, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=True)
    generate_continual_tasks("spider_cxt_1_4.0_mix_ctco", run_id=5, seed=31, combine_K=6, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=True)

    #
    # generate_continual_tasks("spider_cxt_1_4.0_mix", run_id=1, seed=23, combine_K=6, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("spider_cxt_1_4.0_mix", run_id=2, seed=17, combine_K=6, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("spider_cxt_1_4.0_mix", run_id=3, seed=19, combine_K=6, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("spider_cxt_1_4.0_mix", run_id=4, seed=29, combine_K=6, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("spider_cxt_1_4.0_mix", run_id=5, seed=31, combine_K=6, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=False)
    #
    # generate_continual_tasks("spider_cxt_1_4.0_ctco", run_id=1, seed=23, combine_K=6, context_num=1, context_threshold=4.0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=True)
    #
    # generate_continual_tasks("spider_cxt_1_4.0", run_id=1, seed=23, combine_K=6, context_num=1, context_threshold=4.0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("spider_cxt_1_4.0", run_id=2, seed=17, combine_K=6, context_num=1, context_threshold=4.0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("spider_cxt_1_4.0", run_id=3, seed=19, combine_K=6, context_num=1, context_threshold=4.0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("spider_cxt_1_4.0", run_id=4, seed=29, combine_K=6, context_num=1, context_threshold=4.0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("spider_cxt_1_4.0", run_id=5, seed=31, combine_K=6, context_num=1, context_threshold=4.0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)

    # generate_continual_tasks("spider", run_id=1, seed=23, combine_K=6, context_num=0, context_threshold=0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("spider", run_id=2, seed=17, combine_K=6, context_num=0, context_threshold=0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("spider", run_id=3, seed=19, combine_K=6, context_num=0, context_threshold=0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("spider", run_id=4, seed=29, combine_K=6, context_num=0, context_threshold=0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("spider", run_id=5, seed=31, combine_K=6, context_num=0, context_threshold=0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)


    generate_continual_tasks("combine1_cxt_1_4.0_mix_ctco", run_id=1, seed=23, combine_K=5, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=True)
    generate_continual_tasks("combine1_cxt_1_4.0_mix_ctco", run_id=2, seed=17, combine_K=5, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=True)
    generate_continual_tasks("combine1_cxt_1_4.0_mix_ctco", run_id=3, seed=19, combine_K=5, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=True)
    generate_continual_tasks("combine1_cxt_1_4.0_mix_ctco", run_id=4, seed=29, combine_K=5, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=True)
    generate_continual_tasks("combine1_cxt_1_4.0_mix_ctco", run_id=5, seed=31, combine_K=5, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=True)

    #
    # generate_continual_tasks("combine1_cxt_1_4.0_mix", run_id=1, seed=23, combine_K=5, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine1_cxt_1_4.0_mix", run_id=2, seed=17, combine_K=5, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine1_cxt_1_4.0_mix", run_id=3, seed=19, combine_K=5, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine1_cxt_1_4.0_mix", run_id=4, seed=29, combine_K=5, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine1_cxt_1_4.0_mix", run_id=5, seed=31, combine_K=5, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=False)
    #
    # generate_continual_tasks("combine1_cxt_1_4.0_ctco", run_id=1, seed=23, combine_K=5, context_num=1, context_threshold=4.0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=True)
    #
    # generate_continual_tasks("combine1_cxt_1_4.0", run_id=1, seed=23, combine_K=5, context_num=1, context_threshold=4.0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine1_cxt_1_4.0", run_id=2, seed=17, combine_K=5, context_num=1, context_threshold=4.0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine1_cxt_1_4.0", run_id=3, seed=19, combine_K=5, context_num=1, context_threshold=4.0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine1_cxt_1_4.0", run_id=4, seed=29, combine_K=5, context_num=1, context_threshold=4.0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine1_cxt_1_4.0", run_id=5, seed=31, combine_K=5, context_num=1, context_threshold=4.0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    #
    # generate_continual_tasks("combine1", run_id=1, seed=23, combine_K=5, context_num=0, context_threshold=0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine1", run_id=2, seed=17, combine_K=5, context_num=0, context_threshold=0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine1", run_id=3, seed=19, combine_K=5, context_num=0, context_threshold=0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine1", run_id=4, seed=29, combine_K=5, context_num=0, context_threshold=0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine1", run_id=5, seed=31, combine_K=5, context_num=0, context_threshold=0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)


    # generate_continual_tasks("combine2_cxt_1_4.0_mix_ctco", run_id=1, seed=23, combine_K=6, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=True)
    # generate_continual_tasks("combine2_cxt_1_4.0_mix", run_id=1, seed=23, combine_K=6, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine2_cxt_1_4.0_mix", run_id=2, seed=17, combine_K=6, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine2_cxt_1_4.0_mix", run_id=3, seed=19, combine_K=6, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine2_cxt_1_4.0_mix", run_id=4, seed=29, combine_K=6, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine2_cxt_1_4.0_mix", run_id=5, seed=31, combine_K=6, context_num=1, context_threshold=4.0, mix=True, train_without_context=False,template_idx=1,current_task_context_only=False)
    #
    # generate_continual_tasks("combine2_cxt_1_4.0_ctco", run_id=1, seed=23, combine_K=6, context_num=1, context_threshold=4.0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=True)
    #
    # generate_continual_tasks("combine2_cxt_1_4.0", run_id=1, seed=23, combine_K=6, context_num=1, context_threshold=4.0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine2_cxt_1_4.0", run_id=2, seed=17, combine_K=6, context_num=1, context_threshold=4.0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine2_cxt_1_4.0", run_id=3, seed=19, combine_K=6, context_num=1, context_threshold=4.0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine2_cxt_1_4.0", run_id=4, seed=29, combine_K=6, context_num=1, context_threshold=4.0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine2_cxt_1_4.0", run_id=5, seed=31, combine_K=6, context_num=1, context_threshold=4.0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    #
    # generate_continual_tasks("combine2", run_id=1, seed=23, combine_K=6, context_num=0, context_threshold=0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine2", run_id=2, seed=17, combine_K=6, context_num=0, context_threshold=0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine2", run_id=3, seed=19, combine_K=6, context_num=0, context_threshold=0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine2", run_id=4, seed=29, combine_K=6, context_num=0, context_threshold=0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)
    # generate_continual_tasks("combine2", run_id=5, seed=31, combine_K=6, context_num=0, context_threshold=0, mix=False, train_without_context=False,template_idx=1,current_task_context_only=False)



    # generate_continual_tasks("sparc_cxt_1_3.0_mix_ctco", run_id=1, seed=23, combine_K=6, context_num=1, context_threshold=3.0, mix=True, train_without_context=False, template_idx=1, current_task_context_only=True, share_p=0.)
    #
    # generate_continual_tasks("sparc_cxt_1_3.0_mix", run_id=1, seed=23, combine_K=6, context_num=1, context_threshold=3.0, mix=True, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)
    # generate_continual_tasks("sparc_cxt_1_3.0_mix", run_id=2, seed=17, combine_K=6, context_num=1, context_threshold=3.0, mix=True, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)
    # generate_continual_tasks("sparc_cxt_1_3.0_mix", run_id=3, seed=19, combine_K=6, context_num=1, context_threshold=3.0, mix=True, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)
    # generate_continual_tasks("sparc_cxt_1_3.0_mix", run_id=4, seed=29, combine_K=6, context_num=1, context_threshold=3.0, mix=True, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)
    # generate_continual_tasks("sparc_cxt_1_3.0_mix", run_id=5, seed=31, combine_K=6, context_num=1, context_threshold=3.0, mix=True, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)
    #
    # generate_continual_tasks("sparc_cxt_1_3.0_ctco", run_id=1, seed=23, combine_K=6, context_num=1, context_threshold=3.0, mix=False, train_without_context=False, template_idx=1, current_task_context_only=True, share_p=0.)
    #
    # generate_continual_tasks("sparc_cxt_1_3.0", run_id=1, seed=23, combine_K=6, context_num=1, context_threshold=3.0, mix=False, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)
    # generate_continual_tasks("sparc_cxt_1_3.0", run_id=2, seed=17, combine_K=6, context_num=1, context_threshold=3.0, mix=False, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)
    # generate_continual_tasks("sparc_cxt_1_3.0", run_id=3, seed=19, combine_K=6, context_num=1, context_threshold=3.0, mix=False, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)
    # generate_continual_tasks("sparc_cxt_1_3.0", run_id=4, seed=29, combine_K=6, context_num=1, context_threshold=3.0, mix=False, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)
    # generate_continual_tasks("sparc_cxt_1_3.0", run_id=5, seed=31, combine_K=6, context_num=1, context_threshold=3.0, mix=False, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)
    # 
    # generate_continual_tasks("sparc", run_id=1, seed=23, combine_K=6, context_num=0, context_threshold=0.0, mix=False, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)


    # generate_continual_tasks("cosql_cxt_1_3.0_mix_ctco", run_id=1, seed=23, combine_K=6, context_num=1, context_threshold=3.0, mix=True, train_without_context=False, template_idx=1, current_task_context_only=True, share_p=0.)
    #
    # generate_continual_tasks("cosql_cxt_1_3.0_mix", run_id=1, seed=23, combine_K=6, context_num=1, context_threshold=3.0, mix=True, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)
    # generate_continual_tasks("cosql_cxt_1_3.0_mix", run_id=2, seed=17, combine_K=6, context_num=1, context_threshold=3.0, mix=True, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)
    # generate_continual_tasks("cosql_cxt_1_3.0_mix", run_id=3, seed=19, combine_K=6, context_num=1, context_threshold=3.0, mix=True, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)
    # generate_continual_tasks("cosql_cxt_1_3.0_mix", run_id=4, seed=29, combine_K=6, context_num=1, context_threshold=3.0, mix=True, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)
    # generate_continual_tasks("cosql_cxt_1_3.0_mix", run_id=5, seed=31, combine_K=6, context_num=1, context_threshold=3.0, mix=True, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)
    #
    # generate_continual_tasks("cosql_cxt_1_3.0_ctco", run_id=1, seed=23, combine_K=6, context_num=1, context_threshold=3.0, mix=False, train_without_context=False, template_idx=1, current_task_context_only=True, share_p=0.)
    #
    # generate_continual_tasks("cosql_cxt_1_3.0", run_id=1, seed=23, combine_K=6, context_num=1, context_threshold=3.0, mix=False, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)
    # generate_continual_tasks("cosql_cxt_1_3.0", run_id=2, seed=17, combine_K=6, context_num=1, context_threshold=3.0, mix=False, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)
    # generate_continual_tasks("cosql_cxt_1_3.0", run_id=3, seed=19, combine_K=6, context_num=1, context_threshold=3.0, mix=False, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)
    # generate_continual_tasks("cosql_cxt_1_3.0", run_id=4, seed=29, combine_K=6, context_num=1, context_threshold=3.0, mix=False, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)
    # generate_continual_tasks("cosql_cxt_1_3.0", run_id=5, seed=31, combine_K=6, context_num=1, context_threshold=3.0, mix=False, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)
    #
    # generate_continual_tasks("cosql", run_id=1, seed=23, combine_K=6, context_num=0, context_threshold=0.0, mix=False, train_without_context=False, template_idx=1, current_task_context_only=False, share_p=0.)
