classdef AFNN_CSIV
    properties 
        learning_rate 
        momentum 
        n_inputs
        n_outputs_decoder
        n_outputs_classifier
        n_hidden_layers_shared
        n_hidden_units_shared
        n_hidden_layers_decoder
        n_hidden_units_decoder
        n_hidden_layers_classifier
        n_hidden_units_classifier

        hidden_weights_shared
        hidden_weights_decoder
        hidden_weights_classifier
        output_weights_decoder
        output_weights_classifier

        prev_hidden_weight_delta_shared
        prev_hidden_weight_delta_decoder
        prev_hidden_weight_delta_classifier
        prev_output_weight_delta_decoder
        prev_output_weight_delta_classifier
    end
    
    methods
        %Construction
        function obj = AFNN_CSIV(n_inputs, n_outputs_decoder, n_outputs_classifier, ...
                            n_hidden_layers_shared, n_hidden_layers_decoder, n_hidden_layers_classifier,...
                            n_hidden_units_shared, n_hidden_units_decoder, n_hidden_units_classifier, ...
                            learning_rate, momentum)
            obj.n_inputs = n_inputs;
            
            obj.n_outputs_decoder = n_outputs_decoder;
            obj.n_outputs_classifier = n_outputs_classifier;

            obj.n_hidden_layers_shared = n_hidden_layers_shared;
            obj.n_hidden_units_shared = n_hidden_units_shared;

            obj.n_hidden_layers_decoder = n_hidden_layers_decoder;
            obj.n_hidden_units_decoder = n_hidden_units_decoder;

            obj.n_hidden_layers_classifier = n_hidden_layers_classifier;
            obj.n_hidden_units_classifier = n_hidden_units_classifier;

            obj.learning_rate = learning_rate;
            obj.momentum = momentum;
            
           
            obj.hidden_weights_shared = cell(1, n_hidden_layers_shared);
            for i = 1:n_hidden_layers_shared
                if i == 1
                    input_size = n_inputs;
                else
                    input_size = n_hidden_units_shared(i-1);
                end
                obj.hidden_weights_shared{i} = randn(input_size, n_hidden_units_shared(i));
            end

           
            obj.prev_hidden_weight_delta_shared = cell(1,n_hidden_layers_shared);
            for i = 1:n_hidden_layers_shared
                obj.prev_hidden_weight_delta_shared{i} = zeros(size(obj.hidden_weights_shared{i}));
            end
            
            if n_hidden_layers_decoder >= 1
                obj.hidden_weights_decoder = cell(1, n_hidden_layers_decoder);
                for i = 1:n_hidden_layers_decoder
                    if i == 1
                        input_size = n_hidden_units_shared(end);
                    else
                        input_size = n_hidden_units_decoder(i-1);
                    end
                    obj.hidden_weights_decoder{i} = randn(input_size, n_hidden_units_decoder(i));
                end
                obj.output_weights_decoder = randn(n_hidden_units_decoder(end), n_outputs_decoder);

                obj.prev_hidden_weight_delta_decoder = cell(1,n_hidden_layers_decoder);
                for i = 1:n_hidden_layers_decoder
                    obj.prev_hidden_weight_delta_decoder{i} = zeros(size(obj.hidden_weights_decoder{i}));
                end

                obj.prev_output_weight_delta_decoder = zeros(n_hidden_units_decoder(end), n_outputs_decoder);
            else
                obj.output_weights_decoder = randn(n_hidden_units_shared(end), n_outputs_decoder);
                obj.prev_output_weight_delta_decoder = zeros(n_hidden_units_shared(end), n_outputs_decoder);
            end
            
            if n_hidden_layers_classifier >= 1
                obj.hidden_weights_classifier = cell(1, n_hidden_layers_classifier);
                for i = 1:n_hidden_layers_classifier
                    if i == 1
                        input_size = n_hidden_units_shared(end);
                    else
                        input_size = n_hidden_units_classifier(i-1);
                    end
                    obj.hidden_weights_classifier{i} = randn(input_size, n_hidden_units_classifier(i));
                end
                obj.output_weights_classifier = randn(n_hidden_units_classifier(end), n_outputs_classifier);

                obj.prev_hidden_weight_delta_classifier = cell(1,n_hidden_layers_classifier);
                for i = 1:n_hidden_layers_classifier
                    obj.prev_hidden_weight_delta_classifier{i} = zeros(size(obj.hidden_weights_classifier{i}));
                end
                obj.prev_output_weight_delta_classifier = zeros(n_hidden_units_classifier(end), n_outputs_classifier); 
            else
                obj.output_weights_classifier = randn(n_hidden_units_shared(end), n_outputs_classifier);
                obj.prev_output_weight_delta_classifier = zeros(n_hidden_units_shared(end), n_outputs_classifier);
            end      
        end


        function [outputs_decoder, hidden_outputs_shared, hidden_outputs_decoder] = forwardDecoder(obj, inputs)
            
            hidden_outputs_shared = cell(1, obj.n_hidden_layers_shared);
            for i = 1:obj.n_hidden_layers_shared
                if i == 1
                    hidden_inputs_shared = inputs * obj.hidden_weights_shared{i};
                else
                    hidden_inputs_shared = hidden_outputs_shared{i-1} * obj.hidden_weights_shared{i};
                end
                hidden_outputs_shared{i} = sigmoid(hidden_inputs_shared);
            end
            
            
            if obj.n_hidden_layers_decoder >= 1
                hidden_outputs_decoder = cell(1, obj.n_hidden_layers_decoder);
                for i = 1:obj.n_hidden_layers_decoder
                    if i == 1
                        hidden_inputs_decoder = hidden_outputs_shared{end} * obj.hidden_weights_decoder{i};
                    else
                        hidden_inputs_decoder = hidden_outputs_decoder{i-1} * obj.hidden_weights_decoder{i};
                    end
                    hidden_outputs_decoder{i} = sigmoid(hidden_inputs_decoder);
                end
                outputs_decoder = sigmoid(hidden_outputs_decoder{end} * obj.output_weights_decoder);
            else
                hidden_outputs_decoder = 0;
                outputs_decoder = sigmoid(hidden_outputs_shared{end} * obj.output_weights_decoder);
            end   
        end
		
		function hidden_outputs_shared = forwardEncoder(obj, inputs)
            
            hidden_outputs_shared = cell(1, obj.n_hidden_layers_shared);
            for i = 1:obj.n_hidden_layers_shared
                if i == 1
                    hidden_inputs_shared = inputs * obj.hidden_weights_shared{i};
                else
                    hidden_inputs_shared = hidden_outputs_shared{i-1} * obj.hidden_weights_shared{i};
                end
                hidden_outputs_shared{i} = sigmoid(hidden_inputs_shared);
            end
        end
		
		function outputs_decoder = forwardDecoderFromHidden(obj, inputs)
           
            if obj.n_hidden_layers_decoder >= 1
                hidden_outputs_decoder = cell(1, obj.n_hidden_layers_decoder);
                for i = 1:obj.n_hidden_layers_decoder
                    if i == 1
                        hidden_inputs_decoder = inputs * obj.hidden_weights_decoder{i};
                    else
                        hidden_inputs_decoder = hidden_outputs_decoder{i-1} * obj.hidden_weights_decoder{i};
                    end
                    hidden_outputs_decoder{i} = sigmoid(hidden_inputs_decoder);
                end
                outputs_decoder = sigmoid(hidden_outputs_decoder{end} * obj.output_weights_decoder);
            else
                hidden_outputs_decoder = 0;
                outputs_decoder = sigmoid(inputs * obj.output_weights_decoder);
            end   
        end

        function outputs_classifier = forwardClassifierFromHidden(obj, inputs)
           
            if obj.n_hidden_layers_classifier >= 1
                hidden_outputs_classifier = cell(1, obj.n_hidden_layers_classifier);
                for i = 1:obj.n_hidden_layers_classifier
                    if i == 1
                        hidden_inputs_classifier = inputs * obj.hidden_weights_classifier{i};
                    else
                        hidden_inputs_classifier = hidden_outputs_classifier{i-1} * obj.hidden_weights_classifier{i};
                    end
                    hidden_outputs_classifier{i} = sigmoid(hidden_inputs_classifier);
                end
                outputs_classifier = sigmoid(hidden_outputs_classifier{end} * obj.output_weights_classifier);
            else
                hidden_outputs_classifier = 0;
                outputs_classifier = sigmoid(inputs * obj.output_weights_classifier);
            end   
        end

        function [outputs_classifier, hidden_outputs_shared, hidden_outputs_classifier] = forwardClassifier(obj, inputs)
            
            hidden_outputs_shared = cell(1, obj.n_hidden_layers_shared);
            for i = 1:obj.n_hidden_layers_shared
                if i == 1
                    hidden_inputs_shared = inputs * obj.hidden_weights_shared{i};
                else
                    hidden_inputs_shared = hidden_outputs_shared{i-1} * obj.hidden_weights_shared{i};
                end
                hidden_outputs_shared{i} = sigmoid(hidden_inputs_shared);
            end
            
            
            if obj.n_hidden_layers_classifier >= 1
                hidden_outputs_classifier = cell(1, obj.n_hidden_layers_classifier);
                for i = 1:obj.n_hidden_layers_classifier
                    if i == 1
                        hidden_inputs_classifier = hidden_outputs_shared{end} * obj.hidden_weights_classifier{i};
                    else
                        hidden_inputs_classifier = hidden_outputs_classifier{i-1} * obj.hidden_weights_classifier{i};
                    end
                    hidden_outputs_classifier{i} = sigmoid(hidden_inputs_classifier);
                end
                outputs_classifier = sigmoid(hidden_outputs_classifier{end} * obj.output_weights_classifier);
            else
                hidden_outputs_classifier = 0;
                outputs_classifier = sigmoid(hidden_outputs_shared{end} * obj.output_weights_classifier);
            end   
        end

        function train_shared_decoder(obj, inputs, targets_decoder)
            
            [outputs_decoder, hidden_outputs_shared, hidden_outputs_decoder] = obj.forwardDecoder(inputs);
            
            
            output_errors_decoder = (outputs_decoder - targets_decoder) .* sigmoid_derivative(outputs_decoder);
            if obj.n_hidden_layers_decoder >= 1 
                hidden_errors_decoder = cell(1, obj.n_hidden_layers_decoder);
                hidden_errors_decoder{end} = output_errors_decoder * obj.output_weights_decoder' .* sigmoid_derivative(hidden_outputs_decoder{end});
                for i = obj.n_hidden_layers_decoder-1:-1:1
                    hidden_errors_decoder{i} = hidden_errors_decoder{i+1} * obj.hidden_weights_decoder{i+1}' .* sigmoid_derivative(hidden_outputs_decoder{i});
                end
                
                obj.prev_output_weight_delta_decoder = obj.momentum * obj.prev_output_weight_delta_decoder - obj.learning_rate * hidden_outputs_decoder{end}' * output_errors_decoder;
                obj.output_weights_decoder = obj.output_weights_decoder + obj.prev_output_weight_delta_decoder;
                
                for i = obj.n_hidden_layers_decoder:-1:1
                    if i == 1
                        hidden_inputs_decoder = hidden_outputs_shared{end};
                    else
                        hidden_inputs_decoder = hidden_outputs_decoder{i-1};
                    end
                    obj.prev_hidden_weight_delta_decoder{i} = obj.momentum * obj.prev_hidden_weight_delta_decoder{i} - obj.learning_rate * hidden_inputs_decoder' * hidden_errors_decoder{i};
                    obj.hidden_weights_decoder{i} = obj.hidden_weights_decoder{i} + obj.prev_hidden_weight_delta_decoder{i};
                end
            else
                
                obj.prev_output_weight_delta_decoder = obj.momentum * obj.prev_output_weight_delta_decoder - obj.learning_rate * hidden_outputs_shared{end}' * output_errors_decoder;
                obj.output_weights_decoder = obj.output_weights_decoder + obj.prev_output_weight_delta_decoder;
            end
            
            
            hidden_errors_shared = cell(1, obj.n_hidden_layers_shared);
            if obj.n_hidden_layers_decoder >= 1
                hidden_errors_shared{end} = hidden_errors_decoder{1} * obj.hidden_weights_decoder{1}' .* sigmoid_derivative(hidden_outputs_shared{end}); 
            else
                hidden_errors_shared{end} = output_errors_decoder * obj.output_weights_decoder' .* sigmoid_derivative(hidden_outputs_shared{end}); 
            end

            for i = obj.n_hidden_layers_shared-1:-1:1
                hidden_errors_shared{i} = hidden_errors_shared{i+1} * obj.hidden_weights_shared{i+1}' .* sigmoid_derivative(hidden_outputs_shared{i});
            end
           
            
            for i = obj.n_hidden_layers_shared:-1:1
                if i == 1
                    hidden_inputs_shared = inputs;
                else
                    hidden_inputs_shared = hidden_outputs_shared{i-1};
                end
                obj.prev_hidden_weight_delta_shared{i} = obj.momentum * obj.prev_hidden_weight_delta_shared{i} - obj.learning_rate * hidden_inputs_shared' * hidden_errors_shared{i};
                obj.hidden_weights_shared{i} = obj.hidden_weights_shared{i} + obj.prev_hidden_weight_delta_shared{i};
            end
        end

        function train_shared_classifier(obj, inputs, targets_classifier)
            
            [outputs_classifier, hidden_outputs_shared, hidden_outputs_classifier] = obj.forwardClassifier(inputs);
            
            
            output_errors_classifier = (outputs_classifier - targets_classifier) .* sigmoid_derivative(outputs_classifier);
            if obj.n_hidden_layers_classifier >= 1 
                hidden_errors_classifier = cell(1, obj.n_hidden_layers_classifier);
                hidden_errors_classifier{end} = output_errors_classifier * obj.output_weights_classifier' .* sigmoid_derivative(hidden_outputs_classifier{end});
                for i = obj.n_hidden_layers_classifier-1:-1:1
                    hidden_errors_classifier{i} = hidden_errors_classifier{i+1} * obj.hidden_weights_classifier{i+1}' .* sigmoid_derivative(hidden_outputs_classifier{i});
                end
                
                obj.prev_output_weight_delta_classifier = obj.momentum * obj.prev_output_weight_delta_classifier - obj.learning_rate * hidden_outputs_classifier{end}' * output_errors_classifier;
                obj.output_weights_classifier = obj.output_weights_classifier + obj.prev_output_weight_delta_classifier;
                
                for i = obj.n_hidden_layers_classifier:-1:1
                    if i == 1
                        hidden_inputs_classifier = hidden_outputs_shared{end};
                    else
                        hidden_inputs_classifier = hidden_outputs_classifier{i-1};
                    end
                    obj.prev_hidden_weight_delta_classifier{i} = obj.momentum * obj.prev_hidden_weight_delta_classifier{i} - obj.learning_rate * hidden_inputs_classifier' * hidden_errors_classifier{i};
                    obj.hidden_weights_classifier{i} = obj.hidden_weights_classifier{i} + obj.prev_hidden_weight_delta_classifier{i};
                end
            else
                
                obj.prev_output_weight_delta_classifier = obj.momentum * obj.prev_output_weight_delta_classifier - obj.learning_rate * hidden_outputs_shared{end}' * output_errors_classifier;
                obj.output_weights_classifier = obj.output_weights_classifier + obj.prev_output_weight_delta_classifier;
            end
            
            
            hidden_errors_shared = cell(1, obj.n_hidden_layers_shared);
            if obj.n_hidden_layers_classifier >= 1
                hidden_errors_shared{end} = hidden_errors_classifier{1} * obj.hidden_weights_classifier{1}' .* sigmoid_derivative(hidden_outputs_shared{end}); 
            else
                hidden_errors_shared{end} = output_errors_classifier * obj.output_weights_classifier' .* sigmoid_derivative(hidden_outputs_shared{end}); 
            end

            for i = obj.n_hidden_layers_shared-1:-1:1
                hidden_errors_shared{i} = hidden_errors_shared{i+1} * obj.hidden_weights_shared{i+1}' .* sigmoid_derivative(hidden_outputs_shared{i});
            end
           
            
            for i = obj.n_hidden_layers_shared:-1:1
                if i == 1
                    hidden_inputs_shared = inputs;
                else
                    hidden_inputs_shared = hidden_outputs_shared{i-1};
                end
                obj.prev_hidden_weight_delta_shared{i} = obj.momentum * obj.prev_hidden_weight_delta_shared{i} - obj.learning_rate * hidden_inputs_shared' * hidden_errors_shared{i};
                obj.hidden_weights_shared{i} = obj.hidden_weights_shared{i} + obj.prev_hidden_weight_delta_shared{i};
            end
        end
		
		function train_classifier(obj, inputs, targets_classifier)
            
            [outputs_classifier, hidden_outputs_shared, hidden_outputs_classifier] = obj.forwardClassifier(inputs);
            
            
            output_errors_classifier = (outputs_classifier - targets_classifier) .* sigmoid_derivative(outputs_classifier);
            if obj.n_hidden_layers_classifier >= 1 
                hidden_errors_classifier = cell(1, obj.n_hidden_layers_classifier);
                hidden_errors_classifier{end} = output_errors_classifier * obj.output_weights_classifier' .* sigmoid_derivative(hidden_outputs_classifier{end});
                for i = obj.n_hidden_layers_classifier-1:-1:1
                    hidden_errors_classifier{i} = hidden_errors_classifier{i+1} * obj.hidden_weights_classifier{i+1}' .* sigmoid_derivative(hidden_outputs_classifier{i});
                end
                
                obj.prev_output_weight_delta_classifier = obj.momentum * obj.prev_output_weight_delta_classifier - obj.learning_rate * hidden_outputs_classifier{end}' * output_errors_classifier;
                obj.output_weights_classifier = obj.output_weights_classifier + obj.prev_output_weight_delta_classifier;
                
                for i = obj.n_hidden_layers_classifier:-1:1
                    if i == 1
                        hidden_inputs_classifier = hidden_outputs_shared{end};
                    else
                        hidden_inputs_classifier = hidden_outputs_classifier{i-1};
                    end
                    obj.prev_hidden_weight_delta_classifier{i} = obj.momentum * obj.prev_hidden_weight_delta_classifier{i} - obj.learning_rate * hidden_inputs_classifier' * hidden_errors_classifier{i};
                    obj.hidden_weights_classifier{i} = obj.hidden_weights_classifier{i} + obj.prev_hidden_weight_delta_classifier{i};
                end
            else
                
                obj.prev_output_weight_delta_classifier = obj.momentum * obj.prev_output_weight_delta_classifier - obj.learning_rate * hidden_outputs_shared{end}' * output_errors_classifier;
                obj.output_weights_classifier = obj.output_weights_classifier + obj.prev_output_weight_delta_classifier;
            end
        end

        function train_classifier_v2(obj, inputs, targets_classifier, p, epoch)
            xn = 0;
			tmp = 0.0;
            while tmp < 1.0-p && xn < epoch
                
                [outputs_classifier, hidden_outputs_shared, hidden_outputs_classifier] = obj.forwardClassifier(inputs);
                
                output_errors_classifier = (outputs_classifier - targets_classifier) .* sigmoid_derivative(outputs_classifier);
                if obj.n_hidden_layers_classifier >= 1
                    hidden_errors_classifier = cell(1, obj.n_hidden_layers_classifier);
                    hidden_errors_classifier{end} = output_errors_classifier * obj.output_weights_classifier' .* sigmoid_derivative(hidden_outputs_classifier{end});
                    for i = obj.n_hidden_layers_classifier-1:-1:1
                        hidden_errors_classifier{i} = hidden_errors_classifier{i+1} * obj.hidden_weights_classifier{i+1}' .* sigmoid_derivative(hidden_outputs_classifier{i});
                    end
                    
                    obj.prev_output_weight_delta_classifier = obj.momentum * obj.prev_output_weight_delta_classifier - obj.learning_rate * hidden_outputs_classifier{end}' * output_errors_classifier;
                    obj.output_weights_classifier = obj.output_weights_classifier + obj.prev_output_weight_delta_classifier;
                    
                    for i = obj.n_hidden_layers_classifier:-1:1
                        if i == 1
                            hidden_inputs_classifier = hidden_outputs_shared{end};
                        else
                            hidden_inputs_classifier = hidden_outputs_classifier{i-1};
                        end
                        obj.prev_hidden_weight_delta_classifier{i} = obj.momentum * obj.prev_hidden_weight_delta_classifier{i} - obj.learning_rate * hidden_inputs_classifier' * hidden_errors_classifier{i};
                        obj.hidden_weights_classifier{i} = obj.hidden_weights_classifier{i} + obj.prev_hidden_weight_delta_classifier{i};
                    end
                else
                    
                    obj.prev_output_weight_delta_classifier = obj.momentum * obj.prev_output_weight_delta_classifier - obj.learning_rate * hidden_outputs_shared{end}' * output_errors_classifier;
                    obj.output_weights_classifier = obj.output_weights_classifier + obj.prev_output_weight_delta_classifier;
                end

                tmp = reportModel(output_errors_classifier, p);

                xn = xn + 1;
                if mod(xn,100) == 0
                    p = p + 0.01; %Progressively increase the value of p,
                end
            end
            
        end

        function train_shared_decoder_v2(obj, inputs, targets_decoder, p, epoch)

            xn = 0;
			tmp = 0.0;

            while tmp < 1.0-p && xn < epoch
                
                [outputs_decoder, hidden_outputs_shared, hidden_outputs_decoder] = obj.forwardDecoder(inputs);

                
                output_errors_decoder = (outputs_decoder - targets_decoder) .* sigmoid_derivative(outputs_decoder);
                if obj.n_hidden_layers_decoder >= 1
                    hidden_errors_decoder = cell(1, obj.n_hidden_layers_decoder);
                    hidden_errors_decoder{end} = output_errors_decoder * obj.output_weights_decoder' .* sigmoid_derivative(hidden_outputs_decoder{end});
                    for i = obj.n_hidden_layers_decoder-1:-1:1
                        hidden_errors_decoder{i} = hidden_errors_decoder{i+1} * obj.hidden_weights_decoder{i+1}' .* sigmoid_derivative(hidden_outputs_decoder{i});
                    end
                    
                    obj.prev_output_weight_delta_decoder = obj.momentum * obj.prev_output_weight_delta_decoder - obj.learning_rate * hidden_outputs_decoder{end}' * output_errors_decoder;
                    obj.output_weights_decoder = obj.output_weights_decoder + obj.prev_output_weight_delta_decoder;
                    
                    for i = obj.n_hidden_layers_decoder:-1:1
                        if i == 1
                            hidden_inputs_decoder = hidden_outputs_shared{end};
                        else
                            hidden_inputs_decoder = hidden_outputs_decoder{i-1};
                        end
                        obj.prev_hidden_weight_delta_decoder{i} = obj.momentum * obj.prev_hidden_weight_delta_decoder{i} - obj.learning_rate * hidden_inputs_decoder' * hidden_errors_decoder{i};
                        obj.hidden_weights_decoder{i} = obj.hidden_weights_decoder{i} + obj.prev_hidden_weight_delta_decoder{i};
                    end
                else
                    
                    obj.prev_output_weight_delta_decoder = obj.momentum * obj.prev_output_weight_delta_decoder - obj.learning_rate * hidden_outputs_shared{end}' * output_errors_decoder;
                    obj.output_weights_decoder = obj.output_weights_decoder + obj.prev_output_weight_delta_decoder;
                end

                
                hidden_errors_shared = cell(1, obj.n_hidden_layers_shared);
                if obj.n_hidden_layers_decoder >= 1
                    hidden_errors_shared{end} = hidden_errors_decoder{1} * obj.hidden_weights_decoder{1}' .* sigmoid_derivative(hidden_outputs_shared{end});
                else
                    hidden_errors_shared{end} = output_errors_decoder * obj.output_weights_decoder' .* sigmoid_derivative(hidden_outputs_shared{end});
                end

                for i = obj.n_hidden_layers_shared-1:-1:1
                    hidden_errors_shared{i} = hidden_errors_shared{i+1} * obj.hidden_weights_shared{i+1}' .* sigmoid_derivative(hidden_outputs_shared{i});
                end

                
                for i = obj.n_hidden_layers_shared:-1:1
                    if i == 1
                        hidden_inputs_shared = inputs;
                    else
                        hidden_inputs_shared = hidden_outputs_shared{i-1};
                    end
                    obj.prev_hidden_weight_delta_shared{i} = obj.momentum * obj.prev_hidden_weight_delta_shared{i} - obj.learning_rate * hidden_inputs_shared' * hidden_errors_shared{i};
                    obj.hidden_weights_shared{i} = obj.hidden_weights_shared{i} + obj.prev_hidden_weight_delta_shared{i};
                end

                tmp = reportModel(output_errors_decoder, p);

                xn = xn + 1;
                if mod(xn,100) == 0
                    p = p + 0.01; %Progressively increase the value of p,
                end
            end
        end

        function outputs = predictClassifier(obj, inputs)
            
            [outputs, ~] = obj.forwardClassifier(inputs);
        end

        function outputs = predictDecoder(obj, inputs)
            
            [outputs, ~] = obj.forwardDecoder(inputs);
        end
    
        function set_learning_rate(obj, learning_rate)
            
            obj.learning_rate = learning_rate;
        end
    
        function set_momentum(obj, momentum)
            
            obj.momentum = momentum;
        end
    end
end

function count = reportModel(errors, p)
    count = 0;
    avg_Errors = sum(errors,2)/size(errors,2);
    length  = size(errors,1);
    for i = 1:length
        if avg_Errors(i) < p
            count = count + 1;
        end
    end
    count = count/length;
end

function y = sigmoid(x)
    
    y = 1./(1+exp(-x));
end

function y = sigmoid_derivative(x)
    
    y = sigmoid(x) .* (1-sigmoid(x));
end
