classdef AFNN_Autoencoder_REMOIV
    properties 
        learning_rate 
        momentum 
        n_inputs
        n_outputs_decoder
        n_hidden_layers_encoder
        n_hidden_units_encoder
        n_hidden_layers_decoder
        n_hidden_units_decoder

        hidden_weights_encoder
        hidden_weights_decoder
        output_weights_decoder

        prev_hidden_weight_delta_encoder
        prev_hidden_weight_delta_decoder
        prev_output_weight_delta_decoder
    end
    
    methods
        %Construction
        function obj = AFNN_Autoencoder_REMOIV(n_inputs, n_outputs_decoder, ...
                            n_hidden_layers_encoder, n_hidden_layers_decoder, ...
                            n_hidden_units_encoder, n_hidden_units_decoder, ...
                            learning_rate, momentum)
            obj.n_inputs = n_inputs;
            
            obj.n_outputs_decoder = n_outputs_decoder;

            obj.n_hidden_layers_encoder = n_hidden_layers_encoder;
            obj.n_hidden_units_encoder = n_hidden_units_encoder;

            obj.n_hidden_layers_decoder = n_hidden_layers_decoder;
            obj.n_hidden_units_decoder = n_hidden_units_decoder;

            obj.learning_rate = learning_rate;
            obj.momentum = momentum;
            
            obj.hidden_weights_encoder = cell(1, n_hidden_layers_encoder);
            for i = 1:n_hidden_layers_encoder
                if i == 1
                    input_size = n_inputs;
                else
                    input_size = n_hidden_units_encoder(i-1);
                end
                obj.hidden_weights_encoder{i} = randn(input_size, n_hidden_units_encoder(i));
            end

            obj.prev_hidden_weight_delta_encoder = cell(1,n_hidden_layers_encoder);
            for i = 1:n_hidden_layers_encoder
                obj.prev_hidden_weight_delta_encoder{i} = zeros(size(obj.hidden_weights_encoder{i}));
            end
            
            if n_hidden_layers_decoder >= 1
                obj.hidden_weights_decoder = cell(1, n_hidden_layers_decoder);
                for i = 1:n_hidden_layers_decoder
                    if i == 1
                        input_size = n_hidden_units_encoder(end);
                    else
                        input_size = n_hidden_units_decoder(i-1);
                    end
                    obj.hidden_weights_decoder{i} = randn(input_size, n_hidden_units_decoder(i));
                end
                obj.output_weights_decoder = randn(n_hidden_units_decoder(end), n_outputs_decoder);

                obj.prev_hidden_weight_delta_decoder = cell(1,n_hidden_layers_decoder);
                for i = 1:n_hidden_layers_decoder
                    obj.prev_hidden_weight_delta_decoder{i} = zeros(size(obj.hidden_weights_decoder{i}));
                end

                obj.prev_output_weight_delta_decoder = zeros(n_hidden_units_decoder(end), n_outputs_decoder);
            else
                obj.output_weights_decoder = randn(n_hidden_units_encoder(end), n_outputs_decoder);
                obj.prev_output_weight_delta_decoder = zeros(n_hidden_units_encoder(end), n_outputs_decoder);
            end
        end


        function [outputs_decoder, hidden_outputs_encoder, hidden_outputs_decoder] = forwardDecoder(obj, inputs)
            hidden_outputs_encoder = cell(1, obj.n_hidden_layers_encoder);
            for i = 1:obj.n_hidden_layers_encoder
                if i == 1
                    hidden_inputs_encoder = inputs * obj.hidden_weights_encoder{i};
                else
                    hidden_inputs_encoder = hidden_outputs_encoder{i-1} * obj.hidden_weights_encoder{i};
                end
                hidden_outputs_encoder{i} = sigmoid(hidden_inputs_encoder);
            end
            
            if obj.n_hidden_layers_decoder >= 1
                hidden_outputs_decoder = cell(1, obj.n_hidden_layers_decoder);
                for i = 1:obj.n_hidden_layers_decoder
                    if i == 1
                        hidden_inputs_decoder = hidden_outputs_encoder{end} * obj.hidden_weights_decoder{i};
                    else
                        hidden_inputs_decoder = hidden_outputs_decoder{i-1} * obj.hidden_weights_decoder{i};
                    end
                    hidden_outputs_decoder{i} = sigmoid(hidden_inputs_decoder);
                end
                outputs_decoder = sigmoid(hidden_outputs_decoder{end} * obj.output_weights_decoder);
            else
                hidden_outputs_decoder = 0;
                outputs_decoder = sigmoid(hidden_outputs_encoder{end} * obj.output_weights_decoder);
            end   
        end
		
		function outputs_decoder = forwardDecoderFromHidden(obj, inputs)
            if obj.n_hidden_layers_decoder >= 1
                hidden_outputs_decoder = cell(1, obj.n_hidden_layers_decoder);
                for i = 1:obj.n_hidden_layers_decoder
                    if i == 1
                        hidden_inputs_decoder = inputs * obj.hidden_weights_decoder{i};
                    else
                        hidden_inputs_decoder = hidden_outputs_decoder{i-1} * obj.hidden_weights_decoder{i};
                    end
                    hidden_outputs_decoder{i} = sigmoid(hidden_inputs_decoder);
                end
                outputs_decoder = sigmoid(hidden_outputs_decoder{end} * obj.output_weights_decoder);
            else
                hidden_outputs_decoder = 0;
                outputs_decoder = sigmoid(inputs * obj.output_weights_decoder);
            end   
        end

        function train_encoder_decoder_v1(obj, inputs, targets_decoder)
            [outputs_decoder, hidden_outputs_encoder, hidden_outputs_decoder] = obj.forwardDecoder(inputs);
            
            output_errors_decoder = (outputs_decoder - targets_decoder) .* sigmoid_derivative(outputs_decoder);
            if obj.n_hidden_layers_decoder >= 1 
                hidden_errors_decoder = cell(1, obj.n_hidden_layers_decoder);
                hidden_errors_decoder{end} = output_errors_decoder * obj.output_weights_decoder' .* sigmoid_derivative(hidden_outputs_decoder{end});
                for i = obj.n_hidden_layers_decoder-1:-1:1
                    hidden_errors_decoder{i} = hidden_errors_decoder{i+1} * obj.hidden_weights_decoder{i+1}' .* sigmoid_derivative(hidden_outputs_decoder{i});
                end
                obj.prev_output_weight_delta_decoder = obj.momentum * obj.prev_output_weight_delta_decoder - obj.learning_rate * hidden_outputs_decoder{end}' * output_errors_decoder;
                obj.output_weights_decoder = obj.output_weights_decoder + obj.prev_output_weight_delta_decoder;
                for i = obj.n_hidden_layers_decoder:-1:1
                    if i == 1
                        hidden_inputs_decoder = hidden_outputs_encoder{end};
                    else
                        hidden_inputs_decoder = hidden_outputs_decoder{i-1};
                    end
                    obj.prev_hidden_weight_delta_decoder{i} = obj.momentum * obj.prev_hidden_weight_delta_decoder{i} - obj.learning_rate * hidden_inputs_decoder' * hidden_errors_decoder{i};
                    obj.hidden_weights_decoder{i} = obj.hidden_weights_decoder{i} + obj.prev_hidden_weight_delta_decoder{i};
                end
            else
                obj.prev_output_weight_delta_decoder = obj.momentum * obj.prev_output_weight_delta_decoder - obj.learning_rate * hidden_outputs_encoder{end}' * output_errors_decoder;
                obj.output_weights_decoder = obj.output_weights_decoder + obj.prev_output_weight_delta_decoder;
            end
            
            hidden_errors_encoder = cell(1, obj.n_hidden_layers_encoder);
            if obj.n_hidden_layers_decoder >= 1
                hidden_errors_encoder{end} = hidden_errors_decoder{1} * obj.hidden_weights_decoder{1}' .* sigmoid_derivative(hidden_outputs_encoder{end}); 
            else
                hidden_errors_encoder{end} = output_errors_decoder * obj.output_weights_decoder' .* sigmoid_derivative(hidden_outputs_encoder{end}); 
            end

            for i = obj.n_hidden_layers_encoder-1:-1:1
                hidden_errors_encoder{i} = hidden_errors_encoder{i+1} * obj.hidden_weights_encoder{i+1}' .* sigmoid_derivative(hidden_outputs_encoder{i});
            end
           
            for i = obj.n_hidden_layers_encoder:-1:1
                if i == 1
                    hidden_inputs_encoder = inputs;
                else
                    hidden_inputs_encoder = hidden_outputs_encoder{i-1};
                end
                obj.prev_hidden_weight_delta_encoder{i} = obj.momentum * obj.prev_hidden_weight_delta_encoder{i} - obj.learning_rate * hidden_inputs_encoder' * hidden_errors_encoder{i};
                obj.hidden_weights_encoder{i} = obj.hidden_weights_encoder{i} + obj.prev_hidden_weight_delta_encoder{i};
            end
        end

        function train_encoder_decoder_v2(obj, inputs, targets_decoder, p, epoch)

            xn = 0;
			tmp = 0.0;

            while tmp < 1.0-p && xn < epoch
                [outputs_decoder, hidden_outputs_encoder, hidden_outputs_decoder] = obj.forwardDecoder(inputs);

                output_errors_decoder = (outputs_decoder - targets_decoder) .* sigmoid_derivative(outputs_decoder);
                if obj.n_hidden_layers_decoder >= 1
                    hidden_errors_decoder = cell(1, obj.n_hidden_layers_decoder);
                    hidden_errors_decoder{end} = output_errors_decoder * obj.output_weights_decoder' .* sigmoid_derivative(hidden_outputs_decoder{end});
                    for i = obj.n_hidden_layers_decoder-1:-1:1
                        hidden_errors_decoder{i} = hidden_errors_decoder{i+1} * obj.hidden_weights_decoder{i+1}' .* sigmoid_derivative(hidden_outputs_decoder{i});
                    end
                    obj.prev_output_weight_delta_decoder = obj.momentum * obj.prev_output_weight_delta_decoder - obj.learning_rate * hidden_outputs_decoder{end}' * output_errors_decoder;
                    obj.output_weights_decoder = obj.output_weights_decoder + obj.prev_output_weight_delta_decoder;
                    for i = obj.n_hidden_layers_decoder:-1:1
                        if i == 1
                            hidden_inputs_decoder = hidden_outputs_encoder{end};
                        else
                            hidden_inputs_decoder = hidden_outputs_decoder{i-1};
                        end
                        obj.prev_hidden_weight_delta_decoder{i} = obj.momentum * obj.prev_hidden_weight_delta_decoder{i} - obj.learning_rate * hidden_inputs_decoder' * hidden_errors_decoder{i};
                        obj.hidden_weights_decoder{i} = obj.hidden_weights_decoder{i} + obj.prev_hidden_weight_delta_decoder{i};
                    end
                else
                    obj.prev_output_weight_delta_decoder = obj.momentum * obj.prev_output_weight_delta_decoder - obj.learning_rate * hidden_outputs_encoder{end}' * output_errors_decoder;
                    obj.output_weights_decoder = obj.output_weights_decoder + obj.prev_output_weight_delta_decoder;
                end

                hidden_errors_encoder = cell(1, obj.n_hidden_layers_encoder);
                if obj.n_hidden_layers_decoder >= 1
                    hidden_errors_encoder{end} = hidden_errors_decoder{1} * obj.hidden_weights_decoder{1}' .* sigmoid_derivative(hidden_outputs_encoder{end});
                else
                    hidden_errors_encoder{end} = output_errors_decoder * obj.output_weights_decoder' .* sigmoid_derivative(hidden_outputs_encoder{end});
                end

                for i = obj.n_hidden_layers_encoder-1:-1:1
                    hidden_errors_encoder{i} = hidden_errors_encoder{i+1} * obj.hidden_weights_encoder{i+1}' .* sigmoid_derivative(hidden_outputs_encoder{i});
                end

                for i = obj.n_hidden_layers_encoder:-1:1
                    if i == 1
                        hidden_inputs_encoder = inputs;
                    else
                        hidden_inputs_encoder = hidden_outputs_encoder{i-1};
                    end
                    obj.prev_hidden_weight_delta_encoder{i} = obj.momentum * obj.prev_hidden_weight_delta_encoder{i} - obj.learning_rate * hidden_inputs_encoder' * hidden_errors_encoder{i};
                    obj.hidden_weights_encoder{i} = obj.hidden_weights_encoder{i} + obj.prev_hidden_weight_delta_encoder{i};
                end

                tmp = reportModel(output_errors_decoder, p);

                xn = xn + 1;
                if mod(xn,100) == 0
                    p = p + 0.01; %Progressively increase the value of p,
                end
            end
        end

        function outputs = predictClassifier(obj, inputs)
            [outputs, ~] = obj.forwardClassifier(inputs);
        end

        function outputs = predictDecoder(obj, inputs)
            [outputs, ~] = obj.forwardDecoder(inputs);
        end
    
        function set_learning_rate(obj, learning_rate)
            obj.learning_rate = learning_rate;
        end
    
        function set_momentum(obj, momentum)
            obj.momentum = momentum;
        end
    end
end

function count = reportModel(errors, p)
    count = 0;
    avg_Errors = sum(errors,2)/size(errors,2);
    length  = size(errors,1);
    for i = 1:length
        if avg_Errors(i) < p
            count = count + 1;
        end
    end
    count = count/length;
end

function y = sigmoid(x)
    y = 1./(1+exp(-x));
end

function y = sigmoid_derivative(x)
    y = sigmoid(x) .* (1-sigmoid(x));
end