# coding=utf-8
from __future__ import absolute_import, division, print_function

import logging
import argparse
import os
import random
import numpy as np
import time
from datetime import timedelta
import torch
import torch.distributed as dist
from tqdm import tqdm
from torch.nn.parallel import DistributedDataParallel as DDP
import torch.nn.functional as F
import wandb

from models.modeling import VisionTransformer, CONFIGS
from utils.scheduler import WarmupLinearSchedule, WarmupCosineSchedule
from utils.data_utils import get_loader
from utils.dist_util import get_world_size
from utils.exp_utils import kl_div_logits


logger = logging.getLogger(__name__)


class AverageMeter(object):
    """Computes and stores the average and current value"""
    def __init__(self):
        self.reset()

    def reset(self):
        self.val = 0
        self.avg = 0
        self.sum = 0
        self.count = 0

    def update(self, val, n=1):
        self.val = val
        self.sum += val * n
        self.count += n
        self.avg = self.sum / self.count


def simple_accuracy(preds, labels):
    return (preds == labels).mean()


def save_model(args, teacher, student):
    model_to_save = teacher.module if hasattr(teacher, 'module') else teacher
    model_checkpoint = os.path.join(args.save, "%s_teacher_checkpoint.bin" % args.exp_name)
    torch.save(model_to_save.state_dict(), model_checkpoint)
    logger.info("Saved teacher model checkpoint to [DIR: %s]", args.save)
    if student:
        model_to_save = student.module if hasattr(student, 'module') else student
        model_checkpoint = os.path.join(args.save, "%s_student_checkpoint.bin" % args.exp_name)
        torch.save(model_to_save.state_dict(), model_checkpoint)
        logger.info("Saved student model checkpoint to [DIR: %s]", args.save)


def setup(args):
    # Prepare model
    config = CONFIGS[args.model_type]

    num_classes = 10 if args.dataset == "cifar10" else 100
    if args.dataset == "imagenet":
        num_classes=1000

    model = VisionTransformer(config, args.img_size, zero_head=True, num_classes=num_classes,alpha=args.alpha, head_init=args.head_init)
    # print('model architecture:', model)
    model.load_from(np.load(args.pretrained_dir))
    model.to(args.device)
    num_params = count_parameters(model)

    logger.info("{}".format(config))
    logger.info("Training parameters %s", args)
    logger.info("Total Parameter: \t%2.1fM" % num_params)
    print(num_params)
    return args, model


def count_parameters(model):
    params = sum(p.numel() for p in model.parameters() if p.requires_grad)
    return params/1000000


def set_seed(args):
    random.seed(args.seed)
    np.random.seed(args.seed)
    torch.manual_seed(args.seed)
    if args.n_gpu > 0:
        torch.cuda.manual_seed_all(args.seed)


def choose_loss(teacher_pred, student_pred, targets, type, calculate, args):
    if not isinstance(student_pred, int) and type=='kl_ce':
        if calculate == 'teacher':
            ce_loss = F.cross_entropy(teacher_pred, targets)
            lot_loss = args.alpha*kl_div_logits(teacher_pred, student_pred.detach(), args.T)
        else:
            ce_loss = F.cross_entropy(student_pred, targets) 
            lot_loss = args.alpha*kl_div_logits(student_pred, teacher_pred.detach(), args.T)
    elif not isinstance(student_pred, int) and args.loss=='kl':
        if calculate == 'teacher':
            ce_loss = F.cross_entropy(teacher_pred, targets)
            lot_loss = args.alpha*kl_div_logits(teacher_pred, student_pred.detach(), args.T)
        else:
            ce_loss = 0
            lot_loss = kl_div_logits(student_pred, teacher_pred.detach(), args.T)
    elif not isinstance(student_pred, int) and args.loss=='symmetric_kl':
        if calculate == 'teacher':
            ce_loss = F.cross_entropy(teacher_pred, targets)
            lot_loss = args.alpha*(kl_div_logits(teacher_pred, student_pred.detach(), args.T)+kl_div_logits(student_pred.detach(), teacher_pred, args.T))
        else:
            ce_loss = 0
            lot_loss = kl_div_logits(student_pred, teacher_pred.detach(), args.T) + kl_div_logits(teacher_pred.detach(), student_pred, args.T)
    elif not isinstance(student_pred, int) and args.loss=='symmetric_kl_ce':
        if calculate == 'teacher':
            ce_loss = F.cross_entropy(teacher_pred, targets)
            lot_loss = args.alpha*(kl_div_logits(teacher_pred, student_pred.detach(), args.T)+kl_div_logits(student_pred.detach(), teacher_pred, args.T))
        else:
            ce_loss = F.cross_entropy(student_pred, targets)
            lot_loss = args.alpha * (kl_div_logits(student_pred, teacher_pred.detach(), args.T) + kl_div_logits(teacher_pred.detach(), student_pred, args.T))
            total_loss = ce_loss + lot_loss
    else:
        ce_loss = F.cross_entropy(teacher_pred, targets)
        lot_loss = 0
    total_loss = ce_loss + lot_loss
        
    return total_loss, ce_loss, lot_loss


def valid(args, model, test_loader, global_step):
    # Validation!
    eval_losses = AverageMeter()

    logger.info("***** Running Validation *****")
    logger.info("  Num steps = %d", len(test_loader))
    logger.info("  Batch size = %d", args.eval_batch_size)

    model.eval()
    all_preds, all_label = [], []
    epoch_iterator = tqdm(test_loader,
                          desc="Validating... (loss=X.X)",
                          bar_format="{l_bar}{r_bar}",
                          dynamic_ncols=True,
                          disable=args.local_rank not in [-1, 0])
    loss_fct = torch.nn.CrossEntropyLoss()
    for step, batch in enumerate(epoch_iterator):
        batch = tuple(t.to(args.device) for t in batch)
        x, y = batch
        with torch.no_grad():
            with torch.cuda.amp.autocast():
                logits = model(x)[0]
                eval_loss = loss_fct(logits, y)

            eval_losses.update(eval_loss.item())
            preds = torch.argmax(logits, dim=-1)

        if len(all_preds) == 0:
            all_preds.append(preds.detach().cpu().numpy())
            all_label.append(y.detach().cpu().numpy())
        else:
            all_preds[0] = np.append(
                all_preds[0], preds.detach().cpu().numpy(), axis=0
            )
            all_label[0] = np.append(
                all_label[0], y.detach().cpu().numpy(), axis=0
            )
        epoch_iterator.set_description("Validating... (loss=%2.5f)" % eval_losses.val)

    all_preds, all_label = all_preds[0], all_label[0]
    accuracy = simple_accuracy(all_preds, all_label)
    accuracy_tensor = torch.tensor(accuracy, device = args.device)
    dist.all_reduce(accuracy_tensor, op=dist.ReduceOp.SUM)
    accuracy_tensor /= dist.get_world_size()
    accuracy = accuracy_tensor.item()

    logger.info("\n")
    logger.info("Validation Results")
    logger.info("Global Steps: %d" % global_step)
    logger.info("Valid Loss: %2.5f" % eval_losses.avg)
    logger.info("Valid Accuracy: %2.5f" % accuracy)

    return accuracy


def lot_train(args, teacher, student, train_loader, test_loader, teacher_optimizer, student_optimizer, teacher_scheduler, student_scheduler, global_step, t_total, best_acc):
    scaler = torch.cuda.amp.GradScaler()
    teacher.zero_grad()
    if student:
        student.zero_grad()
    losses = AverageMeter()
    while True:
        teacher.train()
        if student:
            student.train()
        epoch_iterator = tqdm(train_loader, desc="Training (X / X Steps) (loss=X.X)", bar_format="{l_bar}{r_bar}", dynamic_ncols=True,
                              disable=args.local_rank not in [-1, 0])
        for step, batch in enumerate(epoch_iterator):
            batch = tuple(t.to(args.device) for t in batch)
            x, y = batch
            with torch.cuda.amp.autocast():
                teacher_pred = F.log_softmax(teacher(x)[0], dim = -1)
                if student:
                    student_pred = F.log_softmax(student(x)[0], dim = -1)
                else:
                    student_pred = 0
                args.alpha = args.original_alpha * (1 + np.random.normal(0, 0.01))
                if args.alpha < 0:
                    args.alpha = args.original_alpha
                teacher_loss, teacher_ce_loss, teacher_lot_loss = choose_loss(teacher_pred, student_pred, y, args.loss, 'teacher', args)
                if student:
                    args.alpha = args.original_alpha * (1 + np.random.normal(0, 0.01))
                    if args.alpha < 0:
                        args.alpha = args.original_alpha
                    student_loss, student_ce_loss, student_lot_loss = choose_loss(teacher_pred, student_pred, y, args.loss, 'student', args)
            if args.gradient_accumulation_steps > 1:
                teacher_loss = teacher_loss / args.gradient_accumulation_steps
                if student:
                    student_loss = student_loss / args.gradient_accumulation_steps
            if args.fp16:
                scaler.scale(teacher_loss).backward()
                if student:
                    scaler.scale(student_loss).backward()
            else:
                teacher_loss.backward()
                if student:
                    student_loss.backward()

            if (step + 1) % args.gradient_accumulation_steps == 0:
                losses.update(teacher_loss.item()*args.gradient_accumulation_steps)
                if args.fp16:
                    scaler.unscale_(teacher_optimizer)
                torch.nn.utils.clip_grad_norm_(teacher.parameters(), args.max_grad_norm)
                scaler.step(teacher_optimizer)
                teacher_optimizer.zero_grad()
                if global_step <= 500:
                    teacher_scheduler.step()
                if student:
                    scaler.unscale_(student_optimizer)
                    torch.nn.utils.clip_grad_norm_(student.parameters(), args.max_grad_norm)
                    scaler.step(student_optimizer)
                    student_optimizer.zero_grad()
                    if global_step <= 500:
                        student_scheduler.step()
                scaler.update()
                global_step += 1

                epoch_iterator.set_description("Training (%d / %d Steps) (loss=%2.5f)" % (global_step, t_total, losses.avg))
                if args.local_rank in [-1, 0]:
                    wandb.log({'teacher_lr': teacher_scheduler.get_last_lr()[0], 'teacher_train_loss': losses.avg, \
                    'teacher_ce_loss': teacher_ce_loss, 'teacher_lot_loss': teacher_lot_loss}, step=global_step)
                # if global_step % args.eval_every == 0 and args.local_rank in [-1, 0]:
                if global_step % args.eval_every == 0:
                    teacher_accuracy = valid(args, teacher, test_loader, global_step)
                    if student:
                        student_accuracy = valid(args, student, test_loader, global_step)
                    else:
                        student_accuracy = 0
                    if args.local_rank in [-1, 0]:
                        print("Teacher Accuracy:",teacher_accuracy)
                        if student:
                            print("Student Accuracy:",student_accuracy)
                        wandb.log({'teacher test acc': 100. * teacher_accuracy, 'student test acc': 100. * student_accuracy}, step=global_step)
                        if best_acc < teacher_accuracy:
                            save_model(args, teacher, student)
                            best_acc = teacher_accuracy
                    teacher.train()
                    if student:
                        student.train()

                if global_step % t_total == 0:
                    break
        # losses.reset()
        if global_step % t_total == 0:
            break
    return best_acc


def train(args, teacher, student):
    """ Train the model """
    if args.local_rank in [-1, 0]:
        os.makedirs(args.save, exist_ok=True)
        # writer = SummaryWriter(log_dir=os.path.join("logs", args.exp_name))
        wandb_username=os.environ.get('WANDB_USER_NAME')
        wandb_key=os.environ.get('WANDB_API_KEY')    
        wandb.login(key=wandb_key)
        wandb.init(project='LoT_Image_Classification_'+args.dataset, entity=wandb_username, name=args.exp_name)

    args.train_batch_size = args.train_batch_size // args.gradient_accumulation_steps

    # Prepare dataset
    train_loader, test_loader = get_loader(args)

    # Prepare optimizer and scheduler
    teacher_optimizer = torch.optim.SGD(teacher.parameters(), lr=args.learning_rate, momentum=0.9, weight_decay=args.weight_decay)
    t_total = args.num_steps
    if args.decay_type == "cosine":
        teacher_scheduler = WarmupCosineSchedule(teacher_optimizer, warmup_steps=args.warmup_steps, t_total=t_total)
    else:
        teacher_scheduler = WarmupLinearSchedule(teacher_optimizer, warmup_steps=args.warmup_steps, t_total=t_total)
    if student:
        student_optimizer = torch.optim.SGD(student.parameters(), lr=args.learning_rate, momentum=0.9, weight_decay=args.weight_decay)
        if args.decay_type == "cosine":
            student_scheduler = WarmupCosineSchedule(student_optimizer, warmup_steps=args.warmup_steps, t_total=t_total)
        else:
            student_scheduler = WarmupLinearSchedule(student_optimizer, warmup_steps=args.warmup_steps, t_total=t_total)
    else:
        student_optimizer = None
        student_scheduler = None

    # Distributed training
    if args.local_rank != -1:
        teacher = DDP(teacher, device_ids=[args.local_rank], output_device=args.local_rank)
        if student:
            student = DDP(student, device_ids=[args.local_rank], output_device=args.local_rank)
    # Train!
    logger.info("***** Running training *****")
    logger.info("  Total optimization steps = %d", args.num_steps)
    logger.info("  Instantaneous batch size per GPU = %d", args.train_batch_size)
    logger.info("  Total train batch size (w. parallel, distributed & accumulation) = %d",
                args.train_batch_size * args.gradient_accumulation_steps * (
                    torch.distributed.get_world_size() if args.local_rank != -1 else 1))
    logger.info("  Gradient Accumulation steps = %d", args.gradient_accumulation_steps)

    set_seed(args)  # Added here for reproducibility (even between python 2 and 3)
    best_acc = lot_train(args, teacher, None, train_loader, test_loader, teacher_optimizer, None, teacher_scheduler, None, 0, args.warmup_steps, 0)
    if student:
        lot_train(args, student, None, train_loader, test_loader, student_optimizer, None, student_scheduler, None, 0, args.warmup_steps, 0)
    best_acc = lot_train(args, teacher, student, train_loader, test_loader, teacher_optimizer, student_optimizer, teacher_scheduler, student_scheduler, args.warmup_steps, t_total, best_acc)

    if args.local_rank in [-1, 0]:
        wandb.finish()
    logger.info("Best Accuracy: \t%f" % best_acc)
    logger.info("End Training!")
    print("Best Accuracy:",best_acc)


def main():
    parser = argparse.ArgumentParser()
    # LoT
    parser.add_argument('--loss', type=str, default='kl_ce', choices=['kl', 'kl_ce', 'symmetric_kl', 'symmetric_kl_ce'])
    parser.add_argument('--teacher_network', type=str, default='ViT-B_16', choices=['ViT-B_16', 'ViT-L_16', 'Swin-B', 'Swin-L'])
    parser.add_argument('--teacher_head', type=str, default='zero', choices=['uniform', 'normal', 'xavier_uniform', 'kaiming_normal', 'zero', 'default', 'pretain'])
    parser.add_argument('--teacher_pretrain', type=str, default='../model/ViT-B_16.npz')
    parser.add_argument('--student_network', type=str, default='', choices=['', 'ViT-B_16', 'ViT-L_16', 'Swin-B', 'Swin-L'])
    parser.add_argument('--student_head', type=str, default='zero', choices=['uniform', 'normal', 'xavier_uniform', 'kaiming_normal', 'zero', 'default', 'pretain'])
    parser.add_argument('--student_pretrain', type=str, default='../model/ViT-B_16.npz')
    parser.add_argument("--alpha", default=1, type=float, help="alpha for LoT loss")
    parser.add_argument("--T", default=1.5, type=float, help="temperature")
    parser.add_argument("--shuffle", default=1, type=int, help="whether shuffle the train dataset")
    # Required parameters
    parser.add_argument("--exp_name", default='debug',
                        help="Name of this run. Used for monitoring.")
    parser.add_argument("--dataset", choices=["cifar10", "cifar100","imagenet"], default="cifar100",
                        help="Which downstream task.")
    parser.add_argument("--model_type", choices=["ViT-B_16", "ViT-B_32", "ViT-L_16",
                                                 "ViT-L_32", "ViT-H_14"],
                        default="ViT-B_16",
                        help="Which variant to use.")
    parser.add_argument("--pretrained_dir", type=str, default="../model/ViT-L_16.npz",
                        help="Where to search for pretrained ViT models.")
    randomhash = ''.join(str(time.time()).split('.'))
    parser.add_argument("--save", default='ckpt/LoT_ViT'+randomhash+'CIFAR', help='path to save the final model')

    parser.add_argument("--img_size", default=384, type=int,
                        help="Resolution size")
    parser.add_argument("--train_batch_size", default=512, type=int,
                        help="Total batch size for training.")
    parser.add_argument("--eval_batch_size", default=32, type=int,
                        help="Total batch size for eval.")
    parser.add_argument("--eval_every", default=200, type=int,
                        help="Run prediction on validation set every so many steps."
                             "Will always run one evaluation at the end of training.")

    parser.add_argument("--learning_rate", default=1e-2, type=float,
                        help="The initial learning rate for SGD.")
    parser.add_argument("--weight_decay", default=0, type=float,
                        help="Weight deay if we apply some.")
    parser.add_argument("--num_steps", default=20000, type=int,
                        help="Total number of training epochs to perform. 20000 for ImageNet, 10000 for cifar100")
    parser.add_argument("--decay_type", choices=["cosine", "linear"], default="cosine",
                        help="How to decay the learning rate.")
    parser.add_argument("--warmup_steps", default=500, type=int,
                        help="Step of training to perform learning rate warmup for.")
    parser.add_argument("--max_grad_norm", default=1.0, type=float,
                        help="Max gradient norm.")
    parser.add_argument("--local_rank", type=int, default=-1,
                        help="local_rank for distributed training on gpus")
    parser.add_argument('--seed', type=int, default=0,
                        help="random seed for initialization")
    parser.add_argument('--gradient_accumulation_steps', type=int, default=16,
                        help="Number of updates steps to accumulate before performing a backward/update pass.")
    parser.add_argument('--fp16', default=True, type=bool,
                        help="Whether to use 16-bit float precision instead of 32-bit")
    parser.add_argument('--fp16_opt_level', type=str, default='O2',
                        help="For fp16: Apex AMP optimization level selected in ['O0', 'O1', 'O2', and 'O3']."
                             "See details at https://nvidia.github.io/apex/amp.html")
    parser.add_argument('--loss_scale', type=float, default=0,
                        help="Loss scaling to improve fp16 numeric stability. Only used when fp16 set to True.\n"
                             "0 (default value): dynamic loss scaling.\n"
                             "Positive power of 2: static loss scaling value.\n")
    args = parser.parse_args()
    if 'LOCAL_RANK' in os.environ:
        args.local_rank = int(os.environ['LOCAL_RANK'])
    args.original_alpha = args.alpha
    # Setup logging
    logging.basicConfig(format='%(asctime)s - %(levelname)s - %(name)s - %(message)s',
                        datefmt='%m/%d/%Y %H:%M:%S',
                        level=logging.INFO if args.local_rank in [-1, 0] else logging.WARN)

    # Setup CUDA, GPU & distributed training
    if args.local_rank == -1:
        device = torch.device("cuda" if torch.cuda.is_available() else "cpu")
        args.n_gpu = torch.cuda.device_count()
    else:  # Initializes the distributed backend which will take care of sychronizing nodes/GPUs
        torch.cuda.set_device(args.local_rank)
        device = torch.device("cuda", args.local_rank)
        dist.init_process_group(backend='nccl', init_method='env://')
        args.n_gpu = 1
        logger.info(f"Process {args.local_rank} is using GPU {torch.cuda.current_device()}")
        logger.info(f"Initialized process group; rank: {dist.get_rank()}, world size: {dist.get_world_size()}")
    args.device = device
    logger.warning("Process rank: %s, device: %s, n_gpu: %s, distributed training: %s, 16-bits training: %s" %
                   (args.local_rank, args.device, args.n_gpu, bool(args.local_rank != -1), args.fp16))

    # Set seed
    set_seed(args)

    # Model & Tokenizer Setup
    args.model_type = args.teacher_network
    args.head_init = args.teacher_head
    args.pretrained_dir=args.teacher_pretrain
    args, teacher = setup(args)
    if args.student_network:
        args.model_type = args.student_network
        args.head_init = args.student_head
        args.pretrained_dir=args.student_pretrain
        args, student = setup(args)
    else:
        student = None
    # Training
    train(args, teacher, student)


if __name__ == "__main__":
    main()
