import torch
from torch_geometric.graphgym.config import cfg
from torch_geometric.graphgym.register import (
    register_edge_encoder,
    register_node_encoder,
)

from graphgps.slt.monarch_linear import MonarchLinear
from graphgps.slt.sparse_modules import (
    NMSparseMultiLinear,
    SparseLinear,
    SparseLinearMulti_mask,
)

"""
=== Description of the VOCSuperpixels dataset ===
Each graph is a tuple (x, edge_attr, edge_index, y)
Shape of x : [num_nodes, 14]
Shape of edge_attr : [num_edges, 1] or [num_edges, 2]
Shape of edge_index : [2, num_edges]
Shape of y : [num_nodes]
"""

VOC_node_input_dim = 14
# VOC_edge_input_dim = 1 or 2; defined in class VOCEdgeEncoder


@register_node_encoder("VOCNode")
class VOCNodeEncoder(torch.nn.Module):
    def __init__(self, emb_dim):
        super().__init__()

        if cfg.slt.encoder is True:
            if cfg.monarch.encoder is True:
                self.encoder = MonarchLinear(
                    VOC_node_input_dim, emb_dim, bias=False
                )
            elif cfg.slt.srste is True:
                self.encoder = NMSparseMultiLinear(VOC_node_input_dim, emb_dim)
            elif cfg.slt.sm is True:
                self.encoder = SparseLinear(
                    VOC_node_input_dim, emb_dim, bias=False
                )
            elif cfg.slt.mm is True:
                self.encoder = SparseLinearMulti_mask(
                    VOC_node_input_dim, emb_dim, bias=False
                )
        else:
            self.encoder = torch.nn.Linear(VOC_node_input_dim, emb_dim)
        # torch.nn.init.xavier_uniform_(self.encoder.weight.data)

    def forward(
        self,
        batch,
        cur_epoch=None,
        mpnn_th=None,
        msa_th=None,
        ffn_th=None,
        encoder_th=None,
        pred_th=None,
        global_th=None,
    ):
        if (
            cfg.slt.sm is True or cfg.slt.mm is True
        ) and cfg.slt.encoder is True:
            if cfg.slt.pruning == "blockwise":
                batch.x = self.encoder(batch.x, encoder_th)
            elif cfg.slt.pruning == "global":
                batch.x = self.encoder(batch.x, global_th)
        else:
            batch.x = self.encoder(batch.x)

        return batch


@register_edge_encoder("VOCEdge")
class VOCEdgeEncoder(torch.nn.Module):
    def __init__(self, emb_dim):
        super().__init__()

        VOC_edge_input_dim = (
            2 if cfg.dataset.name == "edge_wt_region_boundary" else 1
        )

        if cfg.slt.encoder is True:
            if cfg.monarch.encoder is True:
                self.encoder = MonarchLinear(
                    VOC_edge_input_dim, emb_dim, bias=False
                )
            if cfg.slt.srste is True:
                self.encoder = SparseLinearMulti_mask(
                    VOC_edge_input_dim, emb_dim, bias=False
                )
            if cfg.slt.sm is True:
                self.encoder = SparseLinear(
                    VOC_edge_input_dim, emb_dim, bias=False
                )
            elif cfg.slt.mm is True:
                self.encoder = SparseLinearMulti_mask(
                    VOC_edge_input_dim, emb_dim, bias=False
                )
        else:
            self.encoder = torch.nn.Linear(VOC_edge_input_dim, emb_dim)

        # torch.nn.init.xavier_uniform_(self.encoder.weight.data)

    def forward(
        self,
        batch,
        cur_epoch=None,
        mpnn_th=None,
        msa_th=None,
        ffn_th=None,
        encoder_th=None,
        pred_th=None,
        global_th=None,
    ):
        if (
            cfg.slt.sm is True or cfg.slt.mm is True
        ) and cfg.slt.encoder is True:
            if cfg.slt.pruning == "blockwise":
                batch.edge_attr = self.encoder(batch.edge_attr, encoder_th)
            elif cfg.slt.pruning == "global":
                batch.edge_attr = self.encoder(batch.edge_attr, global_th)
        else:
            batch.edge_attr = self.encoder(batch.edge_attr)

        return batch
