import copy
from typing import Dict, List, Any, Tuple, Optional

from initialization.agent_Class import SDA_Child, SDA_Daycare



def children_propose_da(
        no_siblings_children: List[SDA_Child], 
        daycares: List[SDA_Daycare], 
        child_next_rank: Dict[int, int]
        ) -> Dict[int, int]:
    """
    Executes the children-propose Deferred Acceptance (DA) algorithm.

    Parameters
    ----------
    no_siblings_children : List[SDA_Child]
        A list of Child objects without siblings.
    daycares : List[SDA_Daycare]
        A list of Daycare objects.
    child_next_rank : Dict[int, int]
        A dictionary where child_next_rank[child_id] is the preference rank (starting from 0) of the next daycare to propose to for child_id.

    Returns
    -------
    Dict[int, int]
        Updated child_next_rank dictionary.
    """

    capacity: Dict[int, List[int]] = {}
    for daycare in daycares:
        capacity[daycare.id] = copy.copy(daycare.total_numbers)

    # Temporary assignment 
    tmp = dict()
    for daycare in daycares:
        tmp[daycare.id] = [[] for _ in range(6)]


    # Flag to indicate when all proposals are done
    done: bool = False
    while not done:
        done = True
        for child in no_siblings_children:
            if child.assigned_daycare != None:
                continue
            if child_next_rank[child.id] >= len(child.pref):
                continue
            
            done = False
            daycare = child.pref[child_next_rank[child.id]]
            
            if child in daycare.priority:
                if len(tmp[daycare.id][child.age]) < capacity[daycare.id][child.age]:
                    tmp[daycare.id][child.age].append(child)
                    tmp[daycare.id][child.age].sort(key=lambda c: daycare.priority.index(c))
                    child.assigned_daycare = daycare
                elif capacity[daycare.id][child.age] == 0:
                    # Skip if there's no capacity
                    pass
                else:
                    # If the daycare is full, compare priorities to potentially evict a less preferred child
                    opponent = tmp[daycare.id][child.age][-1]
                    if daycare.priority.index(opponent) < daycare.priority.index(child):
                        pass
                    else:
                        # Replace the less preferred child with the current child
                        opponent.assigned_daycare = None
                        tmp[daycare.id][child.age][-1] = child
                        tmp[daycare.id][child.age].sort(key=lambda c: daycare.priority.index(c))
                        child.assigned_daycare = daycare

            # Move to the next preferred daycare for the next proposal
            child_next_rank[child.id] += 1

    for daycare in daycares:
        daycare.assigned_children = tmp[daycare.id]

    return child_next_rank