import copy
import time
from typing import Dict, List, Any, Tuple, Optional
from initialization.setup_inputs import SDA_Child, SDA_Daycare



def choice_function_for_family(
        family_children_pref_dict: Dict[SDA_Child, SDA_Daycare]
    ) -> Tuple[
        int, 
        List[SDA_Child], 
        Dict[SDA_Daycare, List[List[SDA_Child]]]
    ]:
    """
    A choice function to check whether the current family's application to a tuple of daycares is accepted,
    and which children are displaced as a result of this application.

    Parameters
    ----------
    family_children_pref_dict : Dict[SDA_Child, SDA_Daycare]
        A dictionary mapping each child object from the family to the daycare they are applying to next,
        according to the family's preference tuple.

    Returns
    -------
    Tuple[
        int, 
        List[SDA_Child], 
        Dict[SDA_Daycare, List[List[SDA_Child]]]
    ]
        - An integer indicating if the application is accepted (0 if yes, >0 if no),
        - A list of all children rejected due to this family's application,
        - A dictionary with daycare objects as keys and lists of lists of children newly matched to each daycare, organized by age.
    """

    # Initialize structures for processing applications
    daycares_list_for_family = []
    for child in family_children_pref_dict:
        daycares_list_for_family.append(family_children_pref_dict[child])
    daycares_list_for_family_unique = list(set(daycares_list_for_family))
    # example: family_children_pref_dict = {child 1176102: daycare 2241056, child 1176103: daycare 2241056} 
    # example: daycares_list_for_family_unique = [daycare 2241056]

    daycares_list_each_age_list = dict()
    for daycare in daycares_list_for_family_unique:
        each_age_list = [[] for _ in range(6)]
        for child in family_children_pref_dict:
            if family_children_pref_dict[child] == daycare:
                each_age_list[child.age].append(child)
        daycares_list_each_age_list[daycare] = each_age_list
    # example: each_age_list = [[child 1176103], [], [], [], [], [child 1176102]], daycares_list_each_age_list= {daycare 2241056: [[child 1176103], [], [], [], [], [child 1176102]]}
    rejected_children = {daycare: [[] for _ in range(6)] for daycare in daycares_list_for_family_unique}
    rejected_children_all: List[SDA_Child] = []
    
    new_daycare_match_dict = {daycare: [[] for _ in range(6)] for daycare in daycares_list_for_family_unique}
    accepted = 0
    
    for daycare in daycares_list_for_family_unique:
        rejected_children_list = [[] for _ in range(6)]
        new_daycare_match = [[] for _ in range(6)]
        for i in range(6):
            merged_list = daycare.assigned_children[i] + daycares_list_each_age_list[daycare][i] 

            if len(merged_list) > daycare.total_numbers[i]:
                # If the capacity of the daycare is exceeded,
                merged_list.sort(key=lambda c: daycare.priority.index(c))
                new_daycare_match[i] = merged_list[0:daycare.total_numbers[i]]

                for index in range(daycare.total_numbers[i],len(merged_list)):
                    rejected_children_list[i].append(merged_list[index])
                    rejected_children_all.append(merged_list[index])

                for c in daycares_list_each_age_list[daycare][i]:
                    if merged_list.index(c)+1 > daycare.total_numbers[i]: # Family's application is not fully accepted if any child is rejected
                        accepted+=1
            else:
                new_daycare_match[i] = merged_list # All applications accepted
                rejected_children_list[i] = []

        rejected_children[daycare] = rejected_children_list
        # example: rejected_children[daycare] = [[child 1176103], [], [], [], [], [child 1176102]]
        new_daycare_match_dict[daycare] = new_daycare_match

    rejected_children_all = list(set(rejected_children_all))

    return accepted, rejected_children_all, new_daycare_match_dict