#include <iostream>
#include <Eigen/Dense>
#include "utilities.h"
#include "fsg.h"
#include "fgt.hpp"
#include <chrono>
#include "cluster.h"

void compare_gaussian_transforms(Eigen::MatrixXd points, double bandwidth) {
  // Compute the degrees of each point by the naive method
  auto start = std::chrono::high_resolution_clock::now();
  Eigen::VectorXd gauss_transform_naive = fgt::direct(points, points, bandwidth);
  auto end = std::chrono::high_resolution_clock::now();
  auto duration = std::chrono::duration_cast<std::chrono::milliseconds>(end - start);

  // Display the computed values
  std::cout << "Direct method" << std::endl;
  std::cout << "Time taken: " << duration.count() << " milliseconds" << std::endl;
  std::cout << gauss_transform_naive.transpose() << std::endl;

  // Now, we will compute the degrees of each point using the ifgt method.
  start = std::chrono::high_resolution_clock::now();
  Eigen::VectorXd gauss_transform = fgt::ifgt(points, points, bandwidth, 0.5);
  end = std::chrono::high_resolution_clock::now();
  duration = std::chrono::duration_cast<std::chrono::milliseconds>(end - start);

  // Display the computed values
  std::cout << "IFGT method" << std::endl;
  std::cout << "Time taken: " << duration.count() << " milliseconds" << std::endl;
  std::cout << gauss_transform.transpose() << std::endl;
}

void compare_similarity_graph(Eigen::MatrixXd points, double bandwidth) {
  // Compute the similarity graph fast
  auto start = std::chrono::high_resolution_clock::now();
  IFGT ifgtSolver;
  SprsMat adj = fast_similarity_graph(points, bandwidth, ifgtSolver);
  auto end = std::chrono::high_resolution_clock::now();
  auto duration = std::chrono::duration_cast<std::chrono::milliseconds>(end - start);
  std::cout << "Fast method IFGT time taken: " << duration.count() << " milliseconds" << std::endl;
  std::cout << "Fast method IFGT number of edges: " << adj.nonZeros() / 2 << std::endl;

  // Compute the similarity graph fast
//  start = std::chrono::high_resolution_clock::now();
//  LaplaceKDE kdeSolver;
//  adj = fast_similarity_graph(points, bandwidth, kdeSolver);
//  end = std::chrono::high_resolution_clock::now();
//  duration = std::chrono::duration_cast<std::chrono::milliseconds>(end - start);
//  std::cout << "Fast method KDE time taken: " << duration.count() << " milliseconds" << std::endl;
//  std::cout << "Fast method KDE number of edges: " << adj.nonZeros() / 2 << std::endl;

  // Compute the similarity graph fast
//  start = std::chrono::high_resolution_clock::now();
//  ExponentialKDE expKdeSolver;
//  adj = fast_similarity_graph_debug(points, bandwidth, expKdeSolver);
//  end = std::chrono::high_resolution_clock::now();
//  duration = std::chrono::duration_cast<std::chrono::milliseconds>(end - start);
//  std::cout << "Fast method KDE time taken: " << duration.count() << " milliseconds" << std::endl;
//  std::cout << "Fast method KDE number of edges: " << adj.nonZeros() / 2 << std::endl;

  // Compute the similarity graph directly
//  start = std::chrono::high_resolution_clock::now();
//  adj = similarity_graph(points, bandwidth);
//  end = std::chrono::high_resolution_clock::now();
//  duration = std::chrono::duration_cast<std::chrono::milliseconds>(end - start);
//  std::cout << "Direct method time taken: " << duration.count() << " milliseconds" << std::endl;
//  std::cout << "Direct method number of edges: " << adj.nonZeros() / 2 << std::endl;
}

void time_fast_similarity_graph(Eigen::MatrixXd points, double bandwidth) {
  auto start = std::chrono::high_resolution_clock::now();
  IFGT ifgtSolver;
  auto adj = fast_similarity_graph(points, bandwidth, ifgtSolver);
  auto end = std::chrono::high_resolution_clock::now();
  auto duration = std::chrono::duration_cast<std::chrono::milliseconds>(end - start);
  std::cout << "Fast method time taken: " << duration.count() << " milliseconds" << std::endl;
  std::cout << "Fast method number of edges: " << adj.nonZeros() / 2 << std::endl;
}

int main(int argc, char *argv[]) {
  // The first argument should be an integer type with the number of data points
  int num_points = atoi(argv[1]);

  // Generate random points
  long d = 5;
  Eigen::MatrixXd points = generateRandomPoints(num_points, d);

  // Read the two moons data from file
//  Eigen::MatrixXd points = readDataFromFile("../../data/saved/twomoons30.csv");

  // Define the bandwidth of the gaussian
  double bandwidth = 0.2 * pow(d, 2);

  // Compare similarity graph methods
  compare_similarity_graph(points, bandwidth);

  return 0;
}
