function dx = drum(x,u)
% drum - drum-boiler model (see Eq. (5) in [1])
%
% Syntax:  
%    dx = drum(x,u)
%
% Inputs:
%    x - state vector
%    u - input vector
%
% Outputs:
%    dx - time-derivate of the system state
% 
% References:
%   [1] El-Guindy, Ahmed, Dongkun Han, and Matthias Althoff.
%       "Formal analysis of drum-boiler units to maximize
%       the load-following capabilities of power plants."
%       IEEE Transactions on Power Systems 31.6 (2016): 4691-4702.

% Author:        Ahmed El-Guindy, Matthias Althoff, Mark Wetzlinger
% Written:       22-May-2020
% Last update:   ---
% Last revision: ---

%------------- BEGIN CODE --------------

    P      = x(1);
    Vwt    = x(2);
    alphaR = x(3);
    Vsd    = x(4);
    INT_ep = x(5);
    INT_el = x(6);
    Q      = x(7);
    rf_s   = x(8);
    rf     = x(9);
    rs     = x(10);
    qf     = x(11);
    qf_set = x(12);

    % Input Data interpolation
    Pd = (u(1)/4.75)*1e6;
    ep = u(2);
    el = u(3);

    % Equations
    %%%%%%%%%%%%%%%%%%%%%%%%
    % Coefficients Values  %
    %%%%%%%%%%%%%%%%%%%%%%%%
    L = [ 4.9929e+03   1.9811e-13;
          1.0591e-01   2.8611e-05;
          3.9621e-04   6.2676e-06;
          9.6688e-01   1.1409e-02];

    dp     = L(1,1) * ep + L(1,2) * el;
    dVwt   = L(2,1) * ep + L(2,2) * el;
    dalpha = L(3,1) * ep + L(3,2) * el;
    dVsd   = L(4,1) * ep + L(4,2) * el;


    k = [ -5.9159e-06   4.6299e+00  -5.5686e+01   4.0822e-02   1.3350e-03  -9.9822e-05;
          -7.1831e-05  -1.8986e+00   2.0272e+01  -9.1418e-03   4.9911e-02   1.3350e-05];

    F = k(:,1:4);                 % State-feedback matrix
    I = k(:,5:6);                 % Integrated error feedback matrix

    qf_set = -I(1,:)*[x(5);x(6)] - F(1,:)*[x(1);x(2);x(3);x(4)];
    rs_set = -I(2,:)*[x(5);x(6)] - F(2,:)*[x(1);x(2);x(3);x(4)];


    Pbar = P/100000;
    qs = (rs * 14.2 * 28 * (-0.0018*Pbar^2 +  0.7287 *Pbar +  0.0545))/3600;

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %  State equations Coefficients  %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    drf_dt = rf_s - rf;
    drs_dt = rs_set - rs;

    dQ_dt  = (Pd - Q) / 50;
    dqf_dt = (0.16*rf - qf) / 22;


    dP_dt      = dp     + (4431091408691301233543*P)/18889465931478580854784000 + (7000843059508131823*Q)/151115727451828646838272 - (21914442891022511209991*Vwt)/9223372036854775808000 + (289974090407367762109*qf)/11529215046068469760 - (35299555640870761665*qs)/288230376151711744 + (760787597155687*P*Q)/2417851639229258349412352 + (5012809718166005*P*Vwt)/1180591620717411303424 - (1809768406721775*Q*Vwt)/295147905179352825856 - (4667500361383903*P*qf)/36893488147419103232 - (2844460994510281*P*qs)/4611686018427387904 + (5592620768356823*Vwt*qf)/4503599627370496 + (7264524728204535*Vwt*qs)/562949953421312 - (1052674547934277*P^2)/4835703278458516698824704 + (2395298212451809*Vwt^2)/1152921504606846976 - 759984743377850974279510149/11805916207174113034240000;
    dVwt_dt    = dVwt   + (86703963885804306927823*P)/633825300114114700748351602688000 - (1044598028519639181733*Q)/25353012004564588029934064107520 + (35326845737030021297759*Vwt)/19342813113834066795298816000 + (1606211079452521464037*qf)/3022314549036572936765440 - (325230335649710146561*qs)/755578637259143234191360 + (3159231635426647*P*Q)/81129638414606681695789005144064 - (8082408976705149*P*Vwt)/2475880078570760549798248448 + (2884544074301799*Q*Vwt)/618970019642690137449562112 + (394252884839977*P*qf)/9671406556917033397649408 - (341963328121141*P*qs)/2417851639229258349412352 - (2228484185775287*Vwt*qf)/2361183241434822606848 - (2894685540913907*Vwt*qs)/295147905179352825856 - (4407165544355901*P^2)/162259276829213363391578010288128 - (3817805217093597*Vwt^2)/2417851639229258349412352 - 26870560054952031512420235069/396140812571321687967719751680000;
    dalphaR_dt = dalpha + (6301873288273267601163*P)/19807040628566084398385987584000 + (10939805537486237193203*Q)/316912650057057350374175801344000 + (8472700283444522853857*Vwt)/15474250491067253436239052800 - (11189002506933424924667*alphaR)/59029581035870565171200 + (9096253283969497010127*qf)/4835703278458516698824704000 + (64049643402864183018719*qs)/604462909807314587353088000 - (5097768176535769*P*Q)/40564819207303340847894502572032 - (9646635249125807*P*Vwt)/9903520314283042199192993792 + (3417880673076733*Q*Vwt)/2475880078570760549798248448 + (15993886362761921*P*alphaR)/37778931862957161709568 + (1993241914101697*Q*alphaR)/302231454903657293676544 - (57884826682979*Vwt*alphaR)/295147905179352825856 + (10921371064046739*P*qf)/618970019642690137449562112 + (2030000048888163*P*qs)/77371252455336267181195264 - (5281037718698823*Vwt*qf)/18889465931478580854784 - (6859794484034061*Vwt*qs)/2361183241434822606848 - (2162422246697033*alphaR*qf)/18446744073709551616 - (2808874503494301*alphaR*qs)/2305843009213693952 - (6659090938598285*P^2)/2535301200456458802993406410752 - (2261848376893699*Vwt^2)/4835703278458516698824704 - (3564366098975703*alphaR^2)/281474976710656 + 10314978135458578334985475811/38685626227668133590597632000000;
    dVsd_dt    = dVsd   + (48542506824784479959453*P)/9671406556917033397649408000 + (46753763999439745481119*Q)/154742504910672534362390528000 - (252060406412642747180141*Vsd)/1511157274518286468382720 - (137039605074858273491367*Vwt)/75557863725914323419136000 - (4090038174194490291168623*alphaR)/4611686018427387904000 - (8172845051130959451397*qf)/236118324143482260684800 + (11378273715854323547287*qs)/29514790517935282585600 - (6811884438987501*P*Q)/19807040628566084398385987584 + (570074554743*P*Vsd)/2361183241434822606848 + (7817464254376747*P*Vwt)/2417851639229258349412352 - (3276039153640547*Q*Vsd)/9671406556917033397649408 - (5390146415747661*Q*Vwt)/1208925819614629174706176 + (8671928075981397*Vsd*Vwt)/75557863725914323419136 + (864963653354097*P*alphaR)/576460752303423488 + (8691238397954233*Q*alphaR)/2361183241434822606848 + (3755592736062739*Vwt*alphaR)/9223372036854775808 - (288493496816405*P*qf)/151115727451828646838272 - (13763890961911561*P*qs)/18889465931478580854784 + (2530937734981815*Vsd*qf)/36893488147419103232 + (4164213039252819*Vwt*qf)/4611686018427387904 + (6575114073655815*Vsd*qs)/9223372036854775808 + (1352274266848419*Vwt*qs)/144115188075855872 + (2192170250869865*alphaR*qf)/9007199254740992 + (177969726263925*alphaR*qs)/70368744177664 - (1464725153710969*P^2)/154742504910672534362390528 + (7134066451011255*Vwt^2)/4722366482869645213696 - (6736899692695867*alphaR^2)/549755813888 + 42627375707253154860327278031/75557863725914323419136000000;

    dIntep_dt  = 11/2 - x(1)/100000;
    dIntel_dt  = 0    - ((10000*Vsd)/147 - (25216104004394103980739*P)/18889465931478580854784000 + (10000*Vwt)/147 + (16632237092407349809569*alphaR)/281474976710656000 - (681597363781347*P*alphaR)/36028797018963968 + (4902069963793993*P^2)/4835703278458516698824704 - (2299094516071751*alphaR^2)/2147483648 - 65372658936328482820208035816109/43386742061364865400832000000);

    dqf_set_dt = -I(1,:)*[dIntep_dt;dIntel_dt] - F(1,:)*[dP_dt;dVwt_dt ;dalphaR_dt;dVsd_dt];
    drf_s_dt   = (4/15)*(qf_set - qf) + 4*(dqf_set_dt - dqf_dt);

    %dx = zeros(12,1);
    dx(1,1) = dP_dt ;
    dx(2,1) = dVwt_dt;
    dx(3,1) = dalphaR_dt;
    dx(4,1) = dVsd_dt;
    dx(5,1) = dIntep_dt;
    dx(6,1) = dIntel_dt;
    dx(7,1) = dQ_dt;
    dx(8,1) = drf_s_dt;
    dx(9,1) = drf_dt;
    dx(10,1) = drs_dt;
    dx(11,1) = dqf_dt;
    dx(12,1) = dqf_set_dt;

end

%------------- END OF CODE --------------