function gnn_eval_plots(evalpath)

if nargin < 1
    % for local debugging
    evalpath = sprintf("./gnn-verification/results/240416-122102/evaluation");
end

modelEnymes = "Enzymes_34035";
modelProteins = "Proteins_314496";
modelCora2MP = "Cora_332314";
modelCora3MP = "Cora_717338";

% plot time comparison
aux_plotTimeComparison(evalpath,{modelEnymes,modelProteins});

% plot verified instances
pertEdges = [0,0.001,0.005,0.01,0.05];

delta = 0.001;
aux_plotVerifiedInstances(evalpath,{modelEnymes,modelProteins},delta,pertEdges);

delta = 0.000;
aux_plotVerifiedInstances(evalpath,{modelCora2MP,modelCora3MP},delta,pertEdges);

end


% Auxiliary functions -----------------------------------------------------

function name = aux_getEvResultName(evalpath,model,numPertEdges,delta,enum)
    name = sprintf('%s/models/%s/evResult-edge%.3f-d%.3f-enum%i.mat', ...
        evalpath, model, numPertEdges, delta, enum);
end

function aux_plotTimeComparison(evalpath,models)

    % settings
    delta = 0.001;
    pertEdges = [0,1,2,3,4,5,6,7,8,9];

    % init figure
    figure;
    colorSet = [     0    0.4470    0.7410]; % blue
    colorEnum = [    0.8500    0.3250    0.0980]; % red
    facealpha = 0.2;
    
    for m=1:numel(models)
        model = models{m};
        
        % show time comparison for model m

        subplot(1,numel(models),m); hold on;
        
        % init plotting data
        timeEnumMean = nan(1,numel(pertEdges));
        timeEnumStd = nan(1,numel(pertEdges));
        timeSetMean = nan(1,numel(pertEdges));
        timeSetStd = nan(1,numel(pertEdges));
        
        % load evaluation results
        for i=1:numel(pertEdges)
            try
                pertEdges_i = pertEdges(i);
                load(aux_getEvResultName(evalpath,model,pertEdges_i,delta,1), 'evResult');
            
                % read enumeration results
                timeEnum_i = evResult.timeEnum;
                % normalize
                timeEnum_i = timeEnum_i ./ evResult.numNodes;
                % obtain statistic
                [timeEnumStd(i),timeEnumMean(i)] = std(timeEnum_i);
            
                % read our results
                timeSet_i = evResult.timeSet;
                % normalize
                timeSet_i = timeSet_i ./ evResult.numNodes;
                % obtain statistic
                [timeSetStd(i),timeSetMean(i)] = std(timeSet_i);
            catch ME
                % file was not saved properly.
            end
        end
        
        % plot results
        
        plot(pertEdges,timeEnumMean,'Color',colorEnum,'DisplayName','Graph enumeration');
        fill([pertEdges,fliplr(pertEdges)], ...
            [timeEnumMean+timeEnumStd,fliplr(timeEnumMean-timeEnumStd)], ...
            colorEnum,'LineStyle','none','FaceAlpha', facealpha,'HandleVisibility','off');
        
        plot(pertEdges,timeSetMean,'Color',colorSet,'DisplayName','Our approach');
        fill([pertEdges,fliplr(pertEdges)], ...
            [timeSetMean+timeSetStd,fliplr(timeSetMean-timeSetStd)], ...
            colorSet,'LineStyle','none','FaceAlpha', facealpha,'HandleVisibility','off');
        
        title('Verification Rate', strrep(model,'_','\_'))
        xlabel('Number of uncertain edges')
        ylabel('Verificatiom time [s] / number of nodes')
        legend('Location','northwest')
        xlim([min(pertEdges) max(pertEdges)])

    end

end

function aux_plotVerification(evalpath,models)
        
    % show time comparison
    deltas = [0.001];
    pertEdges = [0,0.001,0.005,0.01,0.05];

    % init figure
    figure;
    colorVerified = [     0.4660    0.6740    0.1880]; % green
    colorViolated = [    0.8500    0.3250    0.0980]; % red
    facealpha = 0.2;
    
    for m=1:numel(models)
        model = models{m};
        
        % show time comparison for model m

        subplot(1,numel(models),m); hold on;

        for delta=deltas
    
            % init plotting data
            verifiedSetMean = nan(size(pertEdges));
            violatedSetMean = nan(size(pertEdges));
            
            % load evaluation results
            for i=1:numel(pertEdges)
                pertEdges_i = pertEdges(i);
                load(aux_getEvResultName(evalpath,model,pertEdges_i,delta,0), 'evResult');
            
                % read our results
                res_i = evResult.resVerifiedSet;
                
                % obtain statistic
                verifiedSetMean(i) = mean(res_i);
            
                % read our results
                res_i = evResult.resViolated;
        
                % obtain statistic
                violatedSetMean(i) = mean(res_i);
            end
            
            % plot results
            
            plot(pertEdges,verifiedSetMean,'DisplayName',sprintf('Delta=%.3f',delta));
        end
        
        title('Verification Rate', strrep(model,'_','\_'))
        legend('Location','southwest')
        xlabel('Number of uncertain edges [%o]')
        ylabel('Verification Rate')
        ylim([0,1])
    end

end

function aux_plotVerifiedInstances(evalpath,models,delta,pertEdges)
        
    % show time comparison

    % init figure
    figure;
    
    for m=1:numel(models)
        model = models{m};
        
        % show time comparison for model m

        subplot(1,numel(models),m); hold on;

        for pertEdges_i=pertEdges

            NVpairs = {'-o', 'DisplayName',sprintf('Pert. Edges=%.1f %%',pertEdges_i*100)};
    
            try
                % load results
                load(aux_getEvResultName(evalpath,model,pertEdges_i,delta,0), 'evResult');
            catch 
                plot(nan, nan, NVpairs{:});
                continue
            end

            % read timeSet
            timeSet = evResult.timeSet;
            verifiedInstances = evResult.resVerifiedSet;
            
            % set unverified instances to nan
            timeSet(~verifiedInstances) = nan;

            % sort entries
            [timeSet, idx] = sort(timeSet);
            verifiedInstances = verifiedInstances(idx);


            % plot results
            plot([0,cumsum(timeSet)], ...
                [0,cumsum(verifiedInstances)/numel(verifiedInstances)], ...
                NVpairs{:});
        end
        
        title('Verified Instances', strrep(model,'_','\_'))
        legend('Location','southeast')
        xlabel('Cumulative Verification Time [s]')
        if m == 1
            ylabel('Verified Instances [%]')
        end
        ylim([0,1])
    end

end