import numpy as np
from gym import utils
from gym.envs.mujoco import mujoco_env


class HalfCheetahPosEnv(mujoco_env.MujocoEnv, utils.EzPickle):
    noise_scale = 10
    def __init__(self):
        mujoco_env.MujocoEnv.__init__(self, "half_cheetah.xml", 5)
        utils.EzPickle.__init__(self)

    def step(self, action):
        xposbefore = self.sim.data.qpos[0]
        self.do_simulation(action, self.frame_skip)
        xposafter = self.sim.data.qpos[0]
        ob = self._get_obs()
        reward_ctrl = -0.1 * np.square(action).sum()
        reward_run = (xposafter - xposbefore) / self.dt
        reward = reward_ctrl + reward_run
        done = False

        info = dict(
            reward_run = reward_run,
            reward_ctrl = reward_ctrl,
            x_position = xposafter
        )

        # add noise reward, linearly decrease the randomness
        if xposafter > 0 and xposafter < 20:
            reward += np.random.randn() * self.noise_scale * (1 - xposafter / 20.)
        return ob, reward, done, info

    def _get_obs(self):
        return np.concatenate(
            [
                [self.sim.data.qpos.flat[0] / 150.],
                self.sim.data.qpos.flat[1:],
                self.sim.data.qvel.flat,
            ]
        )

    def reset_model(self):
        qpos = self.init_qpos + self.np_random.uniform(
            low=-0.1, high=0.1, size=self.model.nq
        )
        qvel = self.init_qvel + self.np_random.randn(self.model.nv) * 0.1
        self.set_state(qpos, qvel)
        return self._get_obs()

    def viewer_setup(self):
        self.viewer.cam.distance = self.model.stat.extent * 0.5
