"""Parsing of text files into environment instances."""
from dataclasses import dataclass
from terrain_mass.island import Rectangle


@dataclass
class Map:
    initial_position: tuple[float, float]
    islands: list[Rectangle]
    target_position: tuple[float, float]


def parse(string: str, char_size: float) -> Map:
    """Parse a string into a map."""
    initial_position: tuple[float, float] | None = None
    target_position: tuple[float, float] | None = None
    islands = list()

    for i, line in enumerate(string.splitlines()):
        for j, char in enumerate(line):
            if char == " ":
                continue
            if char == "O":
                assert initial_position is None, "Map has more than one mass!"
                initial_position = (j*char_size, -i*char_size)
            elif char == "L":
                # Turn character into an island
                #
                #    a:(j,-i) ------- d:(j+1, -i)
                #      |                 |
                #      |                 |
                #    b:(j, -(i+1)) ---  c:(j+1, (-i+1))
                #
                #
                a = (j*char_size, -i*char_size)
                b = (j*char_size, -(i+1)*char_size)
                c = ((j+1)*char_size, -(i+1)*char_size)
                d = ((j+1)*char_size, -i*char_size)
                island = Rectangle(a, b, c, d)
                islands.append(island)
            elif char == "G":
                assert target_position is None, "Map has more than one target position!"
                target_position = (j*char_size, -i*char_size)
            else:
                raise ValueError(f"Map contains unsupported character '{char}'!")

    assert initial_position is not None, "Map string has no mass 'O'!"
    assert target_position is not None, "Map string has no target position 'G'!"

    m = Map(
        initial_position=initial_position,
        islands=islands,
        target_position=target_position,
    )
    return m
