"""Tasks in the environment."""
from pathlib import Path
from dataclasses import dataclass
from terrain_mass.environment import EnvironmentInstance
from terrain_mass.parsing import parse
import random


@dataclass
class Task:
    environment: EnvironmentInstance
    target_position: tuple[float, float]
    success_distance_to_target: float
    dt: float


EXAMPLE_MAP_DIR = Path(__file__).parent/"example_maps"


SUCCESS_DISTANCE_TO_TARGET = 0.2
SAND_DRAG_CONSTANT = 0.2
WATER_DRAG_CONSTANT = 8.0
ACTION_MIN = -1.0
ACTION_MAX = 1.0
DT = 0.1


def generate_map(seed: str) -> str:
    """Generate a random map string."""
    # Generate a top-to-bottom map
    pieces = list()
    pieces.append(
        "LL   \n"
        "LL   \n"
    )
    pieces.append(
        "L    \n"
        "L    \n"
    )
    pieces.append(
        "   LL\n"
        "   LL\n"
    )
    pieces.append(
        "    L\n"
        "    L\n"
    )
    pieces.append(
        "LL LL\n"
        "LL LL\n"
    )
    pieces.append(
        "  L  \n"
        "  L  \n"
    )
    pieces.append(
        " L L \n"
        " L L \n"
    )
    pieces.append(
        "L L L\n"
        "L L L\n"
    )
    pieces.append(
        "LLLLL\n"
        "LLLLL\n"
    )
    pieces.append(
        "     \n"
        "     \n"
    )
    pieces.append(
        " L   \n"
        "   L \n"
    )
    pieces.append(
        "   L \n"
        " L   \n"
    )

    start = (
        "  O  \n"
    )
    end = (
        "  G  \n"
    )

    # Generate map
    _random = random.Random(seed)
    length = _random.randint(2, 10)
    mapstr = start
    for _ in range(length):
        piece = _random.choice(pieces)
        mapstr += piece
    mapstr += end

    return mapstr


def get_example_task(seed: str) -> Task:
    _random = random.Random(seed)

    # Decide if generated map or handcrafted
    generated = _random.choice([True, False])
    if generated:
        mstr = generate_map(str(_random.random()))
    else:
        maps = list(EXAMPLE_MAP_DIR.glob("*.txt"))
        chosen_map = _random.choice(maps)
        with open(chosen_map, "rt") as fp:
            mstr = fp.read()

    # Parse map
    m = parse(mstr, char_size=SUCCESS_DISTANCE_TO_TARGET)
    environment = EnvironmentInstance(
        sand_drag_constant=SAND_DRAG_CONSTANT,
        water_drag_constant=WATER_DRAG_CONSTANT,
        action_min=ACTION_MIN,
        action_max=ACTION_MAX,
        initial_mass_position=m.initial_position,
        islands=m.islands,
    )
    task = Task(
        environment=environment,
        target_position=m.target_position,
        success_distance_to_target=SUCCESS_DISTANCE_TO_TARGET,
        dt=DT,
    )
    return task
