# Copyright (c) Meta Platforms, Inc. and affiliates.
# All rights reserved.

# This source code is licensed under the license found in the
# LICENSE file in the root directory of this source tree.

"""
A minimal training script for DiT using PyTorch DDP.
"""
import torch
# the first flag below was False when we tested this script but True makes A100 training a lot faster:
torch.backends.cuda.matmul.allow_tf32 = True
torch.backends.cudnn.allow_tf32 = True
import torch.distributed as dist
from torch.nn.parallel import DistributedDataParallel as DDP
from torch.utils.data import DataLoader
from torch.utils.data.distributed import DistributedSampler
from torchvision.datasets import ImageFolder
from torchvision import transforms
import numpy as np
from collections import OrderedDict
from PIL import Image
from copy import deepcopy
from glob import glob
from time import time
from download import find_model
import argparse
import logging
import os

from models import DiT_models
from diffusion import create_diffusion
from diffusers.models import AutoencoderKL

import h5py
import torch.utils.data as data
import torch.nn.functional as F
from torchvision.utils import save_image, make_grid
import math

from torch.utils.data import Dataset
import torchvision.transforms.functional as TF

# import open_clip
from transformers import CLIPModel, CLIPProcessor
from torchvision.transforms.functional import to_pil_image

import lpips




class PNGDataset(Dataset):
    def __init__(self, image_dir, map_dir, mask_dir):
        self.image_paths = sorted(glob(os.path.join(image_dir, "*.png")))
        self.map_paths = sorted(glob(os.path.join(map_dir, "*.png")))
        self.mask_paths = sorted(glob(os.path.join(mask_dir, "*.png")))

        assert len(self.image_paths) == len(self.map_paths) == len(self.mask_paths), \
            "Mismatch in number of images, maps, and masks."

    def __len__(self):
        return len(self.image_paths)

    def __getitem__(self, idx):
        # Load grayscale images
        image = Image.open(self.image_paths[idx]).convert("L")  # Grayscale
        cond_map = Image.open(self.map_paths[idx]).convert("L")
        cond_mask = Image.open(self.mask_paths[idx]).convert("L")

        # Convert to tensor (will be [0,1]) and unsqueeze to [1, H, W]
        image = TF.to_tensor(image)  # [1, H, W]
        cond_map = TF.to_tensor(cond_map)
        cond_mask = TF.to_tensor(cond_mask)

        # Normalize to [0, 1] if needed (not required since TF.to_tensor() already does this)
        return image, cond_map, cond_mask
    

class HDF5Dataset(data.Dataset):
    def __init__(self, hdf5_file):
        self.data = h5py.File(hdf5_file, 'r')
        self.images = self.data['images']
        self.maps = self.data['maps']
        self.mask = self.data['maps']

    def __len__(self):
        return len(self.images)

    def __getitem__(self, idx):
        image = torch.tensor(self.images[idx], dtype=torch.float32).unsqueeze(0)  # Grayscale
        map = torch.tensor(self.maps[idx], dtype=torch.float32).unsqueeze(0)  # Grayscale
        mask = torch.tensor(self.maps[idx], dtype=torch.float32).unsqueeze(0)  # Grayscale
            # Normalize the image and map to the [0, 1] range if they are in [0, 255]
        if image.max() > 1.0:  # Check if normalization is needed
            image = image / 255.0
        if map.max() > 1.0:
            map = map / 255.0
        if mask.max() > 1.0:
            mask = mask / 255.0
        return image, map, mask




#################################################################################
#                             Training Helper Functions                         #
#################################################################################

@torch.no_grad()
def update_ema(ema_model, model, decay=0.9999):
    """
    Step the EMA model towards the current model.
    """
    ema_params = OrderedDict(ema_model.named_parameters())
    model_params = OrderedDict(model.named_parameters())

    for name, param in model_params.items():
        # TODO: Consider applying only to params that require_grad to avoid small numerical changes of pos_embed
        ema_params[name].mul_(decay).add_(param.data, alpha=1 - decay)


def requires_grad(model, flag=True):
    """
    Set requires_grad flag for all parameters in a model.
    """
    for p in model.parameters():
        p.requires_grad = flag


def cleanup():
    """
    End DDP training.
    """
    dist.destroy_process_group()


def create_logger(logging_dir):
    """
    Create a logger that writes to a log file and stdout.
    """
    if dist.get_rank() == 0:  # real logger
        logging.basicConfig(
            level=logging.INFO,
            format='[\033[34m%(asctime)s\033[0m] %(message)s',
            datefmt='%Y-%m-%d %H:%M:%S',
            handlers=[logging.StreamHandler(), logging.FileHandler(f"{logging_dir}/log.txt")]
        )
        logger = logging.getLogger(__name__)
    else:  # dummy logger (does nothing)
        logger = logging.getLogger(__name__)
        logger.addHandler(logging.NullHandler())
    return logger


def center_crop_arr(pil_image, image_size):
    """
    Center cropping implementation from ADM.
    https://github.com/openai/guided-diffusion/blob/8fb3ad9197f16bbc40620447b2742e13458d2831/guided_diffusion/image_datasets.py#L126
    """
    while min(*pil_image.size) >= 2 * image_size:
        pil_image = pil_image.resize(
            tuple(x // 2 for x in pil_image.size), resample=Image.BOX
        )

    scale = image_size / min(*pil_image.size)
    pil_image = pil_image.resize(
        tuple(round(x * scale) for x in pil_image.size), resample=Image.BICUBIC
    )

    arr = np.array(pil_image)
    crop_y = (arr.shape[0] - image_size) // 2
    crop_x = (arr.shape[1] - image_size) // 2
    return Image.fromarray(arr[crop_y: crop_y + image_size, crop_x: crop_x + image_size])


#################################################################################
#                                  Training Loop                                #
#################################################################################

def main(args):
    """
    Trains a new DiT model.
    """
    assert torch.cuda.is_available(), "Training currently requires at least one GPU."

    # Setup DDP:
    dist.init_process_group("nccl")
    assert args.global_batch_size % dist.get_world_size() == 0, f"Batch size must be divisible by world size."
    rank = dist.get_rank()
    device = rank % torch.cuda.device_count()
    seed = args.global_seed * dist.get_world_size() + rank
    torch.manual_seed(seed)
    torch.cuda.set_device(device)
    print(f"Starting rank={rank}, seed={seed}, world_size={dist.get_world_size()}.")

    # Setup an experiment folder:
    if rank == 0:
        if args.resume_from_checkpoint is not None:
            print("Resumeeeeeeee fromm checkpointttt ##########dekwkndwkljcbjhdbcjkhcbmmmmmmmmmmmmmmmmmmxmxmxmxmxmxmxmmxmxmx")
            experiment_dir = args.resume_from_checkpoint.split("/checkpoints")[0]
            checkpoint_dir = f"{experiment_dir}/checkpoints"
            print("checkpoint dir ", checkpoint_dir)

        else:    
            os.makedirs(args.results_dir, exist_ok=True)  # Make results folder (holds all experiment subfolders)
            experiment_index = len(glob(f"{args.results_dir}/*"))
            model_string_name = args.model.replace("/", "-")  # e.g., DiT-XL/2 --> DiT-XL-2 (for naming folders)
            experiment_dir = f"{args.results_dir}/{experiment_index:03d}-{model_string_name}"  # Create an experiment folder
            image_dir = f"{experiment_dir}/reconstructed"  # Stores saved model images
            os.makedirs(image_dir, exist_ok=True)
            checkpoint_dir = f"{experiment_dir}/checkpoints"  # Stores saved model checkpoints
            os.makedirs(checkpoint_dir, exist_ok=True)
        logger = create_logger(experiment_dir)
        logger.info(f"Experiment directory created at {experiment_dir}")
    else:
        logger = create_logger(None)

    # Create model:
    assert args.image_size % 8 == 0, "Image size must be divisible by 8 (for the VAE encoder)."
    latent_size = args.image_size // 8
    model = DiT_models[args.model](
        input_size=latent_size,
        in_channels=4,        # To match x_latent channels
        cond_channels=1,      # Grayscale conditioning maps
    ).to(device)

    # Note that parameter initialization is done within the DiT constructor
    ema = deepcopy(model).to(device)  # Create an EMA of the model for use after training
    requires_grad(ema, False)
        # vae.encoder.conv_in = torch.nn.Conv2d(
    # in_channels=1,  # Change from 3 to 1
    # out_channels=vae.encoder.conv_in.out_channels,
    # kernel_size=3,
    # stride=1,
    # padding=1
    # )
    # vae = vae.to(device)


    # Setup optimizer (we used default Adam betas=(0.9, 0.999) and a constant learning rate of 1e-4 in our paper):
    opt = torch.optim.AdamW(model.parameters(), lr=1e-4, weight_decay=0)


    if args.resume_from_checkpoint:
        checkpoint = torch.load(args.resume_from_checkpoint, map_location='cpu')
        model.load_state_dict(checkpoint['model'], strict=False)
        ema.load_state_dict(checkpoint['ema'], strict=False)
        # opt.load_state_dict(checkpoint['opt'])
        if len(checkpoint["opt"]["param_groups"][0]["params"]) == len(list(model.parameters())):
            opt.load_state_dict(checkpoint["opt"])
            logger.info("Optimizer state restored.")
        else:
            logger.info("Param count changed → skipping optimizer state.")
        del checkpoint
        logger.info(f"Using checkpoint: {args.resume_from_checkpoint}")
        
        print("diffuser model loaded successfully")

    logger.info(f"args {str(args)}")

    model = DDP(model.to(device), device_ids=[rank])

    image_size = 256 #@param [256, 512]


    diffusion = create_diffusion(timestep_respacing="")  # default: 1000 steps, linear noise schedule
    vae = AutoencoderKL.from_pretrained(f"stabilityai/sd-vae-ft-{args.vae}").to(device)
    ema = ema.to(device)
    logger.info(f"DiT Parameters: {sum(p.numel() for p in model.parameters()):,}")

    # for clip encoder
    # clip_model, _, preprocess_clip = open_clip.create_model_and_transforms('ViT-B-16', pretrained='openai')
    # clip_model = clip_model.eval().to(device)
    # print(clip_model)
    clip_model = CLIPModel.from_pretrained("openai/clip-vit-large-patch14").to(device)
    clip_processor = CLIPProcessor.from_pretrained("openai/clip-vit-large-patch14")
    # print(clip_model.config.vision_config.hidden_size)
    # Should be 1024 for ViT-L/14
    lpips_model = lpips.LPIPS(net='vgg').to(device)


    # Setup data:
    transform = transforms.Compose([
        transforms.Lambda(lambda pil_image: center_crop_arr(pil_image, args.image_size)),
        transforms.RandomHorizontalFlip(),
        transforms.ToTensor(),
        transforms.Normalize(mean=[0.5, 0.5, 0.5], std=[0.5, 0.5, 0.5], inplace=True)
    ])

    dataset = PNGDataset(
        image_dir='',
        map_dir='',
        mask_dir=''
    )   

    sampler = DistributedSampler(
        dataset,
        num_replicas=dist.get_world_size(),
        rank=rank,
        shuffle=True,
        seed=args.global_seed
    )
    loader = DataLoader(
        dataset,
        batch_size=int(args.global_batch_size // dist.get_world_size()),
        shuffle=False,
        sampler=sampler,
        num_workers=args.num_workers,
        pin_memory=True,
        drop_last=True
    )
    logger.info(f"Dataset contains {len(dataset):,} images ({args.data_path})")

    # Prepare models for training:
    # update_ema(ema, model.module, decay=0)  # Ensure EMA is initialized with synced weights
    # model.train()  # important! This enables embedding dropout for classifier-free guidance
    # ema.eval()  # EMA model should always be in eval mode

    # Variables for monitoring/logging purposes:
    train_steps = 0
    log_steps = 0
    running_loss = 0
    start_time = time()

    first_epoch = 0

    print(len(loader))
    num_update_steps_per_epoch = len(loader)
    print("num_update_steps_per_epoch", num_update_steps_per_epoch)
    # Prepare models for training:
    if args.resume_from_checkpoint:

        train_steps = int((args.resume_from_checkpoint.split("/")[-1]).split(".")[0])

        print("train step ", train_steps)

        first_epoch = math.ceil(train_steps / num_update_steps_per_epoch)

        print("first epoch: ", first_epoch)
    else:
        update_ema(ema, model.module, decay=0)  # Ensure EMA is initialized with synced weights

    model.train()  # important! This enables embedding dropout for classifier-free guidance
    ema.eval()  # EMA model should always be in eval mode



    logger.info(f"Training for {args.epochs} epochs...")
    for epoch in range(first_epoch, args.epochs):
        sampler.set_epoch(epoch)
        logger.info(f"Beginning epoch {epoch}...")
        for x, cond_map, cond_mask in loader:
            x, cond_map, cond_mask = x.to(device), cond_map.to(device), cond_mask.to(device)

            # DEBUG: Check raw grayscale input and noisy conditioning map

            # Normalize x to [-1, 1] for the VAE
            # x = (x * 2) - 1  # Scale from [0, 1] --> [-1, 1]

            # Converting the grayscale images to pseudo-RGB for VAE
            x_rgb = x.repeat(1, 3, 1, 1)  # Shape: [N, 3, H, W]
            map_rgb = cond_map.repeat(1, 3, 1, 1)  # Shape: [N, 3, H, W]
            mask_rgb = cond_mask.repeat(1, 3, 1, 1)  # Shape: [N, 3, H, W]

            # Resize and normalize cond_map for CLIP


            # target_size = 224  # or 256 or 32 etc.
            cond_map = [to_pil_image(img.squeeze(0)) for img in cond_map]
            inputs_map = clip_processor(images=cond_map, return_tensors="pt").to(device)

            cond_mask = [to_pil_image(img.squeeze(0)) for img in cond_mask]
            inputs_mask = clip_processor(images=cond_mask, return_tensors="pt").to(device)




        
 

            # Encode images using VAE (removed 0.18215 scaling for debugging)s
            with torch.no_grad():
                x_latent = vae.encode(x_rgb).latent_dist.sample() * 0.18215
                # map_latent = vae.encode(map_rgb).latent_dist.sample() * 0.18215
                # cond_emb = clip_model(cond_img_for_clip).pooler_output  # [batch, 1024] or [batch, 768], depending on model
                cond_map = clip_model.get_image_features(**inputs_map)
                cond_mask = clip_model.get_image_features(**inputs_mask)


                # vision_outputs_map = clip_model.vision_model(**inputs_map)
                # clip_tokens_map = vision_outputs_map.last_hidden_state  # Shape: [B, num_patches+1, D]
                # cond_map = clip_tokens_map[:, 1:, :]  # Remove CLS token if needed → [B, num_patches, D]

                # vision_outputs_mask = clip_model.vision_model(**inputs_mask)
                # clip_tokens_mask = vision_outputs_mask.last_hidden_state  # Shape: [B, num_patches+1, D]
                # cond_mask = clip_tokens_mask[:, 1:, :]  # Remove CLS token if needed → [B, num_patches, D]
                # print(cond_mask.shape)
                # print(cond_mask.shape)

                # clip_outputs = clip_model.vision_model(**inputs)
                # clip_tokens = clip_outputs.last_hidden_state  # [batch, seq_len, hidden]
                # clip_tokens = clip_tokens[:, 1:, :]  # [batch, seq_len-1, hidden]
                # cond_emb = clip_model.encode_image(cond_img_for_clip).float()
                # print(cond_emb.shape)  # Should be [batch, 1024]
                
                # mask_latent = vae.encode(mask_rgb).latent_dist.sample() * 0.18215

            # print("get_image_features shape:", features.shape)
            # print("Model name:", clip_model.config._name_or_path)

            

            # Generate random timesteps
            t = torch.randint(0, diffusion.num_timesteps, (x_latent.shape[0],), device=device)

            # Prepare model kwargs
            # model_kwargs = {
            #     'cond_emb': cond_emb
            #     #  'clip_tokens': clip_tokens
            #     }


            # Prepare model kwargs
            model_kwargs = {
                'cond_map': cond_map,
                 'cond_mask': cond_mask
                }
            
            # for unconditional
            # model_kwargs = {
            #     'cond_map': None,
            #     'cond_mask': None
            #     }

            # Compute loss using diffusion.training_losses
            loss_dict = diffusion.training_losses(model, x_latent, t, model_kwargs)
            loss = loss_dict["loss"].mean()
            x_pred = loss_dict["x_pred"]
            # with torch.no_grad():
            #     x_recon = vae.decode(x_pred / 0.18215).sample
            #     x_gt = vae.decode(x_latent / 0.18215).sample

            # # If needed, make sure both are 3-channel
            # if x_recon.shape[1] != 3:
            #     x_recon = x_recon.repeat(1, 3, 1, 1)
            #     x_gt = x_gt.repeat(1, 3, 1, 1)

            # # 1. L1 Reconstruction Loss
            # L_rec = F.l1_loss(x_recon, x_gt)

            # # 2. LPIPS Loss
            # L_LPIPS = lpips_model(x_recon, x_gt).mean()

            # # 3. CLIP Loss
            # gen_imgs = [to_pil_image(x_recon[i].cpu()) for i in range(x_recon.shape[0])]
            # gen_inputs = clip_processor(images=gen_imgs, return_tensors="pt").to(device)
            # clip_emb_pred = clip_model.get_image_features(**gen_inputs)
            # clip_emb_pred = F.normalize(clip_emb_pred, dim=-1)
            # clip_emb_src = F.normalize(cond_emb, dim=-1)
            # L_CLIP = 1 - (clip_emb_pred * clip_emb_src).sum(-1).mean()

            # # Combine losses
            # lambda_rec = 1.0
            # lambda_lpips = 1.0
            # lambda_clip = 1.0
            # loss = lambda_rec * L_rec + lambda_lpips * L_LPIPS + lambda_clip * L_CLIP

            # with torch.no_grad():
            #     decoded_pred = vae.decode(x_pred / 0.18215).sample
            #     decoded_real = vae.decode(x_latent / 0.18215).sample

            # if cond_mask.shape[1] == 1 and decoded_pred.shape[1] == 3:
            #     mask_rgb = cond_mask.repeat(1, 3, 1, 1)
            # else:
            #     mask_rgb = cond_mask
            # if cond_map.shape[1] == 1 and decoded_pred.shape[1] == 3:
            #     map_rgb = cond_map.repeat(1, 3, 1, 1)
            # else:
            #     map_rgb = cond_map


            # if mask_rgb.sum() > 0:
            #     mask_loss = F.l1_loss(decoded_pred * mask_rgb, decoded_real * mask_rgb)
            # else:
            #     mask_loss = 0.0

            # noise_loss = F.l1_loss((decoded_pred - decoded_real) * map_rgb, torch.zeros_like(decoded_pred))
            # loss = loss + 0.5 * mask_loss + 1.0 * noise_loss

            # Optimize
            opt.zero_grad()
            loss.backward()
            opt.step()
            # Update EMA model
            update_ema(ema, model.module)

            # Log loss values:
            running_loss += loss.item()
            log_steps += 1
            train_steps += 1
            
            if train_steps % args.log_every == 0:
                # Measure training speed:
                torch.cuda.synchronize()
                end_time = time()
                steps_per_sec = log_steps / (end_time - start_time)
                # Reduce loss history over all processes:
                avg_loss = torch.tensor(running_loss / log_steps, device=device)
                dist.all_reduce(avg_loss, op=dist.ReduceOp.SUM)
                avg_loss = avg_loss.item() / dist.get_world_size()
                logger.info(f"(step={train_steps:07d}) Train Loss: {avg_loss:.4f}, Train Steps/Sec: {steps_per_sec:.2f}")
                # Reset monitoring variables:
                running_loss = 0
                log_steps = 0
                start_time = time()
            # Save DiT checkpoint:
            if train_steps % args.ckpt_every == 0 and train_steps > 0:
                if rank == 0:
                    checkpoint = {
                        "model": model.module.state_dict(),
                        "ema": ema.state_dict(),
                        "opt": opt.state_dict(),
                        "args": args
                    }
                    checkpoint_path = f"{checkpoint_dir}/{train_steps:07d}.pt"
                    torch.save(checkpoint, checkpoint_path)
                    logger.info(f"Saved checkpoint to {checkpoint_path}")

                    with torch.no_grad():
                        x_recon = vae.decode(x_pred / 0.18215).sample
                        save_image(x_recon, f"{checkpoint_dir}/new_sample{train_steps:07d}.png", nrow=4, normalize=True)
                        save_image(x, f"{checkpoint_dir}/real_sample{train_steps:07d}.png", nrow=4, normalize=True, value_range=(-1, 1))
                        save_image(cond_map, f"{checkpoint_dir}/cond_sample{train_steps:07d}.png", nrow=4, normalize=True, value_range=(-1, 1))
                        save_image(cond_mask, f"{checkpoint_dir}/cond_sample_mask{train_steps:07d}.png", nrow=4, normalize=True, value_range=(-1, 1))


                dist.barrier()


    model.eval()  # important! This disables randomized embedding dropout
    # do any sampling/FID calculation/etc. with ema (or model) in eval mode ...

    logger.info("Done!")
    cleanup()


if __name__ == "__main__":
    # Default args here will train DiT-XL/2 with the hyperparameters we used in our paper (except training iters).
    parser = argparse.ArgumentParser()
    parser.add_argument("--data-path", type=str)
    parser.add_argument("--results-dir", type=str, default="results")
    parser.add_argument("--model", type=str, choices=list(DiT_models.keys()), default="DiT-XL/2")
    parser.add_argument("--image-size", type=int, choices=[256, 512], default=256)
    # parser.add_argument("--num-classes", type=int, default=1000)
    parser.add_argument("--epochs", type=int, default=1400)
    parser.add_argument("--global-batch-size", type=int, default=256)
    parser.add_argument("--global-seed", type=int, default=0)
    parser.add_argument("--vae", type=str, choices=["ema", "mse"], default="ema")  # Choice doesn't affect training
    parser.add_argument("--num-workers", type=int, default=4)
    parser.add_argument("--log-every", type=int, default=100)
    parser.add_argument("--ckpt-every", type=int, default=50_000)
    parser.add_argument("--resume_from_checkpoint", type=str, default=None)

    args = parser.parse_args()
    main(args)