import sys
from functools import reduce

import numpy as np
from torch.nn.modules.module import _addindent
import torch
from .config import F_MIN, Q


def cycle(iterable):
    while True:
        for item in iterable:
            yield item


def summary(model, file=sys.stdout):
    def repr(model):
        # We treat the extra repr like the sub-module, one item per line
        extra_lines = []
        extra_repr = model.extra_repr()
        # empty string will be split into list ['']
        if extra_repr:
            extra_lines = extra_repr.split('\n')
        child_lines = []
        total_params = 0
        for key, module in model._modules.items():
            mod_str, num_params = repr(module)
            mod_str = _addindent(mod_str, 2)
            child_lines.append('(' + key + '): ' + mod_str)
            total_params += num_params
        lines = extra_lines + child_lines

        for name, p in model._parameters.items():
            if hasattr(p, 'shape'):
                total_params += reduce(lambda x, y: x * y, p.shape)

        main_str = model._get_name() + '('
        if lines:
            # simple one-liner info, which most builtin Modules will use
            if len(extra_lines) == 1 and not child_lines:
                main_str += extra_lines[0]
            else:
                main_str += '\n  ' + '\n  '.join(lines) + '\n'

        main_str += ')'
        if file is sys.stdout:
            main_str += ', \033[92m{:,}\033[0m params'.format(total_params)
        else:
            main_str += ', {:,} params'.format(total_params)
        return main_str, total_params

    string, count = repr(model)
    if file is not None:
        if isinstance(file, str):
            file = open(file, 'w')
        print(string, file=file)
        file.flush()

    return count


def pitchlabel2freqs(pitch, pitch_threshold=0.5):
    freqs_index = torch.argmax(pitch, dim=-1)
    pitch_v = (pitch > pitch_threshold)
    pitch_v = torch.sum(pitch_v, dim=-1)
    pitch_v[pitch_v != 0] = 1
    freqs = F_MIN * (2 ** (freqs_index / Q)) * pitch_v
    return freqs.cpu().numpy()


def freqs2cents(freqs):
    cents = np.zeros_like(freqs)
    if freqs.ndim == 2:
        b, seq_l = freqs.shape
        for i in range(b):
            for j in range(seq_l):
                if freqs[i][j] != 0:
                    cents[i][j] = 1200.0 * np.log2(np.abs(freqs[i][j])/10)
    else:
        seq_l = freqs.shape[0]
        for j in range(seq_l):
            if freqs[j] != 0:
                cents[j] = 1200.0 * np.log2(np.abs(freqs[j]) / 10)
    return cents


def to_local_average_cents(salience, center=None, th=0.5):
    """
    find the weighted average cents near the argmax bin
    """

    if not hasattr(to_local_average_cents, 'cents_mapping'):
        # the bin number-to-cents mapping
        to_local_average_cents.cents_mapping = (
                np.linspace(0, 7180, 360) + 1997.3794084376191)

    if salience.ndim == 1:
        if center is None:
            center = int(np.argmax(salience))
        start = max(0, center - 4)
        end = min(len(salience), center + 5)
        salience = salience[start:end]
        product_sum = np.sum(
            salience * to_local_average_cents.cents_mapping[start:end])
        weight_sum = np.sum(salience)
        return product_sum / weight_sum if np.max(salience) > th else 0
    if salience.ndim == 2:
        return np.array([to_local_average_cents(salience[i, :], center, th) for i in
                         range(salience.shape[0])])
    if salience.ndim == 3:
        return np.array(
            [[to_local_average_cents(salience[i, j, :], center, th) for j in range(salience.shape[1])] for i in
             range(salience.shape[0])]
        )

    raise Exception("label should be either 1d or 2d or 3d ndarray")
