import torch
import torch.nn as nn
import torch.nn.functional as F


class ActFunction(nn.Module):
    def __init__(self, act='relu'):
        super(ActFunction, self).__init__()
        self.act = act

    def forward(self, x):
        if self.act == "relu":
            return nn.ReLU()(x)
        elif self.act == "leaky_relu":
            return nn.LeakyReLU()(x)
        elif self.act == "swish":
            return x * nn.Sigmoid()(x)
        elif self.act == 'prelu':
            return nn.PReLU()(x)
        else:
            raise Exception("Incorrect activation!")


class ConvBlock(nn.Module):
    def __init__(self, in_channels, out_channels, act='relu', momentum=0.01, bias=False):
        super(ConvBlock, self).__init__()

        self.act = ActFunction(act)
        self.conv = nn.Sequential(
            nn.Conv2d(in_channels=in_channels,
                      out_channels=out_channels,
                      kernel_size=(3, 3),
                      stride=(1, 1),
                      padding=(1, 1),
                      bias=bias),
            nn.BatchNorm2d(out_channels, momentum=momentum),
            self.act,

            nn.Conv2d(in_channels=out_channels,
                      out_channels=out_channels,
                      kernel_size=(3, 3),
                      stride=(1, 1),
                      padding=(1, 1),
                      bias=bias),
            nn.BatchNorm2d(out_channels, momentum=momentum),
            self.act
        )

    def forward(self, x):
        return self.conv(x)


class EncoderBlock(nn.Module):
    def __init__(self, in_channels, out_channels, act='relu', momentum=0.01):
        super(EncoderBlock, self).__init__()
        self.conv = ConvBlock(in_channels, out_channels, act, momentum)
        self.pool = nn.AvgPool2d(kernel_size=(1, 2))

    def forward(self, x):
        x = self.conv(x)
        x_pool = self.pool(x)
        return x, x_pool


class DecoderBlock(nn.Module):
    def __init__(self, in_channels, out_channels, act='relu', momentum=0.01):
        super(DecoderBlock, self).__init__()
        self.act = ActFunction(act)
        self.conv1 = nn.Sequential(
            nn.ConvTranspose2d(in_channels=in_channels,
                               out_channels=out_channels,
                               kernel_size=(3, 3),
                               stride=(1, 2),
                               padding=(1, 1),
                               output_padding=(0, 1),
                               bias=False),
            nn.BatchNorm2d(out_channels, momentum=momentum),
            self.act
        )
        self.conv2 = ConvBlock(out_channels * 2, out_channels, act, momentum)

    def forward(self, x, concat_tensor):
        x = self.conv1(x)
        x = torch.cat((x, concat_tensor), dim=1)
        x = self.conv2(x)
        return x


class Encoder(nn.Module):
    def __init__(self, in_channels, in_size, blocks_num=5, out_channels=16, act='relu', momentum=0.01):
        super(Encoder, self).__init__()
        self.blocks_num = blocks_num
        self.bn = nn.BatchNorm2d(in_channels, momentum=momentum)
        self.layers = nn.ModuleList()
        self.latent_channels = []
        for i in range(self.blocks_num):
            self.layers.append(EncoderBlock(in_channels, out_channels, act=act, momentum=momentum))
            self.latent_channels.append([out_channels, in_size])
            in_channels = out_channels
            out_channels *= 2
            in_size //= 2
        self.conv = ConvBlock(in_channels, out_channels, act, momentum)
        self.out_size = in_size
        self.out_channel = out_channels

    def forward(self, x):
        concat_tensors = []
        x = self.bn(x)
        for i in range(self.blocks_num):
            _, x = self.layers[i](x)
            concat_tensors.append(_)
        x = self.conv(x)
        return x, concat_tensors


class Decoder(nn.Module):
    def __init__(self, in_channels, blocks_num=5, output_channel=1):
        super(Decoder, self).__init__()
        self.layers = nn.ModuleList()
        self.blocks_num = blocks_num
        for i in range(self.blocks_num):
            out_channels = in_channels // 2
            self.layers.append(DecoderBlock(in_channels, out_channels))
            in_channels = out_channels
        self.conv = nn.Conv2d(in_channels=in_channels,
                              out_channels=output_channel,
                              kernel_size=(1, 1),
                              stride=(1, 1),
                              bias=True)

    def forward(self, x, concat_tensors):
        for i in range(self.blocks_num):
            x = self.layers[i](x, concat_tensors[-1-i])
        x = self.conv(x)
        return x


class Unet(nn.Module):
    def __init__(self, in_channels, in_size, hop_length, encoder_out_channels=16, decoder_out_channel=1, blocks_num=5,
                 act='relu', momentum=0.01):
        super(Unet, self).__init__()
        self.encoder = Encoder(in_channels, in_size, blocks_num, encoder_out_channels, act, momentum)
        self.decoder = Decoder(self.encoder.out_channel, blocks_num, decoder_out_channel)

    def forward(self, spec_m):
        # spec_m, cos_m, sin_m = self.to_spec(audio_m)
        x = spec_m[..., 0: spec_m.shape[-1] - 1]
        x, concat_tensors = self.encoder(x)
        x = self.decoder(x, concat_tensors)
        out_spec = F.pad(x, pad=(0, 1))
        mask_cos = torch.ones_like(out_spec).to(out_spec.device)
        mask_sin = torch.zeros_like(out_spec).to(out_spec.device)
        return out_spec, mask_cos, mask_sin
