import networkx as nx
import graph_tool as gt


def get_prop_type(value, key=None):
    """
    Performs typing and value conversion for the graph_tool PropertyMap class.
    If a key is provided, it also ensures the key is in a format that can be
    used with the PropertyMap. Returns a tuple, (type name, value, key)
    """

    # Deal with the value
    if isinstance(value, bool):
        tname = "bool"

    elif isinstance(value, int):
        tname = "float"
        value = float(value)

    elif isinstance(value, float):
        tname = "float"

    elif isinstance(value, str):
        tname = "string"
        value = value.encode("ascii", errors="replace")

    elif isinstance(value, dict):
        tname = "object"

    else:
        tname = "string"
        value = str(value)

    return tname, value, key


def nx2gt(nxG):
    """
    Converts a networkx graph to a graph-tool graph.
    """
    # Phase 0: Create a directed or undirected graph-tool Graph
    gtG = gt.Graph(directed=nxG.is_directed())

    # Add the Graph properties as "internal properties"
    for key, value in nxG.graph.items():
        # Convert the value and key into a type for graph-tool
        tname, value, key = get_prop_type(value, key)

        prop = gtG.new_graph_property(tname)  # Create the PropertyMap
        gtG.graph_properties[key] = prop  # Set the PropertyMap
        gtG.graph_properties[key] = value  # Set the actual value

    # Phase 1: Add the vertex and edge property maps
    # Go through all nodes and edges and add seen properties
    # Add the node properties first
    nprops = set()  # cache keys to only add properties once
    for node, data in nxG.nodes(data=True):
        # Go through all the properties if not seen and add them.
        for key, val in data.items():
            if key in nprops:
                continue  # Skip properties already added

            # Convert the value and key into a type for graph-tool
            tname, _, key = get_prop_type(val, key)

            prop = gtG.new_vertex_property(tname)  # Create the PropertyMap
            gtG.vertex_properties[key] = prop  # Set the PropertyMap

            # Add the key to the already seen properties
            nprops.add(key)

    # Also add the node id: in NetworkX a node can be any hashable type, but
    # in graph-tool node are defined as indices. So we capture any strings
    # in a special PropertyMap called 'id' -- modify as needed!
    gtG.vertex_properties["id"] = gtG.new_vertex_property("string")

    # Add the edge properties second
    eprops = set()  # cache keys to only add properties once
    for src, dst, data in nxG.edges(data=True):
        # Go through all the edge properties if not seen and add them.
        for key, val in data.items():
            if key in eprops:
                continue  # Skip properties already added

            # Convert the value and key into a type for graph-tool
            tname, _, key = get_prop_type(val, key)

            prop = gtG.new_edge_property(tname)  # Create the PropertyMap
            gtG.edge_properties[key] = prop  # Set the PropertyMap

            # Add the key to the already seen properties
            eprops.add(key)

    # Phase 2: Actually add all the nodes and vertices with their properties
    # Add the nodes
    vertices = {}  # vertex mapping for tracking edges later
    for node, data in nxG.nodes(data=True):
        # Create the vertex and annotate for our edges later
        v = gtG.add_vertex()
        vertices[node] = v

        # Set the vertex properties, not forgetting the id property
        data["id"] = str(node)
        for key, value in data.items():
            gtG.vp[key][v] = value  # vp is short for vertex_properties

    # Add the edges
    for src, dst, data in nxG.edges(data=True):
        # Look up the vertex structs from our vertices mapping and add edge.
        e = gtG.add_edge(vertices[src], vertices[dst])

        # Add the edge properties
        for key, value in data.items():
            gtG.ep[key][e] = value  # ep is short for edge_properties

    # Done, finally!
    return gtG


def gt2nx(gtG, multiedges=False):
    """
    Converts a graph-tool graph to a networkx graph.
    """

    # Create a directed or undirected networkx graph
    if multiedges:
        nxG = nx.MultiDiGraph() if gtG.is_directed() else nx.MultiGraph()
    else:
        nxG = nx.DiGraph() if gtG.is_directed() else nx.Graph()

    # Add the Graph properties
    for key, value in gtG.graph_properties.items():
        nxG.graph[key] = value

    # Add the nodes and their properties
    for i, v in enumerate(gtG.vertices()):
        nxG.add_node(int(v), **{k: gtG.vp[k][v] for k in gtG.vp})

    # Add the edges and their properties
    for e in gtG.edges():
        src, dst = int(e.source()), int(e.target())
        nxG.add_edge(src, dst, **{k: gtG.ep[k][e] for k in gtG.ep})
        edge_type = gtG.ep["type"][e]
        if multiedges:
            multiedge_idx = 0
            while (src, dst, multiedge_idx) in nxG.edges:
                if not isinstance(nxG.edges[src, dst, multiedge_idx]["type"], tuple):
                    # print(f"Edge type {nxG.edges[src, dst, multiedge_idx]['type']} is not a tuple")
                    break
                multiedge_idx += 1
            if isinstance(edge_type, tuple):
                continue
        # edge types are stored as strings in graph-tool, convert them to tuples
        edge_type = tuple(
            edge_type.replace("'", "").replace("(", "").replace(")", "").split(", ")
        )
        if multiedges:
            nxG.edges[src, dst, multiedge_idx]["type"] = edge_type
        else:
            nxG.edges[src, dst]["type"] = tuple(edge_type)

    return nxG
