import argparse
from portfolio_environment import environment
import numpy as np
from itertools import count

import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.optim as optim
from torch.distributions import Categorical

target = 1.5

parser = argparse.ArgumentParser(description='PyTorch REINFORCE example')
parser.add_argument('--gamma', type=float, default=0.99, metavar='G',
                    help='discount factor (default: 0.99)')
parser.add_argument('--seed', type=int, default=543, metavar='N',
                    help='random seed (default: 543)')
parser.add_argument('--render', action='store_true',
                    help='render the environment')
parser.add_argument('--log-interval', type=int, default=10, metavar='N',
                    help='interval between training status logs (default: 10)')
args = parser.parse_args()


env = environment()
torch.manual_seed(args.seed)


class Policy(nn.Module):
    def __init__(self):
        super(Policy, self).__init__()
        self.affine1 = nn.Linear(env.W+2, env.W+2)
        self.affine2 = nn.Linear(env.W+2, 2)

        self.saved_log_probs = []
        self.rewards = []

    def forward(self, x):
        x = self.affine1(x)
        x = F.relu(x)
        action_scores = self.affine2(x)
        return F.softmax(action_scores, dim=1)


def select_action(state, policy):
    state = torch.from_numpy(state).float().unsqueeze(0)
    probs = policy(state)
    m = Categorical(probs)
    action = m.sample()
    policy.saved_log_probs.append(m.log_prob(action))
    return action.item()


def finish_episode(policy, optimizer, eps):
    R = 0
    policy_loss = []
    returns = []
    for r in policy.rewards[::-1]:
        R = r + args.gamma * R
        returns.insert(0, R)
    returns = torch.tensor(returns)
    returns = returns - returns**2/(2*target)
    returns = (returns - returns.mean()) / (returns.std() + eps)
    for log_prob, R in zip(policy.saved_log_probs, returns):
        policy_loss.append(-log_prob * R)
    optimizer.zero_grad()
    policy_loss = torch.cat(policy_loss).sum()
    policy_loss.backward()
    optimizer.step()
    del policy.rewards[:]
    del policy.saved_log_probs[:]


def main():
    ep_reward_list_train = []
    ep_reward_list_test = []

    for tr in range(1000):

        running_ep_reward = []

        policy = Policy()
        optimizer = optim.Adam(policy.parameters(), lr=1e-2)
        eps = np.finfo(np.float32).eps.item()

        for i_episode in range(500):
            state, ep_reward = env.reset(), 0
            for t in range(1, 50):  # Don't infinite loop while learning
                action = select_action(state, policy)
                state, reward = env.step(action)
                policy.rewards.append(reward)
                ep_reward += reward

            running_ep_reward.append(ep_reward)
            finish_episode(policy, optimizer, eps)

            # log results
            if i_episode % args.log_interval == 0:
                print('Episode {}\tLast reward: {:.2f}'.format(
                    i_episode, ep_reward))

        ep_reward_list_train.append(running_ep_reward)
        np.savetxt('results/equm_reinforce_train_%2f.csv' % target,
                   ep_reward_list_train)

        ep_reward_list_test_temp = []

        for i_episode in range(100):

            state = env.reset()
            ep_reward = 0

            for t in range(1, 50):

                # select action from policy
                action = select_action(state, policy)

                # take the action
                state, reward = env.step(action)
                policy.rewards.append(reward)
                ep_reward += reward

            ep_reward_list_test_temp.append(ep_reward)

        ep_reward_list_test.append(ep_reward_list_test_temp)
        np.savetxt('results/equm_reinforce_test_%2f.csv' %
                   target, ep_reward_list_test)


if __name__ == '__main__':
    main()
