from typing import Optional
from qagent_omni.controller.registry.views import ActionModel
from qagent_omni.controller.views import ActionResult


class AutoExecSystemPrompt:
    """
    Generates the system prompt for the Auto Execution Agent's LLM.
    Uses the original requested keys but with improved formatting and examples.
    """

    def __init__(self, available_actions_description: str):
        """
        Initializes the system prompt generator.

        Args:
            available_actions_description: A string describing the available actions,
                                           their parameters, and constraints, including
                                           the action to finish the task (e.g., 'finish_task').
        """
        self.available_actions_description = available_actions_description

    def get_os_specific_hints(self) -> str:
        """
        Get OS specifics Hints
        :return:
        """
        return ""

    def get_system_prompt(self) -> str:
        """
        Constructs the detailed system prompt with the original JSON keys,
        specific format instructions, and an example.
        """
        os_specific_hints = self.get_os_specific_hints()
        if os_specific_hints:
            os_specific_hints = "**OS Specific Hints:**\n" + os_specific_hints

        # Define the exact JSON structure expected using the original keys
        output_format_structure = """
{
  "evaluation_prev_goal": "Success|Failed|Unknown - Evaluate if the previous action visually achieved its intended goal. Base this ONLY on the screen image. Ignore the execution result status provided in the input.",
  "import_contents": "Output important contents closely related to user\'s instruction on the current page. If there is, please output the contents. If not, please output empty string ''.",
  "think": "Provide a step-by-step thinking process. Analyze the current screen, relate it to the overall task and the visual outcome of the previous step ('evaluation_prev_goal'). Decide the next best *single* action. Explain your reasoning clearly, including why you chose the specific action and target (index or coordinates). If 'evaluation_prev_goal' was 'Failed', reflect on why and how the next action addresses it.",
  "next_goal": "Briefly describe the specific, immediate goal of the *next action* you are proposing in the 'action' field.",
  "action": {
    "action_name": { /* dictionary of parameters for the action */ }
  }
}
"""
        # Define a concrete example of the JSON output using the original keys
        output_example = """
{
  "evaluation_prev_goal": "Success - The password was entered into the password field (index 7).",
  "import_contents": "Username field shows 'test', password field shows has been filled. Login button (index 12) is visible and enabled.",
  "think": "Username and password have been successfully entered. The overall task is to log in. The final step for login is to click the 'Login' button, which corresponds to index 12 on the screen. It appears clickable. I will use the 'click' action with this index.",
  "next_goal": "Click the login button to attempt login.",
  "action": {
    "click": {
      "position": 12
    }
  }
}
"""

        system_prompt = f"""
You are an expert GUI automation agent. Your goal is to complete tasks specified by the user by interacting with a PC or mobile device GUI based on screenshots. You will be given the overall task description initially. In subsequent turns, you will receive the current screenshot, the result of the previous action's *execution*, and the current step number.

The screenshot is highlighted of the current screen. UI elements like icons and text detected by a vision model are highlighted with semi-transparent colored boxes. Each box has an index number in its top-left corner.

**Output Format:**
You MUST respond ONLY with a single valid JSON object in the following exact format. Do NOT include any text outside this JSON structure.

{output_format_structure}

**Available Actions:**
{self.available_actions_description}

**Action Position Selection:**
When specifying the target for an action (like 'click', 'input_text', etc.), use ONE of the following methods based on the `position` parameter:

*   **Using Highlight Index (`position`: <int>):**
    *   Prefer this method if the target UI element *directly and accurately* corresponds to one of the numbered highlighted regions. 
    
*   **Using Relative Coordinates within a Highlighted Box (`position`: [<int>, <float>, <float>]):**
    *   Use this method when the target element *is* within a highlighted box, but:
        *   The default highlighted area is too large or encompasses multiple elements.
        *   You need to precisely target a specific part *within* the highlighted box (e.g., a small icon inside a large button's highlight, or accurately hit a text field cursor).
        *   The float values (0.0 to 1.0) represent the relative x and y coordinates *within* the highlighted box (e.g., 0.5, 0.5 for the center, 0.0, 0.0 for top-left, 1.0, 1.0 for bottom-right).

*   **Using Center Coordinates (`position`: [<int>, <int>]):**
    *   Note that the x,y coordinates you output internally are strictly scaled from 0-1000, so 1000 represents the maximum size of the image. Please do not output coordinate points greater than 1000.
    *   Use this method as a fallback if:
        *   The target element does **not** correspond well to any highlighted region.
        *   The target element corresponds to a highlight, but the **index number is unclear or cut off**.
        *   The target element corresponds to a highlight, but the highlighted area is **inaccurate or too large**.

**General Instructions:**
*   Analyze the screenshot carefully. Use the highlighted regions and indices when possible and accurate.
*   Break down the user's task into smaller, manageable steps.
*   Think step-by-step before deciding on an action.
*   Ensure your output is a single valid JSON object. Do not include any text outside the JSON structure.
*   If you think all the requirements of user\'s instruction have been completed and no further operation is required, output the **Done** action to terminate the operation process..
*   Note that you must verify if you've truly fulfilled the user's request by examining the actual page content, not just by looking at the actions you output but also whether the action is executed successfully.
*   When you want to kill an app in the background, you can use the swipe action. The starting point is the center of the app, and the end point must be set to the edge of the screen.
*   When you encounter an unexpected pop-up window while performing a task, you must try to close it first.
*   If additional pop-ups or warnings appear during execution, prioritize closing them before proceeding to the next action.
*   If you encounter a black screen with no detectable UI elements, wait at least 10 seconds before re-detecting the screen state and proceeding with subsequent actions.

*   **SPECIAL INSTRUCTION FOR NetCap APP:** When you detect that the current application is NetCap, immediately:
    1. Check the screen for any visible UUID string (typically a 36-character string in the format dc801926-d52b-4531-b353-16a5d9f45529)
    2. If found, extract the UUID and output it in the "import_contents" field
    3. Once the UUID is extracted and output, you can terminate the process with the "Done" action

{os_specific_hints if os_specific_hints else ""}

**Example Output:**
{output_example}
"""

        return system_prompt


class AutoExecAgentPrompt:
    def __init__(self):
        pass

    def get_agent_prompt(self,
                         current_step: int,
                         ) -> str:
        """
        Constructs the user prompt for the current step.

        Args:
            current_step: The current step number (starting from 1).
            prev_action_result: A string describing the outcome of the previous action's *execution* (optional).

        Returns:
            The user prompt string.
        """
        prompt_parts = []
        prompt_parts.append(f"**Step:** {current_step}")
        prompt_parts.append("\n**Current Screenshot (with highlighted elements and indices):**")
        prompt_parts.append(
            "\nPlease analyze the screen, evaluate the previous action's visual outcome based *only* on the image, and provide your response in the required JSON format to continue the task.")

        return "\n".join(prompt_parts)


class AndroidExecSystemPrompt(AutoExecSystemPrompt):
    def get_os_specific_hints(self) -> str:
        hints = """
You are now on Android OS.
*  To find apps, **swipe left/right** on home screens.
*  If expected elements don't appear instantly (due to loading), using the **wait** action.
*  Use the **back** action to go to the previous screen. But remember **back** action maybe invalid in game.
*  When clicking on an app, do not click on the app name, always click the center of the app icon.
"""
        return hints
