#!/usr/bin/env python3
import os
import sys
import shutil
import subprocess
import argparse
from pathlib import Path


def create_windows_startup_script(output_dir):
    """Create Windows batch startup script."""
    bat_content = """@echo off
.\\tools\\uv.exe venv --python 3.11
call .venv\\Scripts\\activate
.\\tools\\uv.exe sync
.\\tools\\uv.exe run app.py
"""
    with open(os.path.join(output_dir, "start.bat"), "w") as f:
        f.write(bat_content)


def create_unix_startup_script(output_dir):
    """Create Unix shell startup script."""
    sh_content = """#!/bin/bash
./tools/uv venv --python 3.11
source .venv/bin/activate
./tools/uv sync
./tools/uv run app.py
"""
    script_path = os.path.join(output_dir, "start.sh")
    with open(script_path, "w") as f:
        f.write(sh_content)
    # Make the script executable
    os.chmod(script_path, 0o755)


def copy_platform_uv(platform, output_dir):
    """Copy the appropriate uv executable for the given platform."""
    uv_mapping = {
        "windows.x86_64": "tools/uv/uv-x86_64-pc-windows-msvc/uv.exe",
        "linux.x86_64": "tools/uv/uv-x86_64-unknown-linux-gnu/uv",
        "darwin.x86_64": "tools/uv/uv-x86_64-apple-darwin/uv",
        "darwin.arm64": "tools/uv/uv-aarch64-apple-darwin/uv"
    }

    if platform not in uv_mapping:
        print(f"Warning: No uv executable mapping for platform {platform}")
        return

    source_path = uv_mapping[platform]
    if not os.path.exists(source_path):
        print(f"Warning: uv executable not found at {source_path}")
        return

    # Create tools directory
    tools_dir = os.path.join(output_dir, "tools")
    os.makedirs(tools_dir, exist_ok=True)

    # Copy and rename the executable
    shutil.copy2(source_path, tools_dir)

    # Make executable (for Unix platforms)
    if not platform.startswith("windows"):
        os.chmod(os.path.join(tools_dir, "uv"), 0o755)

    print(f"Copied uv executable for {platform} to {tools_dir}")


def main():
    # Project settings
    PROJECT_NAME = "qagent-omni"
    TARGET_PACKAGE = "qagent_omni"
    TARGET_SCRIPT = "app.py"
    BASE_OUTPUT_DIR = "release"

    # Platforms to build for
    platforms = [
        "windows.x86_64",
        "linux.x86_64",
        "darwin.x86_64",
        "darwin.arm64"
    ]

    # Create base output directory if it doesn't exist
    os.makedirs(BASE_OUTPUT_DIR, exist_ok=True)

    # Process each platform
    for platform in platforms:
        # Create platform-specific output directory
        platform_dir_name = f"{PROJECT_NAME}-{platform}"
        platform_output_dir = os.path.join(BASE_OUTPUT_DIR, platform_dir_name)

        # Create necessary directories
        os.makedirs(platform_output_dir, exist_ok=True)
        os.makedirs(os.path.join(platform_output_dir, "data"), exist_ok=True)
        os.makedirs(os.path.join(platform_output_dir, "data", "google_keys"), exist_ok=True)

        # Copy necessary files
        if os.path.exists("data/icon_templates"):
            shutil.copytree("data/icon_templates", os.path.join(platform_output_dir, "data", "icon_templates"),
                            dirs_exist_ok=True)

        if os.path.exists("pyproject.toml"):
            shutil.copy("pyproject.toml", platform_output_dir)

        if os.path.exists(".python-version"):
            shutil.copy(".python-version", platform_output_dir)

        if os.path.exists("uv.lock"):
            shutil.copy("uv.lock", platform_output_dir)

        # Copy platform-specific uv executable
        copy_platform_uv(platform, platform_output_dir)

        # Encrypt Google API key
        subprocess.run([
            "python", "scripts/encrypt_google_json.py",
            "data/google_keys/turinglab-507d7c079329.json",
            os.path.join(platform_output_dir, "data", "google_keys", "turinglab-507d7c079329.encrypted.key")
        ])

        # Encrypt environment file
        subprocess.run([
            "python", "scripts/encrypt_env_file.py",
            ".env.outsource",
            os.path.join(platform_output_dir, ".env.encrypted")
        ])

        # Clean up old obfuscated files if they exist
        target_package_dir = os.path.join(platform_output_dir, TARGET_PACKAGE)
        target_script_path = os.path.join(platform_output_dir, TARGET_SCRIPT)

        if os.path.exists(target_package_dir):
            shutil.rmtree(target_package_dir)

        if os.path.exists(target_script_path):
            os.remove(target_script_path)

        # Run pyarmor to obfuscate code
        print(f"Generating obfuscated code for platform: {platform}")
        subprocess.run([
            "pyarmor", "gen",
            "--platform", platform,
            "-r", TARGET_SCRIPT, TARGET_PACKAGE,
            "-O", platform_output_dir
        ])

        # Create platform-specific startup scripts
        if platform.startswith("windows"):
            create_windows_startup_script(platform_output_dir)
        else:
            create_unix_startup_script(platform_output_dir)

        print(f"Successfully created obfuscated package for {platform} at {platform_output_dir}")

    print(f"All platform packages generated successfully in {BASE_OUTPUT_DIR}")


if __name__ == "__main__":
    main()

