#!/usr/bin/env python3
import subprocess
import os
import sys
import re

def get_available_gpus():
    """
    Check for available NVIDIA GPUs that aren't heavily used.
    Returns a list of available GPU IDs.
    """
    try:
        # Run nvidia-smi to get GPU usage information
        result = subprocess.run(['nvidia-smi', '--query-gpu=index,utilization.gpu,memory.used,memory.total', 
                                '--format=csv,noheader,nounits'], 
                               stdout=subprocess.PIPE, stderr=subprocess.PIPE, text=True, check=True)
        
        available_gpus = []
        for line in result.stdout.strip().split('\n'):
            # Parse the output
            parts = line.split(',')
            if len(parts) >= 4:
                gpu_id = int(parts[0].strip())
                gpu_util = float(parts[1].strip())
                memory_used = float(parts[2].strip())
                memory_total = float(parts[3].strip())
                
                # Calculate memory usage percentage
                memory_percent = (memory_used / memory_total) * 100 if memory_total > 0 else 100
                
                # Consider a GPU available if both utilization and memory usage are below thresholds
                if gpu_util < 10 and memory_percent < 10:  # Less than 10% utilization and memory usage
                    available_gpus.append(gpu_id)
        
        return available_gpus
    except subprocess.CalledProcessError:
        print("Error: Failed to run nvidia-smi. Make sure NVIDIA drivers are installed.", file=sys.stderr)
        return []
    except Exception as e:
        print(f"Error checking GPU availability: {e}", file=sys.stderr)
        return []

def get_gpu_ids_from_config(config_gpu_ids):
    """
    Process GPU IDs from config. If [-1] is provided, return all available GPUs.
    Otherwise, return the configured GPU IDs.
    """
    if len(config_gpu_ids) == 1 and config_gpu_ids[0] == -1:
        # Special case: use all available GPUs
        available_gpus = get_available_gpus()
        if not available_gpus:
            print("Warning: No available GPUs found. Using GPU 0 as fallback.", file=sys.stderr)
            return [0]  # Fallback to GPU 0
        return available_gpus
    else:
        # Return the configured GPUs
        return config_gpu_ids

if __name__ == "__main__":
    # If arguments are provided, interpret them as the GPU IDs from config
    if len(sys.argv) > 1:
        try:
            # Try to parse the arguments as a list of GPU IDs
            gpu_ids = [int(arg) for arg in sys.argv[1:]]
            result = get_gpu_ids_from_config(gpu_ids)
            # Print the result as a space-separated list
            print(" ".join(map(str, result)))
        except ValueError:
            print("Error: Invalid GPU ID format. Must be integers.", file=sys.stderr)
            sys.exit(1)
    else:
        # If no arguments, just print all available GPUs
        available_gpus = get_available_gpus()
        print(" ".join(map(str, available_gpus))) 