import cv2
import numpy as np
from PIL import Image
from diffusers.utils import export_to_video
import argparse
import os

def process_video_fps(video_path, target_fps=30):
    print(f"[Processing video: {video_path}]")
    print(f"[Target FPS: {target_fps}]")
    
    # Read video using cv2
    cap = cv2.VideoCapture(video_path)
    
    # Get video properties
    fps = cap.get(cv2.CAP_PROP_FPS)
    frame_count = int(cap.get(cv2.CAP_PROP_FRAME_COUNT))
    print(f"[Original video FPS: {fps}]")
    print(f"[Total frames: {frame_count}]")
    
    # Read all frames
    frames = []
    print("[Reading frames...]")
    while cap.isOpened():
        ret, frame = cap.read()
        if not ret:
            break
        # Convert BGR to RGB
        frame_rgb = cv2.cvtColor(frame, cv2.COLOR_BGR2RGB)
        # Convert to PIL Image
        pil_frame = Image.fromarray(frame_rgb)
        frames.append(pil_frame)
        
        if len(frames) == 49:
            break
    
    cap.release()
    print(f"[Successfully read {len(frames)} frames]")
    
    # Export video with target fps
    print("[Exporting video with new FPS...]")
    
    # Handle GIF files by changing extension to mp4
    output_path = video_path
    if video_path.lower().endswith('.gif'):
        output_path = os.path.splitext(video_path)[0] + '.mp4'
    
    export_to_video(frames, output_path, target_fps)
    print("[Video processing complete!]")
    
    return output_path

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Process video FPS")
    parser.add_argument("--video_path", type=str, required=True, help="Path to input video")
    parser.add_argument("--target_fps", type=int, default=30, help="Target FPS for output video")
    
    args = parser.parse_args()
    
    process_video_fps(args.video_path, args.target_fps)
