import os
import shutil
from pathlib import Path
import glob
from PIL import Image

def create_directory(dir_path):
    """Create directory if it doesn't exist."""
    if not os.path.exists(dir_path):
        os.makedirs(dir_path)

def subsample_frames(base_dir, data_name, start_frame, stride, total_frames, special_case=None):
    """
    Subsample frames from a directory and save them to a new location.
    
    Args:
        base_dir: Base directory containing all frame folders
        data_name: Name of the folder containing the frames
        start_frame: Starting frame index (0-indexed)
        stride: Stride for sampling
        total_frames: Total number of frames to sample
        special_case: Special handling for specific cases
    """
    # Input and output directories
    source_dir = os.path.join(base_dir, data_name)
    output_dir = os.path.join(base_dir, f"subsampled_{data_name}")
    
    # Create output directory
    create_directory(output_dir)
    
    # Get all frame files and sort them
    frame_files = sorted(glob.glob(os.path.join(source_dir, "*.png")))
    
    if not frame_files:
        print(f"No frames found in {source_dir}")
        return
    
    # Handle special case for "what-are-waiting-for-s3"
    if special_case == "what-are-waiting-for-s3":
        # First part: 0 to 42 (43 frames)
        first_part_frames = frame_files[0:43]
        # Second part: 6 frames starting from index 140
        second_part_frames = frame_files[140:146]
        selected_frames = first_part_frames + second_part_frames
    else:
        # Normal subsampling
        max_idx = start_frame + stride * (total_frames - 1)
        if max_idx >= len(frame_files):
            print(f"Warning: Not enough frames in {data_name}. Need index up to {max_idx} but only have {len(frame_files)} frames.")
        
        selected_frames = [frame_files[start_frame + i * stride] for i in range(total_frames) 
                          if start_frame + i * stride < len(frame_files)]
    
    # Copy selected frames to output directory
    for idx, frame_path in enumerate(selected_frames):
        frame_name = os.path.basename(frame_path)
        # Keep original filename for reference
        new_path = os.path.join(output_dir, frame_name)
        shutil.copy2(frame_path, new_path)
    
    print(f"Subsampled {len(selected_frames)} frames from {data_name} to {output_dir}")
    return output_dir

def create_video_from_frames(frames_dir, output_video_path, fps=30):
    """
    Create a video from frames using diffusers.utils.export_to_video
    
    Args:
        frames_dir: Directory containing the frames
        output_video_path: Path to save the output video
        fps: Frames per second for the video
    """
    try:
        from diffusers.utils import export_to_video
    except ImportError:
        print("Failed to import export_to_video from diffusers.utils. Make sure you've activated the netflix environment.")
        return False

    # Get all frame files and sort them
    frame_files = sorted(glob.glob(os.path.join(frames_dir, "*.png")))
    
    if not frame_files:
        print(f"No frames found in {frames_dir}")
        return False
    
    # Load frames as PIL images
    pil_images = [Image.open(frame_path) for frame_path in frame_files]
    
    # Create video using export_to_video
    print(f"Creating video with {len(pil_images)} frames at {fps} fps...")
    export_to_video(pil_images, output_video_path, fps=fps)
    print(f"Video saved to {output_video_path}")
    return True

def main():
    base_dir = "/home/grads/hidir/camera-control/frames"
    videos_dir = "/home/grads/hidir/camera-control/appendix_scenes"
    create_directory(videos_dir)
    
    # Define subsampling strategies for each data_name
    subsampling_configs = [
        # {"data_name": "without-doubt", "start_frame": 0, "stride": 2, "total_frames": 49},
        # {"data_name": "you-asked-me-inception-s1", "start_frame": 0, "stride": 3, "total_frames": 49},
        # {"data_name": "you-know-i-dont-usually-hold-with-foreign-food", "start_frame": 29, "stride": 3, "total_frames": 49},
        # {"data_name": "you-look-terrible-i-want-you-eat", "start_frame": 0, "stride": 3, "total_frames": 49},
        # {"data_name": "whats-going-on-out-there-wayne", "start_frame": 0, "stride": 1, "total_frames": 49},
        # {"data_name": "what-are-waiting-for-s3", "start_frame": 0, "stride": 1, "total_frames": 49, "special_case": "what-are-waiting-for-s3"},
        # {"data_name": "throughout-human-history-have-been-dependent-on-machines-survive", "start_frame": 50, "stride": 1, "total_frames": 49},
        # {"data_name": "they-are-taking-the-hobbits-isengard-s1", "start_frame": 0, "stride": 1, "total_frames": 49},
        # {"data_name": "senator-cauley-apologized-not-coming-said-youd-understand (1)", "start_frame": 78, "stride": 1, "total_frames": 49},
        # {"data_name": "now-i-were-a-200-pound-sack", "start_frame": 0, "stride": 2, "total_frames": 49},
        # {"data_name": "not-possible-that-many-dreams-within-dreams-too-unstable", "start_frame": 0, "stride": 1, "total_frames": 49},
        # {"data_name": "if-find-a-home", "start_frame": 0, "stride": 2, "total_frames": 49},
        # {"data_name": "can-sail-under-the-command-of-a-pirate", "start_frame": 45, "stride": 1, "total_frames": 49},
        # {"data_name": "penny-penny-penny-s8", "start_frame": 30, "stride": 1, "total_frames": 49},
        # {"data_name": "loki-grunts", "start_frame": 23, "stride": 1, "total_frames": 49},
        # {"data_name": "when-know-tellers-critical-assumptions (1)", "start_frame": 55, "stride": 2, "total_frames": 49},
        # {"data_name": "all-i-have-in-world-my-balls-my-word", "start_frame": 0, "stride": 2, "total_frames": 49},
        # {"data_name": "clears-throat-s59", "start_frame": 0, "stride": 2, "total_frames": 49},
        # {"data_name": "when-did-escape-this-prisoner", "start_frame": 26, "stride": 1, "total_frames": 49},
        # {"data_name": "all-i-know-its-a-mental-hospital-s1", "start_frame": 110, "stride": 1, "total_frames": 49},
        # {"data_name": "morning-temp", "start_frame": 300, "stride": 1, "total_frames": 49},
        # {"data_name": "this-year-i-decided-really-get-into-the-spirit-of-halloween", "start_frame": 25, "stride": 1, "total_frames": 49},
        # {"data_name": "blows", "start_frame": 0, "stride": 1, "total_frames": 49},
        # {"data_name": "everythings-better-with-some-wine-in-the-belly-s1", "start_frame": 0, "stride": 1, "total_frames": 49},
        {"data_name": "jacket-on-s1", "start_frame": 75, "stride": 1, "total_frames": 49},
    ]
    
    # Process each data_name
    for config in subsampling_configs:
        data_name = config["data_name"]
        stride = config["stride"]
        special_case = config.get("special_case", None)
        print(f"Processing {data_name}...")
        
        # Calculate fps based on stride
        fps = 30 // stride
        
        # Subsample frames first (if not already done)
        subsampled_dir = os.path.join(base_dir, f"subsampled_{data_name}")
        if os.path.exists(subsampled_dir):
            continue
        
        subsampled_dir = subsample_frames(
            base_dir,
            data_name,
            config["start_frame"],
            stride,
            config["total_frames"],
            special_case
        )
        
        # Create video from subsampled frames with adjusted fps
        video_output_path = os.path.join(videos_dir, f"{data_name}.mp4")
        create_video_from_frames(subsampled_dir, video_output_path, fps=fps)

if __name__ == "__main__":
    main()
