import os
import cv2
import argparse
from pathlib import Path

def extract_frames(video_path, output_dir=None, target_width=720, target_height=480, resize_image=False, crop=False):
    """
    Extract frames from a video and resize them to the target dimensions.
    
    Args:
        video_path (str): Path to the video file
        output_dir (str, optional): Directory to save the frames. If None, will use frames/[video_name]
        target_width (int): Target width for the resized frames
        target_height (int): Target height for the resized frames
    
    Returns:
        str: Path to the output directory
    """
    # Get video name from path (everything after the last '/' until .mp4)
    video_name = os.path.basename(video_path)
    if video_name.endswith('.mp4'):
        video_name = video_name[:-4]  # Remove .mp4 extension
    
    # Create output directory if not specified
    if output_dir is None:
        output_dir = os.path.join('frames', video_name)
    
    # Create directory if it doesn't exist
    os.makedirs(output_dir, exist_ok=True)
    
    # Open the video file
    video = cv2.VideoCapture(video_path)
    
    if not video.isOpened():
        raise ValueError(f"Could not open video file: {video_path}")
    
    frame_idx = 0
    while True:
        # Read frame
        ret, frame = video.read()
        if not ret:
            break
        
        # Resize frame to target dimensions
        if resize_image:
            resized_frame = cv2.resize(frame, (target_width, target_height))
        else:
            resized_frame = frame
        
        # Save frame as PNG
        output_path = os.path.join(output_dir, f'{frame_idx:04d}.jpg')
        if crop:
            print("a")
            resized_frame = resized_frame[-480:, -720:]
            
        cv2.imwrite(output_path, resized_frame)
        
        frame_idx += 1
    
    video.release()
    print(f"Extracted {frame_idx} frames from {video_path} to {output_dir}")
    return output_dir

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Extract and resize frames from a video")
    parser.add_argument("--video_path", type=str, required=True, help="Path to the video file")
    parser.add_argument("--output_dir", default="figures", type=str, help="Directory to save the frames (default: frames/[video_name])")
    parser.add_argument("--width", type=int, default=720, help="Target width for the resized frames")
    parser.add_argument("--height", type=int, default=480, help="Target height for the resized frames")
    parser.add_argument("--resize_image", action="store_true", help="Resize the frames to the target dimensions")
    parser.add_argument("--crop", action="store_true", help="crop the frames to the target dimensions")
    args = parser.parse_args()
    
    extract_frames(
        args.video_path, 
        args.output_dir, 
        args.width, 
        args.height,
        args.resize_image,
        args.crop
    )
