#    Copyright 2023 Haotian Liu
#
#    Licensed under the Apache License, Version 2.0 (the "License");
#    you may not use this file except in compliance with the License.
#    You may obtain a copy of the License at
#
#        http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS,
#    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#    See the License for the specific language governing permissions and
#    limitations under the License.


from abc import ABC, abstractmethod

import torch
import torch.nn as nn
import torch.nn.functional as F

from .multimodal_encoder.builder import build_vision_tower
from .multimodal_projector.builder import build_vision_projector

from llava.constants import IGNORE_INDEX, IMAGE_TOKEN_INDEX, DEFAULT_IMAGE_PATCH_TOKEN, DEFAULT_IM_START_TOKEN, DEFAULT_IM_END_TOKEN

from llava.mm_utils import get_anyres_image_grid_shape


class LlavaMetaModel:

    def __init__(self, config, **kwargs):
        super(LlavaMetaModel, self).__init__(config, **kwargs)

        if hasattr(config, "mm_vision_tower"):
            self.vision_tower = build_vision_tower(config, delay_load=True)
            self.mm_projector = build_vision_projector(config)

            if 'unpad' in getattr(config, 'mm_patch_merge_type', ''):
                self.image_newline = nn.Parameter(
                    torch.empty(config.hidden_size, dtype=self.dtype)
                )

    def get_vision_tower(self):
        vision_tower = getattr(self, 'vision_tower', None)
        if type(vision_tower) is list:
            vision_tower = vision_tower[0]
        return vision_tower

    def initialize_vision_modules(self, model_args, fsdp=None):
        vision_tower = model_args.vision_tower
        mm_vision_select_layer = model_args.mm_vision_select_layer
        mm_vision_select_feature = model_args.mm_vision_select_feature
        pretrain_mm_mlp_adapter = model_args.pretrain_mm_mlp_adapter
        mm_patch_merge_type = model_args.mm_patch_merge_type

        self.config.mm_vision_tower = vision_tower

        if self.get_vision_tower() is None:
            vision_tower = build_vision_tower(model_args)

            if fsdp is not None and len(fsdp) > 0:
                self.vision_tower = [vision_tower]
            else:
                self.vision_tower = vision_tower
        else:
            if fsdp is not None and len(fsdp) > 0:
                vision_tower = self.vision_tower[0]
            else:
                vision_tower = self.vision_tower
            vision_tower.load_model()

        self.config.use_mm_proj = True
        self.config.mm_projector_type = getattr(model_args, 'mm_projector_type', 'linear')
        self.config.mm_hidden_size = vision_tower.hidden_size
        self.config.mm_vision_select_layer = mm_vision_select_layer
        self.config.mm_vision_select_feature = mm_vision_select_feature
        self.config.mm_patch_merge_type = mm_patch_merge_type

        if getattr(self, 'mm_projector', None) is None:
            self.mm_projector = build_vision_projector(self.config)

            if 'unpad' in mm_patch_merge_type:
                embed_std = 1 / torch.sqrt(torch.tensor(self.config.hidden_size, dtype=self.dtype))
                self.image_newline = nn.Parameter(
                    torch.randn(self.config.hidden_size, dtype=self.dtype) * embed_std
                )
        else:
            # In case it is frozen by LoRA
            for p in self.mm_projector.parameters():
                p.requires_grad = True

        if pretrain_mm_mlp_adapter is not None:
            mm_projector_weights = torch.load(pretrain_mm_mlp_adapter, map_location='cpu')
            def get_w(weights, keyword):
                return {k.split(keyword + '.')[1]: v for k, v in weights.items() if keyword in k}

            self.mm_projector.load_state_dict(get_w(mm_projector_weights, 'mm_projector'))


def unpad_image(tensor, original_size):
    """
    Unpads a PyTorch tensor of a padded and resized image.

    Args:
    tensor (torch.Tensor): The image tensor, assumed to be in CxHxW format.
    original_size (tuple): The original size of PIL image (width, height).

    Returns:
    torch.Tensor: The unpadded image tensor.
    """
    original_width, original_height = original_size
    current_height, current_width = tensor.shape[1:]

    original_aspect_ratio = original_width / original_height
    current_aspect_ratio = current_width / current_height

    if original_aspect_ratio > current_aspect_ratio:
        scale_factor = current_width / original_width
        new_height = int(original_height * scale_factor)
        padding = (current_height - new_height) // 2
        unpadded_tensor = tensor[:, padding:current_height - padding, :]
    else:
        scale_factor = current_height / original_height
        new_width = int(original_width * scale_factor)
        padding = (current_width - new_width) // 2
        unpadded_tensor = tensor[:, :, padding:current_width - padding]

    return unpadded_tensor


class LlavaMetaForCausalLM(ABC):

    @abstractmethod
    def get_model(self):
        pass

    def get_vision_tower(self):
        return self.get_model().get_vision_tower()

    def encode_images(self, images, texts=None):
        if 'prumerge' in self.pruning_method or self.pruning_method == 'visionzip' or self.pruning_method == 'fastervlm':
            image_features, image_attentions, image_keys, image_cls = self.get_model().get_vision_tower()(images, output_attentions=True)
        elif self.pruning_method == 'trim' or self.pruning_method == 'cdpruner':
            image_features, image_embeds, text_embeds = self.get_model().get_vision_tower()(images, texts=texts)
        else:
            image_features = self.get_model().get_vision_tower()(images)
        
        B, N, C = image_features.shape
        device = image_features.device
        index_masks = torch.ones(B, N, dtype=torch.bool, device=device)
        merged_features = None
        
        if 'prumerge' in self.pruning_method:
            cls_attn = image_attentions.mean(dim=1)

            if self.pruning_method == 'prumerge':
                selected_idx = cls_attn.topk(k=self.visual_token_num-1, dim=1).indices
            elif self.pruning_method == 'prumerge+':
                step_length = int(N / self.visual_token_num * 2)
                arithmetic_idx = torch.arange(int(step_length/2), 575, int(step_length), device=device)
                cls_attn[:, arithmetic_idx] = 0
                selected_idx = cls_attn.topk(k=self.visual_token_num//2-1, dim=1).indices
                selected_idx = torch.cat([selected_idx, arithmetic_idx.unsqueeze(0).expand(B, -1)], dim=1)
            
            index_masks = torch.zeros(B, N, dtype=torch.bool, device=device)
            index_masks.scatter_(1, selected_idx, True)
            pruned_idx = index_masks.float().topk(k=N-self.visual_token_num+1, dim=1, largest=False).indices

            selected_feat = torch.gather(image_features, dim=1, index=selected_idx.unsqueeze(-1).expand(-1, -1, C))
            selected_key = torch.gather(image_keys, dim=1, index=selected_idx.unsqueeze(-1).expand(-1, -1, C))
            selected_attn = torch.gather(cls_attn, dim=1, index=selected_idx)
            pruned_feat = torch.gather(image_features, dim=1, index=pruned_idx.unsqueeze(-1).expand(-1, -1, C))
            pruned_key = torch.gather(image_keys, dim=1, index=pruned_idx.unsqueeze(-1).expand(-1, -1, C))
            pruned_attn = torch.gather(cls_attn, dim=1, index=pruned_idx)

            # selected_norm = F.normalize(selected_feat, p=2, dim=-1)
            # pruned_norm = F.normalize(pruned_feat, p=2, dim=-1)
            selected_norm = F.normalize(selected_key, p=2, dim=-1)
            pruned_norm = F.normalize(pruned_key, p=2, dim=-1)

            updated_feat = torch.zeros_like(selected_feat)
            for b in range(B):
                for i in range(updated_feat.shape[1]):
                    center_norm = selected_norm[b, i, :].unsqueeze(0)
                    others_norm = torch.cat([
                        selected_norm[b, :i, :],
                        selected_norm[b, i+1:, :],
                        pruned_norm[b, :, :],
                    ], dim=0)

                    # calculate cosine similarity and get cluster centers
                    cos_sim_matrix = center_norm @ others_norm.t()
                    cluster_idx = torch.topk(cos_sim_matrix, k=32, dim=1).indices

                    others_feat = torch.cat([
                        selected_feat[b, :i, :],
                        selected_feat[b, i+1:, :],
                        pruned_feat[b, :, :],
                    ], dim=0)
                    others_attn = torch.cat([
                        selected_attn[b, :i],
                        selected_attn[b, i+1:],
                        pruned_attn[b, :],
                    ], dim=0)

                    cluster_tokens = others_feat[cluster_idx.squeeze(), :]
                    cluster_weights = others_attn[cluster_idx.squeeze()].unsqueeze(-1) # (1, 32, 1)

                    # update cluster centers
                    cluster_avg = torch.sum(cluster_tokens * cluster_weights, dim=0)
                    cluster_center = cluster_avg + selected_feat[b, i, :] * (1 - torch.sum(cluster_weights))
                    updated_feat[b, i, :] = cluster_center
            image_features[index_masks] = updated_feat.flatten(0, 1)
            
            extra_weight = pruned_attn / torch.sum(pruned_attn, dim=1, keepdim=True)
            merged_features = torch.sum(pruned_feat * extra_weight.unsqueeze(-1), dim=1, keepdim=True)
        
        elif self.pruning_method == 'visionzip':
            dominant_token_num = int(self.visual_token_num * 27 / 32) - 1
            contextual_token_num = self.visual_token_num - dominant_token_num - 1

            # dominant visual tokens
            cls_attn = image_attentions.mean(dim=1)
            topk_idx = cls_attn.topk(dominant_token_num, dim=1).indices

            index_masks = torch.zeros(B, N, dtype=torch.bool, device=device)
            index_masks.scatter_(1, topk_idx, True)
            
            # filter
            features_filtered = image_features.masked_select(~index_masks.unsqueeze(-1)).view(B, N - dominant_token_num, C)
            metric_filtered = image_keys[~index_masks].view(B, N - dominant_token_num, C)
            # metric_normalized = features_filtered / features_filtered.norm(dim=-1, keepdim=True)
            metric_normalized = metric_filtered / metric_filtered.norm(dim=-1, keepdim=True)

            # contextual visual tokens
            target_step = max(1, metric_normalized.shape[1] // contextual_token_num)
            target_indices = torch.arange(0, metric_normalized.shape[1], target_step, device=device)[:contextual_token_num]
            target_tokens = metric_normalized[:, target_indices, :]

            tokens_to_merge = metric_normalized[:, ~torch.isin(torch.arange(metric_normalized.shape[1], device=device), target_indices), :]
            similarity = torch.bmm(tokens_to_merge, target_tokens.transpose(1, 2))
            assign_one_hot = features_filtered.new_zeros(B, tokens_to_merge.shape[1], contextual_token_num)
            assign_one_hot.scatter_(2, similarity.argmax(dim=2).unsqueeze(-1), 1)
            counts = assign_one_hot.sum(dim=1).clamp(min=1).unsqueeze(-1)
            hidden_to_merge = features_filtered[:, ~torch.isin(torch.arange(features_filtered.shape[1], device=device), target_indices), :]
            aggregated_hidden = torch.bmm(assign_one_hot.transpose(1, 2), hidden_to_merge) / counts
            target_hidden = features_filtered[:, target_indices, :]
            contextual_tokens = target_hidden + aggregated_hidden

            # merge
            merged_features = torch.cat([image_cls, contextual_tokens], dim=1)
        
        elif self.pruning_method == 'fastervlm':
            cls_attn = image_attentions.mean(dim=1)

            topk_idx = cls_attn.topk(self.visual_token_num, dim=-1).indices
            topk_idx = torch.sort(topk_idx).values

            index_masks = torch.zeros(B, N, dtype=torch.bool, device=device)
            index_masks.scatter_(1, topk_idx, True)
        
        elif self.pruning_method == 'trim':
            image_embeds = image_embeds / image_embeds.norm(dim=-1, keepdim=True)
            text_embeds = text_embeds / text_embeds.norm(dim=-1, keepdim=True)
            similarity = -torch.matmul(image_embeds, text_embeds.t())
            similarity = similarity.mean(dim=-1)

            # select topk tokens by similarity scores
            topk_idx = similarity.topk(self.visual_token_num - 1, dim=-1).indices
            topk_idx = torch.sort(topk_idx).values

            index_masks = torch.zeros(B, N, dtype=torch.bool, device=device)
            index_masks.scatter_(1, topk_idx, True)

            # aggregate the remaining tokens
            remaining_tokens = image_features[~index_masks].view(B, -1, C)
            merged_features = remaining_tokens.mean(dim=1, keepdim=True)
        
        elif self.pruning_method == 'dart':
            image_normalized = (image_features / image_features.norm(dim=-1, keepdim=True))
            visual_token_num = (self.visual_token_num - 8) // 8
            step = N // (8 + 1)

            retained_idx = []
            for b in range(B):
                retained_set = set(range(step, N, step))
                valid_set = set(range(N)) - retained_set
                for pivot_idx in list(retained_set):
                    valid_list = list(valid_set)
                    cos_sim = torch.matmul(image_normalized[b, pivot_idx], image_normalized[b, valid_list].t())
                    topk_idx = torch.topk(cos_sim, k=visual_token_num, largest=False).indices

                    topk_real_idx = [valid_list[idx] for idx in topk_idx]
                    retained_set.update(topk_real_idx)
                    valid_set.difference_update(topk_real_idx)
                retained_idx.append(list(retained_set))
            
            retained_idx = torch.tensor(retained_idx, dtype=torch.long, device=device)
            retained_idx = torch.sort(retained_idx).values

            index_masks = torch.zeros(B, N, dtype=torch.bool, device=device)
            index_masks.scatter_(1, retained_idx, True)
        
        image_features = self.get_model().mm_projector(image_features)

        if merged_features is not None:
            merged_features = self.get_model().mm_projector(merged_features)
        
        if self.pruning_method == 'divprune':
            image_normalized = (image_features / image_features.norm(dim=-1, keepdim=True))
            cosine_matrix = 1.0 - torch.matmul(image_normalized, image_normalized.transpose(1, 2))
            
            select_idx = torch.empty((B, self.visual_token_num), dtype=torch.long, device=device)
            for i in range(self.visual_token_num):
                m2 = cosine_matrix if i==0 \
                    else cosine_matrix[torch.arange(B).unsqueeze(1).expand(-1, i), select_idx[:, :i]]
                scores = torch.topk(m2, 2, dim=1, largest=False).values[:, 1] \
                    if i==0 else torch.min(m2, dim=1).values
                select_idx[:, i] = torch.argmax(scores, dim=-1)
            
            select_idx = torch.sort(select_idx).values
            index_masks = torch.zeros(B, N, dtype=torch.bool, device=device)
            index_masks.scatter_(1, select_idx, True)
        
        elif self.pruning_method == 'dppruner':
            image_normalized = image_features / image_features.norm(dim=-1, keepdim=True) # (B, N, D)
            image_normalized = image_normalized.float() # (B, N, D)
            similarity = torch.matmul(image_normalized, image_normalized.transpose(1, 2)) # (B, N, N)

            kernel = similarity

            cis = torch.zeros((self.visual_token_num, B, N), device=device) # (T, B, N)
            di2s = torch.diagonal(kernel, dim1=1, dim2=2).clone() # (B, N)
            select_idx = torch.empty((self.visual_token_num, B), dtype=torch.long, device=device) # (T, B)
            for i in range(self.visual_token_num):
                j = torch.argmax(di2s, dim=-1)
                select_idx[i] = j

                eis = (kernel[torch.arange(B), j] - torch.einsum('tb,tbn->bn', cis[:i, torch.arange(B), j], cis[:i])) \
                    / torch.sqrt(di2s[torch.arange(B), j]).unsqueeze(-1)
                cis[i, :, :] = eis
                di2s -= torch.square(eis)
                di2s[torch.arange(B), j] = -float('inf')
            
            select_idx = torch.sort(select_idx.t()).values # (B, T)
            index_masks = torch.zeros(B, N, dtype=torch.bool, device=device)
            index_masks.scatter_(1, select_idx, True)
        
        elif self.pruning_method == 'cdpruner':
            image_normalized = image_features / image_features.norm(dim=-1, keepdim=True) # (B, N, D)
            image_normalized = image_normalized.float() # (B, N, D)
            similarity = torch.matmul(image_normalized, image_normalized.transpose(1, 2)) # (B, N, N)

            image_embeds = image_embeds / image_embeds.norm(dim=-1, keepdim=True) # (B, N, C)
            text_embeds = text_embeds / text_embeds.norm(dim=-1, keepdim=True) # (M, C)
            relevance = torch.matmul(image_embeds, text_embeds.t()) # (B, N, M)
            relevance = (-relevance).mean(dim=-1) # (B, N)
            relevance = (relevance - relevance.min() + 1e-6) / (relevance.max() - relevance.min()) # (B, N)

            kernel = relevance.unsqueeze(2) * similarity * relevance.unsqueeze(1) # (B, N, N)

            cis = torch.zeros((self.visual_token_num, B, N), device=device) # (T, B, N)
            di2s = torch.diagonal(kernel, dim1=1, dim2=2).clone() # (B, N)
            select_idx = torch.empty((self.visual_token_num, B), dtype=torch.long, device=device) # (T, B)
            for i in range(self.visual_token_num):
                j = torch.argmax(di2s, dim=-1)
                select_idx[i] = j

                eis = (kernel[torch.arange(B), j] - torch.einsum('tb,tbn->bn', cis[:i, torch.arange(B), j], cis[:i])) \
                    / torch.sqrt(di2s[torch.arange(B), j]).unsqueeze(-1)
                cis[i, :, :] = eis
                di2s -= torch.square(eis)
                di2s[torch.arange(B), j] = -float('inf')
            
            select_idx = torch.sort(select_idx.t()).values # (B, T)
            index_masks = torch.zeros(B, N, dtype=torch.bool, device=device)
            index_masks.scatter_(1, select_idx, True)
        
        return image_features, index_masks, merged_features

    def prepare_inputs_labels_for_multimodal(
        self, input_ids, position_ids, attention_mask, past_key_values, labels,
        images, modalities=["image"], image_sizes=None, texts=None
    ):
        vision_tower = self.get_vision_tower()
        if vision_tower is None or images is None or input_ids.shape[1] == 1:
            return input_ids, position_ids, attention_mask, past_key_values, None, labels

        if type(images) is list or images.ndim == 5:
            if type(images) is list:
                images = [x.unsqueeze(0) if x.ndim == 3 else x for x in images]
            concat_images = torch.cat([image for image in images], dim=0)
            image_features, index_masks, merged_features = self.encode_images(concat_images, texts=texts)
            split_sizes = [image.shape[0] for image in images]
            image_features = torch.split(image_features, split_sizes, dim=0)
            index_masks = torch.split(index_masks, split_sizes, dim=0)
            if merged_features is not None:
                merged_features = torch.split(merged_features, split_sizes, dim=0)
            mm_patch_merge_type = getattr(self.config, 'mm_patch_merge_type', 'flat')
            mm_patch_merge_type = mm_patch_merge_type.replace('_unpad', '')
            image_aspect_ratio = getattr(self.config, 'image_aspect_ratio', 'square')
            if mm_patch_merge_type == 'flat':
                image_features = [x.flatten(0, 1) for x in image_features]
                index_masks = [x.flatten(0, 1) for x in index_masks]
                image_features = [x[m] for x, m in zip(image_features, index_masks)]
                if merged_features is not None:
                    image_features = [torch.cat((x.reshape(y.shape[0], x.shape[0] // y.shape[0], *x.shape[1:]), y), dim=1).flatten(0, 1) \
                                      for x, y in zip(image_features, merged_features)]
            elif mm_patch_merge_type.startswith('spatial'):
                new_image_features = []
                if merged_features is None:
                    merged_features = [None] * len(image_features)
                for image_idx, (image_feature, index_mask, merged_feature) in enumerate(zip(image_features, index_masks, merged_features)):
                    if image_feature.shape[0] > 1:
                        base_image_feature = image_feature[0]
                        image_feature = image_feature[1:]
                        base_index_mask = index_mask[0]
                        index_mask = index_mask[1:]
                        if merged_feature is not None:
                            base_merged_feature = merged_feature[0]
                            merged_feature = merged_feature[1:]
                        height = width = self.get_vision_tower().num_patches_per_side
                        assert height * width == base_image_feature.shape[0]
                        if image_aspect_ratio == 'anyres':
                            num_patch_width, num_patch_height = get_anyres_image_grid_shape(image_sizes[image_idx], self.config.image_grid_pinpoints, self.get_vision_tower().config.image_size)
                            image_feature = image_feature.view(num_patch_height, num_patch_width, height, width, -1)
                            index_mask = index_mask.view(num_patch_height, num_patch_width, height, width)
                        else:
                            raise NotImplementedError
                        if 'unpad' in mm_patch_merge_type:
                            image_feature = image_feature.permute(4, 0, 2, 1, 3).contiguous()
                            image_feature = image_feature.flatten(1, 2).flatten(2, 3)
                            image_feature = unpad_image(image_feature, image_sizes[image_idx])
                            image_feature = torch.cat((
                                image_feature,
                                self.model.image_newline[:, None, None].expand(*image_feature.shape[:-1], 1).to(image_feature.device)
                            ), dim=-1)
                            image_feature = image_feature.flatten(1, 2).transpose(0, 1)
                            index_mask = index_mask.permute(0, 2, 1, 3).contiguous().unsqueeze(0)
                            index_mask = index_mask.flatten(1, 2).flatten(2, 3)
                            index_mask = unpad_image(index_mask, image_sizes[image_idx])
                            index_mask = torch.cat((
                                index_mask,
                                torch.ones(*index_mask.shape[:-1], 1, dtype=torch.bool).to(index_mask.device)
                            ), dim=-1)
                            index_mask = index_mask.flatten(1, 2).squeeze(0)
                            image_feature = image_feature[index_mask]
                            if merged_feature is not None:
                                image_feature = torch.cat((
                                    image_feature,
                                    merged_feature.flatten(0, 1)
                                ))
                        else:
                            image_feature = image_feature.permute(0, 2, 1, 3, 4).contiguous()
                            image_feature = image_feature.flatten(0, 3)
                            index_mask = index_mask.permute(0, 2, 1, 3).contiguous()
                            index_mask = index_mask.flatten(0, 3)
                            image_feature = image_feature[index_mask]
                            if merged_feature is not None:
                                image_feature = torch.cat((
                                    image_feature,
                                    merged_feature.flatten(0, 1)
                                ))
                        base_image_feature = base_image_feature[base_index_mask]
                        if merged_feature is not None:
                            base_image_feature = torch.cat((base_image_feature, base_merged_feature))
                        image_feature = torch.cat((base_image_feature, image_feature))
                    else:
                        image_feature = image_feature[0]
                        index_mask = index_mask[0]
                        if merged_feature is not None:
                            merged_feature = merged_feature[0]
                        if 'unpad' in mm_patch_merge_type:
                            image_feature = torch.cat((
                                image_feature,
                                self.model.image_newline[None].to(image_feature.device)
                            ), dim=0)
                            index_mask = torch.cat((
                                index_mask,
                                torch.ones(1, dtype=torch.bool).to(index_mask.device)
                            ), dim=0)
                        image_feature = image_feature[index_mask]
                        if merged_feature is not None:
                            image_feature = torch.cat((image_feature, merged_feature))
                    new_image_features.append(image_feature)
                image_features = new_image_features
            else:
                raise ValueError(f"Unexpected mm_patch_merge_type: {self.config.mm_patch_merge_type}")
        else:
            image_features, index_masks, merged_features = self.encode_images(images, texts=texts)
            image_features = image_features[index_masks].unsqueeze(0)
            if merged_features is not None:
                image_features = torch.cat((image_features, merged_features), dim=1)


        # TODO: image start / end is not implemented here to support pretraining.
        if getattr(self.config, 'tune_mm_mlp_adapter', False) and getattr(self.config, 'mm_use_im_start_end', False):
            raise NotImplementedError

        # Let's just add dummy tensors if they do not exist,
        # it is a headache to deal with None all the time.
        # But it is not ideal, and if you have a better idea,
        # please open an issue / submit a PR, thanks.
        _labels = labels
        _position_ids = position_ids
        _attention_mask = attention_mask
        if attention_mask is None:
            attention_mask = torch.ones_like(input_ids, dtype=torch.bool)
        else:
            attention_mask = attention_mask.bool()
        if position_ids is None:
            position_ids = torch.arange(0, input_ids.shape[1], dtype=torch.long, device=input_ids.device)
        if labels is None:
            labels = torch.full_like(input_ids, IGNORE_INDEX)

        # remove the padding using attention_mask -- FIXME
        _input_ids = input_ids
        input_ids = [cur_input_ids[cur_attention_mask] for cur_input_ids, cur_attention_mask in zip(input_ids, attention_mask)]
        labels = [cur_labels[cur_attention_mask] for cur_labels, cur_attention_mask in zip(labels, attention_mask)]

        new_input_embeds = []
        new_labels = []
        cur_image_idx = 0
        for batch_idx, cur_input_ids in enumerate(input_ids):
            num_images = (cur_input_ids == IMAGE_TOKEN_INDEX).sum()
            if num_images == 0:
                cur_image_features = image_features[cur_image_idx]
                cur_input_embeds_1 = self.get_model().embed_tokens(cur_input_ids)
                cur_input_embeds = torch.cat([cur_input_embeds_1, cur_image_features[0:0]], dim=0)
                new_input_embeds.append(cur_input_embeds)
                new_labels.append(labels[batch_idx])
                cur_image_idx += 1
                continue

            image_token_indices = [-1] + torch.where(cur_input_ids == IMAGE_TOKEN_INDEX)[0].tolist() + [cur_input_ids.shape[0]]
            cur_input_ids_noim = []
            cur_labels = labels[batch_idx]
            cur_labels_noim = []
            for i in range(len(image_token_indices) - 1):
                cur_input_ids_noim.append(cur_input_ids[image_token_indices[i]+1:image_token_indices[i+1]])
                cur_labels_noim.append(cur_labels[image_token_indices[i]+1:image_token_indices[i+1]])
            split_sizes = [x.shape[0] for x in cur_labels_noim]
            cur_input_embeds = self.get_model().embed_tokens(torch.cat(cur_input_ids_noim))
            cur_input_embeds_no_im = torch.split(cur_input_embeds, split_sizes, dim=0)
            cur_new_input_embeds = []
            cur_new_labels = []

            for i in range(num_images + 1):
                cur_new_input_embeds.append(cur_input_embeds_no_im[i])
                cur_new_labels.append(cur_labels_noim[i])
                if i < num_images:
                    cur_image_features = image_features[cur_image_idx]
                    cur_image_idx += 1
                    cur_new_input_embeds.append(cur_image_features)
                    cur_new_labels.append(torch.full((cur_image_features.shape[0],), IGNORE_INDEX, device=cur_labels.device, dtype=cur_labels.dtype))

            cur_new_input_embeds = [x.to(self.device) for x in cur_new_input_embeds]

            cur_new_input_embeds = torch.cat(cur_new_input_embeds)
            cur_new_labels = torch.cat(cur_new_labels)

            new_input_embeds.append(cur_new_input_embeds)
            new_labels.append(cur_new_labels)

        # Truncate sequences to max length as image embeddings can make the sequence longer
        tokenizer_model_max_length = getattr(self.config, 'tokenizer_model_max_length', None)
        if tokenizer_model_max_length is not None:
            new_input_embeds = [x[:tokenizer_model_max_length] for x in new_input_embeds]
            new_labels = [x[:tokenizer_model_max_length] for x in new_labels]

        # Combine them
        max_len = max(x.shape[0] for x in new_input_embeds)
        batch_size = len(new_input_embeds)

        new_input_embeds_padded = []
        new_labels_padded = torch.full((batch_size, max_len), IGNORE_INDEX, dtype=new_labels[0].dtype, device=new_labels[0].device)
        attention_mask = torch.zeros((batch_size, max_len), dtype=attention_mask.dtype, device=attention_mask.device)
        position_ids = torch.zeros((batch_size, max_len), dtype=position_ids.dtype, device=position_ids.device)

        for i, (cur_new_embed, cur_new_labels) in enumerate(zip(new_input_embeds, new_labels)):
            cur_len = cur_new_embed.shape[0]
            if getattr(self.config, 'tokenizer_padding_side', 'right') == "left":
                new_input_embeds_padded.append(torch.cat((
                    torch.zeros((max_len - cur_len, cur_new_embed.shape[1]), dtype=cur_new_embed.dtype, device=cur_new_embed.device),
                    cur_new_embed
                ), dim=0))
                if cur_len > 0:
                    new_labels_padded[i, -cur_len:] = cur_new_labels
                    attention_mask[i, -cur_len:] = True
                    position_ids[i, -cur_len:] = torch.arange(0, cur_len, dtype=position_ids.dtype, device=position_ids.device)
            else:
                new_input_embeds_padded.append(torch.cat((
                    cur_new_embed,
                    torch.zeros((max_len - cur_len, cur_new_embed.shape[1]), dtype=cur_new_embed.dtype, device=cur_new_embed.device)
                ), dim=0))
                if cur_len > 0:
                    new_labels_padded[i, :cur_len] = cur_new_labels
                    attention_mask[i, :cur_len] = True
                    position_ids[i, :cur_len] = torch.arange(0, cur_len, dtype=position_ids.dtype, device=position_ids.device)

        new_input_embeds = torch.stack(new_input_embeds_padded, dim=0)

        if _labels is None:
            new_labels = None
        else:
            new_labels = new_labels_padded

        if _attention_mask is None:
            attention_mask = None
        else:
            attention_mask = attention_mask.to(dtype=_attention_mask.dtype)

        if _position_ids is None:
            position_ids = None

        return None, position_ids, attention_mask, past_key_values, new_input_embeds, new_labels, image_features[0].shape[0]

    def initialize_vision_tokenizer(self, model_args, tokenizer):
        if model_args.mm_use_im_patch_token:
            tokenizer.add_tokens([DEFAULT_IMAGE_PATCH_TOKEN], special_tokens=True)
            self.resize_token_embeddings(len(tokenizer))

        if model_args.mm_use_im_start_end:
            num_new_tokens = tokenizer.add_tokens([DEFAULT_IM_START_TOKEN, DEFAULT_IM_END_TOKEN], special_tokens=True)
            self.resize_token_embeddings(len(tokenizer))

            if num_new_tokens > 0:
                input_embeddings = self.get_input_embeddings().weight.data
                output_embeddings = self.get_output_embeddings().weight.data

                input_embeddings_avg = input_embeddings[:-num_new_tokens].mean(
                    dim=0, keepdim=True)
                output_embeddings_avg = output_embeddings[:-num_new_tokens].mean(
                    dim=0, keepdim=True)

                input_embeddings[-num_new_tokens:] = input_embeddings_avg
                output_embeddings[-num_new_tokens:] = output_embeddings_avg

            if model_args.tune_mm_mlp_adapter:
                for p in self.get_input_embeddings().parameters():
                    p.requires_grad = True
                for p in self.get_output_embeddings().parameters():
                    p.requires_grad = False

            if model_args.pretrain_mm_mlp_adapter:
                mm_projector_weights = torch.load(model_args.pretrain_mm_mlp_adapter, map_location='cpu')
                embed_tokens_weight = mm_projector_weights['model.embed_tokens.weight']
                assert num_new_tokens == 2
                if input_embeddings.shape == embed_tokens_weight.shape:
                    input_embeddings[-num_new_tokens:] = embed_tokens_weight[-num_new_tokens:]
                elif embed_tokens_weight.shape[0] == num_new_tokens:
                    input_embeddings[-num_new_tokens:] = embed_tokens_weight
                else:
                    raise ValueError(f"Unexpected embed_tokens_weight shape. Pretrained: {embed_tokens_weight.shape}. Current: {input_embeddings.shape}. Numer of new tokens: {num_new_tokens}.")
        elif model_args.mm_use_im_patch_token:
            if model_args.tune_mm_mlp_adapter:
                for p in self.get_input_embeddings().parameters():
                    p.requires_grad = False
                for p in self.get_output_embeddings().parameters():
                    p.requires_grad = False
