
import cadquery as cq

# Constants for desk dimensions
desk_length = 100.0
desk_width = 60.0
desk_thickness = 2.0
cutout_size = 5.0

# Constants for leg dimensions
leg_side_length = 15.0    # Side length of the hexagon leg
leg_height = 50.0        # Height of the leg

# Create the main desk shape by starting with a rectangle
desk = (
    cq.Workplane("XY")
    .rect(desk_length, desk_width)
    # Cutting out right-angle triangles at each corner to form an octagonal top
    .vertices()
    .rect(cutout_size, cutout_size, forConstruction=False)
    .cutThruAll()
    # Extrude the octagonal shape to form the horizontal desk surface, check thickness
    .extrude(desk_thickness)
)

# Function to create a hexagonal prism for the legs
def hexagonal_leg():
    return (
        cq.Workplane("XY")
        .polygon(6, leg_side_length)
        .extrude(leg_height)
    )

# Adding legs placed symmetrically at both ends of the octagonal surface
# Assuming that 'cutout_size' is sufficient to place the hexagonal leg inside
leg_offset = cutout_size + leg_side_length / 2

# Add the legs to the desk at proper positions
desk = (
    desk
    # Right side leg
    .faces(">Z")
    .workplane(centerOption="CenterOfBoundBox")
    .center(-desk_length / 2 + leg_offset, 0)
    .union(hexagonal_leg())
    # Left side leg
    .center(desk_length - 2 * leg_offset, 0)
    .union(hexagonal, all=True)
)

# Export final model as STL
desk.val().exportStl("Refine_V_1.stl")
