
import cadquery as cq

# Bucket parameters
bucket_height = 10.0
bucket_radius = 5.0
bucket_thickness = 0.5

# Handle parameters - slightly larger than the diameter of the cylinder
handle_diameter = 0.5
handle_length = 2 * bucket_radius + 2.0

# Create the main outer cylinder (bucket)
bucket_outer = cq.Workplane("XY").circle(bucket_radius).extrude(bucket_height)

# Cut the inner cylinder to make the bucket hollow and with a bottom
inner_depth = bucket_height - 1.0  # Does not cut through the bottom
bucket_inner = bucket_outer.faces(">Z").workplane().circle(bucket_radius - bucket_thickness).cutBlind(-inner_depth)

# Creating cylindrical handle
# Offset it to position correctly outside the surface of the bucket
handle = cq.Workplane("XY").transformed(offset=(0, 0, bucket_height - 1)).circle(handle_diameter / 2).extrude(handle_length)

# Translate handle to the correct position
handle = handle.translate((bucket_radius + handle_diameter / 2, 0, 0))

# Make two cutouts on the bucket for the handle to pass through
cutout = cq.Workplane("XY").circle(handle_diameter / 2).extrude(bucket_thickness + 0.1)
# Position the cutouts appropriately where the handle intersects the bucket wall
bucket_final = bucket_inner.cut(cutout.translate((bucket_radius, 0, bucket_height - 1)))
bucket_final = bucket_final.cut(cutout.translate((-bucket_radius, 0, bucket_height - 1)))

# Combine the bucket with the handle
result = bucket_final.union(handle)

result.val().exportStl("Refine_V_2.stl")
