import math
import torch
import os
import sys

import imageio
import numpy as np


def generate_skill_disc(dim, eval_idx = -1):

    vector = np.full(dim, 0)

    if eval_idx != -1:
        vector[eval_idx] = 1
    else:
        idx = np.random.randint(dim)
        vector[idx] = 1
    
    return vector


def generate_skill_cont(dim):

    while True:
        vector = np.random.normal(0, 1, dim)
        norm = np.linalg.norm(vector)
        if norm > 1e-6:
            break

    normalized_vector = vector / norm
    
    return normalized_vector


def normalize_vector(v):
    norm = np.linalg.norm(v)
    if norm == 0: 
        return v
    return v / norm


def add_noise_to_skill(skill, noise_scale, step):
    if step % 4 in [0, 1]:
        noisy_skill = skill + (noise_scale, -noise_scale)
    else:
        noisy_skill = skill + (-noise_scale, noise_scale)
    return normalize_vector(noisy_skill)


def generate_random_radius(radius_bound, dim, num_intervals, current_index = 0, eval=False):

    radius_values = np.linspace(radius_bound[0], radius_bound[1], num=num_intervals, dtype=int)
    if eval:
        radius_value = radius_values[current_index]
    else:
        radius_value = np.random.choice(radius_values)
    
    pos_enc = positional_encoding(radius_value, dim)
    return radius_value, pos_enc


def positional_encoding(R, dim):
    pos_enc = np.zeros(dim)
    position = np.arange(dim)
    div_term = np.power(10000.0, -2 * (position // 2) / dim)
    
    pos_enc[0::2] = np.sin(R * div_term[0::2])
    pos_enc[1::2] = np.cos(R * div_term[1::2])
    
    return pos_enc

def compute_cosine_weight(x, N, max_weight):
    if x <= N:
        return (1 - np.cos(torch.pi * x / N)) / 2 * max_weight
    else:
        return 1.0 * max_weight
