__credits__ = ["Rushiv Arora"]
import numpy as np
from gym import utils
from gym.envs.mujoco import mujoco_env


DEFAULT_CAMERA_CONFIG = {
    "distance": 4.0,
}


class HalfCheetahFrictionEnv(mujoco_env.MujocoEnv, utils.EzPickle):

    def __init__(
        self,
        xml_file="PATH",
        reset_noise_scale=0.01,
    ):
        utils.EzPickle.__init__(**locals())
        
        self._time = 0
        self._foot_geom_list = ["ffoot", "bfoot"]
        self._friction_schedule = [1.5, 2.0, 0.5] 
        self._current_friction = 0       
        
        self._reset_noise_scale = reset_noise_scale

        mujoco_env.MujocoEnv.__init__(self, xml_file, 5)

    def step(self, action):

        x_position_before = self.sim.data.qpos[0]
        self.do_simulation(action, self.frame_skip)
        x_position_after = self.sim.data.qpos[0]
        
        if self._time % 100 == 0:
            index = (self._time // 100) % len(self._friction_schedule)
            self._current_friction = self._friction_schedule[index]

            for name in self._foot_geom_list:
                geom_id = self.sim.model.geom_name2id(name)
                self.sim.model.geom_friction[geom_id][0] = self._current_friction

        rewards = x_position_after - x_position_before
        self._time += 1
        
        observation = self._get_obs()
        reward = rewards
        done = False

        info = {
            "x_position": x_position_after,
        }

        return observation, reward, done, info


    def _get_obs(self):
        position = self.sim.data.qpos.flat.copy()
        velocity = self.sim.data.qvel.flat.copy()

        observation = np.concatenate(([self._current_friction], position, velocity)).ravel()
        return observation
    

    def reset_model(self):
        
        self._time = 0
        
        noise_low = -self._reset_noise_scale
        noise_high = self._reset_noise_scale

        qpos = self.init_qpos + self.np_random.uniform(
            low=noise_low, high=noise_high, size=self.model.nq
        )
        qvel = (
            self.init_qvel
            + self._reset_noise_scale * self.np_random.standard_normal(self.model.nv)
        )

        self.set_state(qpos, qvel)

        observation = self._get_obs()
        return observation


    def viewer_setup(self):
        for key, value in DEFAULT_CAMERA_CONFIG.items():
            if isinstance(value, np.ndarray):
                getattr(self.viewer.cam, key)[:] = value
            else:
                setattr(self.viewer.cam, key, value)
