__credits__ = ["Rushiv Arora"]
import numpy as np
from gym import utils
from gym.envs.mujoco import mujoco_env


DEFAULT_CAMERA_CONFIG = {
    "distance": 4.0,
}


class HalfCheetahObstacleEnv(mujoco_env.MujocoEnv, utils.EzPickle):

    def __init__(
        self,
        xml_file="PATH",
        reset_noise_scale=0.01,
    ):
        utils.EzPickle.__init__(**locals())
        
        self._passed_obstacles = set()
        self._reset_noise_scale = reset_noise_scale

        mujoco_env.MujocoEnv.__init__(self, xml_file, 5)

    def step(self, action):

        x_position_before = self.sim.data.qpos[0]
        self.do_simulation(action, self.frame_skip)
        x_position_after = self.sim.data.qpos[0]

        obs_xs = []
        for i in range(self.model.ngeom):
            name = self.model.geom_id2name(i)
            if name is not None and name.startswith("obs_"):
                xpos = self.sim.data.geom_xpos[i][0] 
                obs_xs.append(xpos)
        self._obstacle_xs = sorted(obs_xs)

        obstacle_reward = 0
        for i, x in enumerate(self._obstacle_xs):
            if i not in self._passed_obstacles and x_position_before < x <= x_position_after:
                obstacle_reward += 1.0
                self._passed_obstacles.add(i)

        rewards = obstacle_reward

        observation = self._get_obs()
        reward = rewards
        done = False

        info = {
            "x_position": x_position_after,
        }

        return observation, reward, done, info


    def _get_obstacle_x_positions(self):

        obs_xs = []
        for i in range(self.model.ngeom):
            name = self.model.geom_id2name(i)
            if name is not None and name.startswith("obs_"):
                xpos = self.sim.data.geom_xpos[i][0] 
                obs_xs.append(xpos)
        return sorted(obs_xs)


    def get_distance_to_next_obstacle(self, x_pos):
        remaining_obstacles = [x for i, x in enumerate(self._obstacle_xs) if i not in self._passed_obstacles and x > x_pos]
        if not remaining_obstacles:
            return -1
        return min(remaining_obstacles) - x_pos


    def _get_obs(self):
        position = self.sim.data.qpos.flat.copy()
        velocity = self.sim.data.qvel.flat.copy()

        x_pos = self.sim.data.qpos[0]  
        dist_to_next = self.get_distance_to_next_obstacle(x_pos)

        # observation = [distance_to_next, position..., velocity...]
        observation = np.concatenate(([dist_to_next], position, velocity)).ravel()
        return observation


    def reset_model(self):
        self._obstacle_xs = self._get_obstacle_x_positions()
        self._passed_obstacles = set()
        
        noise_low = -self._reset_noise_scale
        noise_high = self._reset_noise_scale

        qpos = self.init_qpos + self.np_random.uniform(
            low=noise_low, high=noise_high, size=self.model.nq
        )
        qvel = (
            self.init_qvel
            + self._reset_noise_scale * self.np_random.standard_normal(self.model.nv)
        )

        self.set_state(qpos, qvel)

        observation = self._get_obs()
        return observation


    def viewer_setup(self):
        for key, value in DEFAULT_CAMERA_CONFIG.items():
            if isinstance(value, np.ndarray):
                getattr(self.viewer.cam, key)[:] = value
            else:
                setattr(self.viewer.cam, key, value)
