import torch
import torch.nn as nn
import torch.nn.functional as F
from torch.distributions import Normal
from torch.optim import Adam


LOG_SIG_MAX = 2
LOG_SIG_MIN = -20
epsilon = 1e-6

# Initialize Policy weights
def weights_init_(m):
    if isinstance(m, nn.Linear):
        torch.nn.init.xavier_uniform_(m.weight, gain=1)
        torch.nn.init.constant_(m.bias, 0)


# SAC encoder
class CNNSACEncoder(nn.Module):
    def __init__(self, args, input_channels=3*3):
        super(CNNSACEncoder, self).__init__()
        # 3*3 channel image (CHW*frame)
        self.conv1 = nn.Conv2d(input_channels, 1*args.cnn_depth, kernel_size=4, stride=2, padding=1)
        self.conv2 = nn.Conv2d(1*args.cnn_depth, 2*args.cnn_depth, kernel_size=4, stride=2, padding=1)
        self.conv3 = nn.Conv2d(2*args.cnn_depth, 4*args.cnn_depth, kernel_size=4, stride=2, padding=1)
        self.conv4 = nn.Conv2d(4*args.cnn_depth, 8*args.cnn_depth, kernel_size=4, stride=2, padding=1)

        self.conv_output_size = 8 * args.cnn_depth * 5 * 5

        self.apply(weights_init_)

    def forward(self, x):
        x = x / 255.

        x = F.elu(self.conv1(x))
        x = F.elu(self.conv2(x))
        x = F.elu(self.conv3(x))
        x = F.elu(self.conv4(x))

        x = x.reshape(x.size(0), -1)  # flatten while keeping batch dim

        return x


# For SAC Critic
class QNetwork(nn.Module):
    def __init__(self, input_dim, num_actions, hidden_dim, CNNencoder):
        super(QNetwork, self).__init__()

        # Q1 architecture
        self.linear1 = nn.Linear(input_dim + num_actions, hidden_dim)
        self.linear2 = nn.Linear(hidden_dim, hidden_dim)
        self.linear3 = nn.Linear(hidden_dim, 1)

        # Q2 architecture
        self.linear4 = nn.Linear(input_dim + num_actions, hidden_dim)
        self.linear5 = nn.Linear(hidden_dim, hidden_dim)
        self.linear6 = nn.Linear(hidden_dim, 1)

        # SAC encoder
        self.sac_encoder = CNNencoder

        self.apply(weights_init_)

    def forward(self, state_img, radius_input, action):

        cnn_feature = self.sac_encoder(state_img)
        xu = torch.cat([cnn_feature, radius_input, action], 1)
        
        x1 = F.relu(self.linear1(xu))
        x1 = F.relu(self.linear2(x1))
        x1 = self.linear3(x1)

        x2 = F.relu(self.linear4(xu))
        x2 = F.relu(self.linear5(x2))
        x2 = self.linear6(x2)

        return x1, x2


# For actor
class GaussianPolicy(nn.Module):
    def __init__(self, input_dim, num_actions, hidden_dim, action_space, CNNencoder):
        super(GaussianPolicy, self).__init__()
        
        self.linear1 = nn.Linear(input_dim, hidden_dim)
        self.linear2 = nn.Linear(hidden_dim, hidden_dim)

        self.mean_linear = nn.Linear(hidden_dim, num_actions)
        self.log_std_linear = nn.Linear(hidden_dim, num_actions)

        # SAC encoder
        self.sac_encoder = CNNencoder

        self.apply(weights_init_)

        # action rescaling
        if action_space is None:
            self.action_scale = torch.tensor(1.)
            self.action_bias = torch.tensor(0.)
        else:
            self.action_scale = torch.FloatTensor(
                (action_space.high - action_space.low) / 2.)
            self.action_bias = torch.FloatTensor(
                (action_space.high + action_space.low) / 2.)

    def forward(self, state_img, radius_input):

        cnn_feature = self.sac_encoder(state_img)
        state = torch.cat([cnn_feature, radius_input], 1)

        x = F.relu(self.linear1(state))
        x = F.relu(self.linear2(x))
        mean = self.mean_linear(x)
        log_std = self.log_std_linear(x)
        log_std = torch.clamp(log_std, min=LOG_SIG_MIN, max=LOG_SIG_MAX)
        return mean, log_std

    def sample(self, state_img, radius_input):
        mean, log_std = self.forward(state_img, radius_input)
        std = log_std.exp()
        normal = Normal(mean, std)
        x_t = normal.rsample()  # for reparameterization trick (mean + std * N(0,1)) , r stands for reparameterization trick in 'r'sample
        y_t = torch.tanh(x_t)
        action = y_t * self.action_scale + self.action_bias
        log_prob = normal.log_prob(x_t)
        # Enforcing Action Bound
        log_prob -= torch.log(self.action_scale * (1 - y_t.pow(2)) + epsilon)
        log_prob = log_prob.sum(1, keepdim=True) # sum of log (prob) == log (product of prob) 
        mean = torch.tanh(mean) * self.action_scale + self.action_bias
        return action, log_prob, mean

    def to(self, device):
        self.action_scale = self.action_scale.to(device)
        self.action_bias = self.action_bias.to(device)
        return super(GaussianPolicy, self).to(device)

