import os
import numpy as np
import torch
import torch.nn.functional as F
from torch.optim import Adam
from utils_sac import soft_update, hard_update
from utils_psd import onehot2radius
from utils_metra import compute_cosine_weight

from model_sac import GaussianPolicy, QNetwork, DeterministicPolicy


class SAC(object):
    def __init__(self, num_inputs, action_space, args):

        self.gamma = args.gamma
        self.tau = args.tau
        self.alpha = args.alpha


        self.policy_type = args.policy
        self.target_update_interval = args.target_update_interval
        self.automatic_entropy_tuning = args.automatic_entropy_tuning

        self.device = torch.device("cuda" if args.cuda else "cpu")

        print("\n Current device is : ", self.device, "\n")

        self.critic = QNetwork(num_inputs, action_space.shape[0], args.hidden_size).to(device=self.device)
        self.critic_target = QNetwork(num_inputs, action_space.shape[0], args.hidden_size).to(self.device)
        self.critic_optim = Adam(self.critic.parameters(), lr=args.lr)
        hard_update(self.critic_target, self.critic)

        if self.policy_type == "Gaussian":
            # Target Entropy = −dim(A) (e.g. , -6 for HalfCheetah-v2) as given in the paper
            if self.automatic_entropy_tuning is True:
                self.target_entropy = -torch.prod(torch.Tensor(action_space.shape).to(self.device)).item()
                self.log_alpha = torch.zeros(1, requires_grad=True, device=self.device)
                self.alpha_optim = Adam([self.log_alpha], lr=args.lr)

            self.policy = GaussianPolicy(num_inputs, action_space.shape[0], args.hidden_size, action_space).to(self.device)
            self.policy_optim = Adam(self.policy.parameters(), lr=args.lr)

        else:
            self.alpha = 0
            self.automatic_entropy_tuning = False
            self.policy = DeterministicPolicy(num_inputs, action_space.shape[0], args.hidden_size, action_space).to(self.device)
            self.policy_optim = Adam(self.policy.parameters(), lr=args.lr)

    def select_action(self, state, evaluate=False):
        state = torch.from_numpy(state).float().to(self.device).unsqueeze(0)
        if evaluate is False:
            action, _, _ = self.policy.sample(state)
        else:
            _, _, action = self.policy.sample(state)
        return action.detach().cpu().numpy()[0]

    def update_parameters(self, memory, psi, updates, episode_idx, args):
        # Sample a batch from memory
        state_batch, action_batch, reward_batch, radius_batch, next_state_batch, mask_batch = memory.sample(batch_size=args.batch_size)

        # Numpy to tensor
        state_batch = torch.from_numpy(state_batch).to(self.device)
        action_batch = torch.from_numpy(action_batch).to(self.device)
        next_state_batch = torch.from_numpy(next_state_batch).to(self.device)

        mask_batch = torch.from_numpy(mask_batch).to(self.device).unsqueeze(1)
        radius_batch = torch.from_numpy(radius_batch).to(self.device).unsqueeze(1)
        reward_batch = torch.from_numpy(reward_batch).to(self.device).unsqueeze(1)

        # Calculate the difference
        psi_next_state = psi.forward(next_state_batch)
        psi_state = psi.forward(state_batch)
        psi_diff = psi_next_state - psi_state

        ## PSD reward batch (length)
        V = radius_batch * torch.sin(torch.tensor(np.pi, device=radius_batch.device) / (2 * radius_batch))
        psi_norm = torch.norm(psi_diff, p=2, dim=-1).unsqueeze(1)
        L2norm = torch.norm(V - psi_norm, p=2, dim=-1)

        square_L2norm = L2norm ** 2
        pseudo_reward_batch = torch.exp(-10*square_L2norm).unsqueeze(1) # ver

        # Total reward batch
        if args.use_reward_scheduling == True:
            pseudo_reward_batch = compute_cosine_weight(episode_idx, args.saturation_episode, args.max_weight)*pseudo_reward_batch + reward_batch
        else:
            pseudo_reward_batch = args.rew_weight*pseudo_reward_batch + reward_batch

        with torch.no_grad():
            next_state_action, next_state_log_pi, _ = self.policy.sample(next_state_batch)
            qf1_next_target, qf2_next_target = self.critic_target(next_state_batch, next_state_action)
            min_qf_next_target = torch.min(qf1_next_target, qf2_next_target) - self.alpha * next_state_log_pi
            next_q_value = pseudo_reward_batch + mask_batch * self.gamma * (min_qf_next_target)
        qf1, qf2 = self.critic(state_batch, action_batch)  # Two Q-functions to mitigate positive bias in the policy improvement step
        qf1_loss = F.mse_loss(qf1, next_q_value)  # JQ = 𝔼(st,at)~D[0.5(Q1(st,at) - r(st,at) - γ(𝔼st+1~p[V(st+1)]))^2]
        qf2_loss = F.mse_loss(qf2, next_q_value)  # JQ = 𝔼(st,at)~D[0.5(Q1(st,at) - r(st,at) - γ(𝔼st+1~p[V(st+1)]))^2]
        qf_loss = qf1_loss + qf2_loss

        self.critic_optim.zero_grad()
        qf_loss.backward()
        self.critic_optim.step()

        pi, log_pi, _ = self.policy.sample(state_batch)

        qf1_pi, qf2_pi = self.critic(state_batch, pi)
        min_qf_pi = torch.min(qf1_pi, qf2_pi)

        policy_loss = ((self.alpha * log_pi) - min_qf_pi).mean() # Jπ = 𝔼st∼D,εt∼N[α * logπ(f(εt;st)|st) − Q(st,f(εt;st))]

        self.policy_optim.zero_grad()
        policy_loss.backward()
        self.policy_optim.step()

        if self.automatic_entropy_tuning:
            alpha_loss = -(self.log_alpha * (log_pi + self.target_entropy).detach()).mean()

            self.alpha_optim.zero_grad()
            alpha_loss.backward()
            self.alpha_optim.step()

            self.alpha = self.log_alpha.exp()
            alpha_tlogs = self.alpha.clone() # For TensorboardX logs
        else:
            alpha_loss = torch.tensor(0.).to(self.device)
            alpha_tlogs = torch.tensor(self.alpha) # For TensorboardX logs


        if updates % self.target_update_interval == 0:
            soft_update(self.critic_target, self.critic, self.tau)
   
        return qf1_loss.item(), qf2_loss.item(), policy_loss.item(), alpha_loss.item(), alpha_tlogs.item()








