import numpy as np
from gym.envs.mujoco import mujoco_env
from gym import utils


DEFAULT_CAMERA_CONFIG = {
    "trackbodyid": 2,
    "distance": 4.0,
    "lookat": np.array((0.0, 0.0, 1.15)),
    "elevation": -20.0,
}


class Walker2dObstacleEnv(mujoco_env.MujocoEnv, utils.EzPickle):

    def __init__(
        self,
        xml_file="PATH",
        forward_reward_weight=1.0,
        ctrl_cost_weight=1e-3,
        healthy_reward_weight = 0.1,
        healthy_reward=1.0,
        terminate_when_unhealthy = False,
        healthy_z_range=(0.8, 2.0),
        healthy_angle_range=(-1.0, 1.0),
        reset_noise_scale=5e-3,
    ):
        utils.EzPickle.__init__(**locals())

        self._forward_reward_weight = forward_reward_weight
        self._ctrl_cost_weight = ctrl_cost_weight
        self._healthy_reward_weight = healthy_reward_weight
        self._healthy_reward = healthy_reward
        self._terminate_when_unhealthy = terminate_when_unhealthy
        self._healthy_z_range = healthy_z_range
        self._healthy_angle_range = healthy_angle_range
        
        self._reset_noise_scale = reset_noise_scale
        self._passed_obstacles = set()

        mujoco_env.MujocoEnv.__init__(self, xml_file, 4)

    @property
    def healthy_reward(self):
        return (
            float(self.is_healthy or self._terminate_when_unhealthy)
            * self._healthy_reward
        )

    def control_cost(self, action):
        control_cost = self._ctrl_cost_weight * np.sum(np.square(action))
        return control_cost

    @property
    def is_healthy(self):
        z, angle = self.sim.data.qpos[1:3]

        min_z, max_z = self._healthy_z_range
        min_angle, max_angle = self._healthy_angle_range

        healthy_z = min_z < z < max_z
        healthy_angle = min_angle < angle < max_angle
        is_healthy = healthy_z and healthy_angle

        return is_healthy

    def step(self, action):
        x_position_before = self.sim.data.qpos[0]
        self.do_simulation(action, self.frame_skip)
        x_position_after = self.sim.data.qpos[0]

        obs_xs = []
        for i in range(self.model.ngeom):
            name = self.model.geom_id2name(i)
            if name is not None and name.startswith("obs_"):
                xpos = self.sim.data.geom_xpos[i][0] 
                obs_xs.append(xpos)
        self._obstacle_xs = sorted(obs_xs)

        obstacle_reward = 0
        for i, x in enumerate(self._obstacle_xs):
            if i not in self._passed_obstacles and x_position_before < x <= x_position_after:
                obstacle_reward += 1.0
                self._passed_obstacles.add(i)

        rewards = obstacle_reward

        observation = self._get_obs()
        reward = rewards
        done = False

        info = {
            "x_position": x_position_after,
        }

        return observation, reward, done, info


    def _get_obstacle_x_positions(self):

        obs_xs = []
        for i in range(self.model.ngeom):
            name = self.model.geom_id2name(i)
            if name is not None and name.startswith("obs_"):
                xpos = self.sim.data.geom_xpos[i][0] 
                obs_xs.append(xpos)
        return sorted(obs_xs)


    def get_distance_to_next_obstacle(self, x_pos):
        remaining_obstacles = [x for i, x in enumerate(self._obstacle_xs) if i not in self._passed_obstacles and x > x_pos]
        if not remaining_obstacles:
            return -1
        return min(remaining_obstacles) - x_pos

    def _get_obs(self):
        position = self.sim.data.qpos.flat.copy()
        velocity = np.clip(self.sim.data.qvel.flat.copy(), -10, 10)

        x_pos = self.sim.data.qpos[0]  
        dist_to_next = self.get_distance_to_next_obstacle(x_pos)

        observation = np.concatenate(([dist_to_next], position, velocity)).ravel()
        return observation

    def reset_model(self):

        self._obstacle_xs = self._get_obstacle_x_positions()
        self._passed_obstacles = set()
        
        noise_low = -self._reset_noise_scale
        noise_high = self._reset_noise_scale

        qpos = self.init_qpos + self.np_random.uniform(
            low=noise_low, high=noise_high, size=self.model.nq
        )
        qvel = self.init_qvel + self.np_random.uniform(
            low=noise_low, high=noise_high, size=self.model.nv
        )

        self.set_state(qpos, qvel)

        observation = self._get_obs()
        return observation

    def viewer_setup(self):
        for key, value in DEFAULT_CAMERA_CONFIG.items():
            if isinstance(value, np.ndarray):
                getattr(self.viewer.cam, key)[:] = value
            else:
                setattr(self.viewer.cam, key, value)
