# Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License").
# You may not use this file except in compliance with the License.
# A copy of the License is located at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# or in the "license" file accompanying this file. This file is distributed
# on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
# express or implied. See the License for the specific language governing
# permissions and limitations under the License.
from typing import Dict
import json
import numpy as np
import pytest

from syne_tune.optimizer.schedulers.searchers.gp_searcher_factory import \
    gp_multifidelity_searcher_defaults, gp_multifidelity_searcher_factory
from syne_tune.optimizer.schedulers.searchers.utils.default_arguments \
    import check_and_merge_defaults
from syne_tune.optimizer.schedulers.searchers.gp_searcher_utils import \
    decode_state_from_old_encoding
from syne_tune.search_space import randint, uniform, loguniform
from syne_tune.optimizer.schedulers.searchers.bayesopt.datatypes.tuning_job_state \
    import TuningJobState


def _common_kwargs(configspace: Dict) -> Dict:
    return {
        'configspace': configspace,
        'max_epochs': configspace['epochs'],
        'metric': 'accuracy',
        'resource_attr': 'epoch',
        'scheduler': 'hyperband_stopping',
        'scheduler_mode': 'max',
        'debug_log': False,
        'normalize_targets': True,
    }


def build_gp_model_factory(configspace: Dict, model_params: Dict) -> Dict:
    kwargs = dict(
        _common_kwargs(configspace),
        model='gp_multitask',
        gp_resource_kernel='freeze-thaw')
    _kwargs = check_and_merge_defaults(
        kwargs, *gp_multifidelity_searcher_defaults(),
        dict_name='search_options')
    kwargs_int = gp_multifidelity_searcher_factory(**_kwargs)
    kwargs_int['model_factory'].set_params(model_params)
    return kwargs_int


def _convert_keys(dict_old, dict_new, pref_old, pref_new):
    for k, v in dict_old.items():
        if k.startswith(pref_old):
            k_new = pref_new + k[len(pref_old):]
            dict_new[k_new] = v


def _convert_model_params(model_params: Dict) -> Dict:
    """
    Convert from `GaussianProcessRegression` to
    `GaussianProcessLearningCurveModel` parameters.

    :param model_params:
    :return:
    """
    new_model_params = {'noise_variance': model_params['noise_variance']}
    _convert_keys(model_params, new_model_params, 'kernel_kernelx', 'kernel')
    _convert_keys(model_params, new_model_params, 'kernel_meanx', 'mean')
    for tag in ['alpha', 'mean_lam', 'gamma']:
        new_model_params['expdecay_' + tag] = model_params['kernel_' + tag]
    return new_model_params


def build_gped_model_factory(
        configspace: Dict, model_params: Dict, **kwargs):
    kwargs = dict(
        _common_kwargs(configspace),
        model='gp_expdecay',
        expdecay_normalize_inputs=True,
        **kwargs)
    _kwargs = check_and_merge_defaults(
        kwargs, *gp_multifidelity_searcher_defaults(),
        dict_name='search_options')
    kwargs_int = gp_multifidelity_searcher_factory(**_kwargs)
    # Need to convert `model_params`
    kwargs_int['model_factory'].set_params(
        _convert_model_params(model_params))
    return kwargs_int


# We ran launch_sample_searcher_states.py to sample the searcher states
# used here, which runs MOBSTER (hyperband_stopping, bayesopt) with the
# mlp_fashionmnist_benchmark

_model_params = []
_state = []

# elapsed_time = 510.4632730484009
# num_observations = 80
# num_configs = 14
_model_params.append('{"noise_variance": 0.0003465269369921202, "kernel_alpha": 0.32006841518063855, "kernel_mean_lam": 23.993482465964025, "kernel_gamma": 1.0, "kernel_kernelx_inv_bw0": 0.00010000000000000009, "kernel_kernelx_inv_bw1": 1.83640681099597, "kernel_kernelx_inv_bw2": 0.0022959259096544463, "kernel_kernelx_inv_bw3": 3.583941260907946, "kernel_kernelx_inv_bw4": 1.1390978939096217, "kernel_kernelx_inv_bw5": 0.0005059210120066444, "kernel_kernelx_inv_bw6": 0.00010000000000000009, "kernel_kernelx_covariance_scale": 2.208213263615384, "kernel_meanx_mean_value": 1.2132901253498967}')
_state.append('{"candidate_evaluations": [{"candidate": {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607}, "metrics": {"cost_metric": {"1": 15.282694101333618, "2": 32.215107917785645, "3": 48.62396502494812, "4": 62.46452808380127, "5": 77.36029410362244, "6": 91.85972809791565, "7": 107.89106607437134, "8": 122.8071219921112, "9": 138.6427869796753, "10": 155.3775441646576, "11": 171.86729407310486, "12": 186.69335007667542, "13": 204.0226011276245, "14": 219.01899409294128, "15": 234.25396490097046, "16": 249.26809406280518, "17": 264.3408589363098, "18": 280.09520292282104, "19": 296.0671851634979, "20": 319.6674599647522, "21": 337.2237899303436, "22": 354.650151014328, "23": 370.17603182792664, "24": 386.20747780799866, "25": 401.40604305267334, "26": 415.9903359413147, "27": 431.2174470424652, "28": 445.8420739173889, "29": 460.1389670372009, "30": 475.33422088623047, "31": 493.9087038040161, "32": 509.9310200214386, "33": 525.421621799469, "34": 540.1411230564117, "35": 554.5012362003326, "36": 569.0483500957489, "37": 583.1704621315002}, "active_metric": {"1": 0.41328783621035725, "2": 0.3706342834203131, "3": 0.351264552388599, "4": 0.33500602167804094, "5": 0.32707747892412686, "6": 0.3204536330790847, "7": 0.3140305098354075, "8": 0.2992773986350863, "9": 0.2865315134484143, "10": 0.27509032517061416, "11": 0.26525491770373344, "12": 0.2552187876354878, "13": 0.24989963869931753, "14": 0.24197109594540345, "15": 0.2352468887996788, "16": 0.22902448815736653, "17": 0.2233038940184665, "18": 0.2191890806904857, "19": 0.2133681252509032, "20": 0.21045764753111196, "21": 0.20503813729425935, "22": 0.20373344038538743, "23": 0.2004215174628663, "24": 0.1956041750301084, "25": 0.19299478121236446, "26": 0.19199116820553996, "27": 0.18908069048574871, "28": 0.18667201926936972, "29": 0.18526696105981533, "30": 0.18426334805299072, "31": 0.18406262545162588, "32": 0.18255720594138902, "33": 0.18145323163388194, "34": 0.18105178643115216, "35": 0.1785427539140907, "36": 0.1785427539140907, "37": 0.17723805700521877}}}, {"candidate": {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07}, "metrics": {"cost_metric": {"1": 22.334229230880737, "2": 47.57227420806885, "3": 68.38872504234314, "4": 91.17699599266052, "5": 114.11172008514404, "6": 137.2291030883789, "7": 161.9407799243927, "8": 186.01491808891296, "9": 210.73575019836426, "10": 233.72683000564575, "11": 256.1681180000305, "12": 279.3303301334381, "13": 311.3050580024719, "14": 336.1765010356903, "15": 362.16334199905396, "16": 385.8361783027649, "17": 408.6689350605011, "18": 430.98025918006897, "19": 453.24402713775635, "20": 474.99853110313416, "21": 501.653440952301, "22": 525.0653901100159, "23": 547.2747151851654, "24": 569.0442740917206, "25": 590.4288313388824}, "active_metric": {"1": 0.1725761217948718, "2": 0.15024038461538458, "3": 0.1353165064102564, "4": 0.13842147435897434, "5": 0.1303084935897436, "6": 0.12239583333333337, "7": 0.12089342948717952, "8": 0.11738782051282048, "9": 0.11909054487179482, "10": 0.11498397435897434, "11": 0.11618589743589747, "12": 0.11388221153846156, "13": 0.11087740384615385, "14": 0.11247996794871795, "15": 0.11227964743589747, "16": 0.11037660256410253, "17": 0.11237980769230771, "18": 0.10917467948717952, "19": 0.10506810897435892, "20": 0.10947516025641024, "21": 0.10627003205128205, "22": 0.10366586538461542, "23": 0.10356570512820518, "24": 0.10837339743589747, "25": 0.10406650641025639}}}, {"candidate": {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05}, "metrics": {"cost_metric": {"1": 23.648607969284058, "2": 50.783905029296875, "3": 94.63147592544556}, "active_metric": {"1": 0.16546618647458988, "2": 0.1404561824729892, "3": 0.1386554621848739}}}, {"candidate": {"n_units_1": 774, "n_units_2": 917, "batch_size": 29, "dropout_1": 0.7778923725289609, "dropout_2": 0.7413003050986398, "learning_rate": 6.472832341968678e-05, "wd": 0.0007744951242384949}, "metrics": {"cost_metric": {"1": 50.06452512741089}, "active_metric": {"1": 0.22955092221331197}}}, {"candidate": {"n_units_1": 91, "n_units_2": 459, "batch_size": 105, "dropout_1": 0.48639033141890325, "dropout_2": 0.21324913218446714, "learning_rate": 0.00013769715715418189, "wd": 0.02017249366944585}, "metrics": {"cost_metric": {"1": 13.569249868392944}, "active_metric": {"1": 0.3023558897243108}}}, {"candidate": {"n_units_1": 673, "n_units_2": 262, "batch_size": 78, "dropout_1": 0.9510740133913004, "dropout_2": 0.3263851441475057, "learning_rate": 0.009715536539110267, "wd": 0.0002984576239921338}, "metrics": {"cost_metric": {"1": 22.84876799583435}, "active_metric": {"1": 0.7708333333333334}}}, {"candidate": {"n_units_1": 29, "n_units_2": 584, "batch_size": 41, "dropout_1": 0.6843134351325143, "dropout_2": 0.20866634010808638, "learning_rate": 0.0007531586746050282, "wd": 8.542702035112525e-08}, "metrics": {"cost_metric": {"1": 16.058557987213135}, "active_metric": {"1": 0.4832881662149955}}}, {"candidate": {"n_units_1": 529, "n_units_2": 660, "batch_size": 30, "dropout_1": 0.5490868842845377, "dropout_2": 0.7920401266415242, "learning_rate": 2.101654064946448e-06, "wd": 9.568417280766271e-08}, "metrics": {"cost_metric": {"1": 33.76100707054138}, "active_metric": {"1": 0.47597597597597596}}}, {"candidate": {"n_units_1": 828, "n_units_2": 634, "batch_size": 123, "dropout_1": 0.4257740273137951, "dropout_2": 0.23113551626017848, "learning_rate": 6.860930897860399e-06, "wd": 8.101814548564974e-08}, "metrics": {"cost_metric": {"1": 27.756996154785156}, "active_metric": {"1": 0.33835190203753884}}}, {"candidate": {"n_units_1": 635, "n_units_2": 538, "batch_size": 69, "dropout_1": 0.45999796301051993, "dropout_2": 0.1659629767457713, "learning_rate": 0.00045244429498637726, "wd": 7.074070723442468e-06}, "metrics": {"cost_metric": {"1": 26.10046887397766, "2": 53.824394941329956, "3": 80.07230305671692, "4": 106.25571393966675, "5": 133.94080090522766, "6": 170.4485969543457, "7": 200.93623304367065, "8": 229.56330180168152, "9": 257.11693382263184, "10": 286.6725299358368, "11": 319.0151889324188, "12": 354.14807295799255, "13": 386.9317948818207, "14": 419.8007171154022}, "active_metric": {"1": 0.14744363929146542, "2": 0.13586956521739135, "3": 0.12983091787439616, "4": 0.12661030595813205, "5": 0.1279186795491143, "6": 0.12328904991948475, "7": 0.11674718196457323, "8": 0.1126207729468599, "9": 0.11292270531400961, "10": 0.11513687600644118, "11": 0.11584138486312401, "12": 0.11101046698872785, "13": 0.1090982286634461, "14": 0.10738727858293073}}}, {"candidate": {"n_units_1": 931, "n_units_2": 527, "batch_size": 20, "dropout_1": 0.41104972550281815, "dropout_2": 0.7568250476630992, "learning_rate": 0.8320700517813453, "wd": 1.9853072351534312e-08}, "metrics": {"cost_metric": {"1": 117.45865964889526}, "active_metric": {"1": 0.9045}}}, {"candidate": {"n_units_1": 576, "n_units_2": 265, "batch_size": 68, "dropout_1": 0.3230142899473453, "dropout_2": 0.3493653937208789, "learning_rate": 0.0004443989879229176, "wd": 0.012741750252890758}, "metrics": {"cost_metric": {"1": 23.032225131988525, "2": 53.855369091033936, "3": 78.89936113357544}, "active_metric": {"1": 0.18497398959583833, "2": 0.1910764305722289, "3": 0.17987194877951185}}}, {"candidate": {"n_units_1": 606, "n_units_2": 635, "batch_size": 36, "dropout_1": 0.43630659905530433, "dropout_2": 0.5387515572964027, "learning_rate": 0.0002641791837664535, "wd": 3.148533916384597e-05}, "metrics": {"cost_metric": {"1": 33.83635210990906, "2": 66.65014815330505, "3": 99.47553634643555}, "active_metric": {"1": 0.1605495387083835, "2": 0.1470116325711993, "3": 0.1400922583233053}}}, {"candidate": {"n_units_1": 534, "n_units_2": 1014, "batch_size": 35, "dropout_1": 0.40293710200390886, "dropout_2": 0.2615463742465658, "learning_rate": 0.0004610005312393892, "wd": 5.324412228404534e-07}, "metrics": {"cost_metric": {"1": 40.18646717071533, "2": 77.50762224197388, "3": 113.17754912376404}, "active_metric": {"1": 0.1589974937343358, "2": 0.1334335839598998, "3": 0.127218045112782}}}], "failed_candidates": [], "pending_candidates": [{"n_units_1": 635, "n_units_2": 538, "batch_size": 69, "dropout_1": 0.45999796301051993, "dropout_2": 0.1659629767457713, "learning_rate": 0.00045244429498637726, "wd": 7.074070723442468e-06, "RESOURCE_ATTR_epoch": 12}, {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607, "RESOURCE_ATTR_epoch": 32}, {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07, "RESOURCE_ATTR_epoch": 22}, {"n_units_1": 534, "n_units_2": 1014, "batch_size": 35, "dropout_1": 0.40293710200390886, "dropout_2": 0.2615463742465658, "learning_rate": 0.0004610005312393892, "wd": 5.324412228404534e-07, "RESOURCE_ATTR_epoch": 2}]}')

# elapsed_time = 266.8967089653015
# num_observations = 40
# num_configs = 11
_model_params.append('{"noise_variance": 0.0013634444543287733, "kernel_alpha": 0.33215948811021134, "kernel_mean_lam": 15.077221388452632, "kernel_gamma": 1.0, "kernel_kernelx_inv_bw0": 0.0025264221960287574, "kernel_kernelx_inv_bw1": 4.586734084695334, "kernel_kernelx_inv_bw2": 0.003154109460037928, "kernel_kernelx_inv_bw3": 0.000894766660821803, "kernel_kernelx_inv_bw4": 0.005214180628382712, "kernel_kernelx_inv_bw5": 10.093642962950414, "kernel_kernelx_inv_bw6": 0.016153252870882064, "kernel_kernelx_covariance_scale": 1.2311611100489817, "kernel_meanx_mean_value": 0.09552280164740308}')
_state.append('{"candidate_evaluations": [{"candidate": {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607}, "metrics": {"cost_metric": {"1": 15.282694101333618, "2": 32.215107917785645, "3": 48.62396502494812, "4": 62.46452808380127, "5": 77.36029410362244, "6": 91.85972809791565, "7": 107.89106607437134, "8": 122.8071219921112, "9": 138.6427869796753, "10": 155.3775441646576, "11": 171.86729407310486, "12": 186.69335007667542, "13": 204.0226011276245, "14": 219.01899409294128, "15": 234.25396490097046, "16": 249.26809406280518, "17": 264.3408589363098, "18": 280.09520292282104, "19": 296.0671851634979, "20": 319.6674599647522, "21": 337.2237899303436, "22": 354.650151014328, "23": 370.17603182792664, "24": 386.20747780799866, "25": 401.40604305267334, "26": 415.9903359413147, "27": 431.2174470424652, "28": 445.8420739173889, "29": 460.1389670372009, "30": 475.33422088623047, "31": 493.9087038040161, "32": 509.9310200214386, "33": 525.421621799469, "34": 540.1411230564117, "35": 554.5012362003326, "36": 569.0483500957489, "37": 583.1704621315002}, "active_metric": {"1": 0.41328783621035725, "2": 0.3706342834203131, "3": 0.351264552388599, "4": 0.33500602167804094, "5": 0.32707747892412686, "6": 0.3204536330790847, "7": 0.3140305098354075, "8": 0.2992773986350863, "9": 0.2865315134484143, "10": 0.27509032517061416, "11": 0.26525491770373344, "12": 0.2552187876354878, "13": 0.24989963869931753, "14": 0.24197109594540345, "15": 0.2352468887996788, "16": 0.22902448815736653, "17": 0.2233038940184665, "18": 0.2191890806904857, "19": 0.2133681252509032, "20": 0.21045764753111196, "21": 0.20503813729425935, "22": 0.20373344038538743, "23": 0.2004215174628663, "24": 0.1956041750301084, "25": 0.19299478121236446, "26": 0.19199116820553996, "27": 0.18908069048574871, "28": 0.18667201926936972, "29": 0.18526696105981533, "30": 0.18426334805299072, "31": 0.18406262545162588, "32": 0.18255720594138902, "33": 0.18145323163388194, "34": 0.18105178643115216, "35": 0.1785427539140907, "36": 0.1785427539140907, "37": 0.17723805700521877}}}, {"candidate": {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07}, "metrics": {"cost_metric": {"1": 22.334229230880737, "2": 47.57227420806885, "3": 68.38872504234314, "4": 91.17699599266052, "5": 114.11172008514404, "6": 137.2291030883789, "7": 161.9407799243927, "8": 186.01491808891296, "9": 210.73575019836426, "10": 233.72683000564575, "11": 256.1681180000305, "12": 279.3303301334381, "13": 311.3050580024719, "14": 336.1765010356903, "15": 362.16334199905396, "16": 385.8361783027649, "17": 408.6689350605011, "18": 430.98025918006897, "19": 453.24402713775635, "20": 474.99853110313416, "21": 501.653440952301, "22": 525.0653901100159, "23": 547.2747151851654, "24": 569.0442740917206, "25": 590.4288313388824}, "active_metric": {"1": 0.1725761217948718, "2": 0.15024038461538458, "3": 0.1353165064102564, "4": 0.13842147435897434, "5": 0.1303084935897436, "6": 0.12239583333333337, "7": 0.12089342948717952, "8": 0.11738782051282048, "9": 0.11909054487179482, "10": 0.11498397435897434, "11": 0.11618589743589747, "12": 0.11388221153846156, "13": 0.11087740384615385, "14": 0.11247996794871795, "15": 0.11227964743589747, "16": 0.11037660256410253, "17": 0.11237980769230771, "18": 0.10917467948717952, "19": 0.10506810897435892, "20": 0.10947516025641024, "21": 0.10627003205128205, "22": 0.10366586538461542, "23": 0.10356570512820518, "24": 0.10837339743589747, "25": 0.10406650641025639}}}, {"candidate": {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05}, "metrics": {"cost_metric": {"1": 23.648607969284058, "2": 50.783905029296875, "3": 94.63147592544556}, "active_metric": {"1": 0.16546618647458988, "2": 0.1404561824729892, "3": 0.1386554621848739}}}, {"candidate": {"n_units_1": 774, "n_units_2": 917, "batch_size": 29, "dropout_1": 0.7778923725289609, "dropout_2": 0.7413003050986398, "learning_rate": 6.472832341968678e-05, "wd": 0.0007744951242384949}, "metrics": {"cost_metric": {"1": 50.06452512741089}, "active_metric": {"1": 0.22955092221331197}}}, {"candidate": {"n_units_1": 91, "n_units_2": 459, "batch_size": 105, "dropout_1": 0.48639033141890325, "dropout_2": 0.21324913218446714, "learning_rate": 0.00013769715715418189, "wd": 0.02017249366944585}, "metrics": {"cost_metric": {"1": 13.569249868392944}, "active_metric": {"1": 0.3023558897243108}}}, {"candidate": {"n_units_1": 673, "n_units_2": 262, "batch_size": 78, "dropout_1": 0.9510740133913004, "dropout_2": 0.3263851441475057, "learning_rate": 0.009715536539110267, "wd": 0.0002984576239921338}, "metrics": {"cost_metric": {"1": 22.84876799583435}, "active_metric": {"1": 0.7708333333333334}}}, {"candidate": {"n_units_1": 29, "n_units_2": 584, "batch_size": 41, "dropout_1": 0.6843134351325143, "dropout_2": 0.20866634010808638, "learning_rate": 0.0007531586746050282, "wd": 8.542702035112525e-08}, "metrics": {"cost_metric": {"1": 16.058557987213135}, "active_metric": {"1": 0.4832881662149955}}}, {"candidate": {"n_units_1": 529, "n_units_2": 660, "batch_size": 30, "dropout_1": 0.5490868842845377, "dropout_2": 0.7920401266415242, "learning_rate": 2.101654064946448e-06, "wd": 9.568417280766271e-08}, "metrics": {"cost_metric": {"1": 33.76100707054138}, "active_metric": {"1": 0.47597597597597596}}}, {"candidate": {"n_units_1": 828, "n_units_2": 634, "batch_size": 123, "dropout_1": 0.4257740273137951, "dropout_2": 0.23113551626017848, "learning_rate": 6.860930897860399e-06, "wd": 8.101814548564974e-08}, "metrics": {"cost_metric": {"1": 27.756996154785156}, "active_metric": {"1": 0.33835190203753884}}}, {"candidate": {"n_units_1": 635, "n_units_2": 538, "batch_size": 69, "dropout_1": 0.45999796301051993, "dropout_2": 0.1659629767457713, "learning_rate": 0.00045244429498637726, "wd": 7.074070723442468e-06}, "metrics": {"cost_metric": {"1": 26.10046887397766, "2": 53.824394941329956, "3": 80.07230305671692, "4": 106.25571393966675, "5": 133.94080090522766, "6": 170.4485969543457, "7": 200.93623304367065, "8": 229.56330180168152, "9": 257.11693382263184, "10": 286.6725299358368, "11": 319.0151889324188, "12": 354.14807295799255, "13": 386.9317948818207, "14": 419.8007171154022}, "active_metric": {"1": 0.14744363929146542, "2": 0.13586956521739135, "3": 0.12983091787439616, "4": 0.12661030595813205, "5": 0.1279186795491143, "6": 0.12328904991948475, "7": 0.11674718196457323, "8": 0.1126207729468599, "9": 0.11292270531400961, "10": 0.11513687600644118, "11": 0.11584138486312401, "12": 0.11101046698872785, "13": 0.1090982286634461, "14": 0.10738727858293073}}}, {"candidate": {"n_units_1": 931, "n_units_2": 527, "batch_size": 20, "dropout_1": 0.41104972550281815, "dropout_2": 0.7568250476630992, "learning_rate": 0.8320700517813453, "wd": 1.9853072351534312e-08}, "metrics": {"cost_metric": {"1": 117.45865964889526}, "active_metric": {"1": 0.9045}}}], "failed_candidates": [], "pending_candidates": [{"n_units_1": 635, "n_units_2": 538, "batch_size": 69, "dropout_1": 0.45999796301051993, "dropout_2": 0.1659629767457713, "learning_rate": 0.00045244429498637726, "wd": 7.074070723442468e-06, "RESOURCE_ATTR_epoch": 4}, {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607, "RESOURCE_ATTR_epoch": 17}, {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07, "RESOURCE_ATTR_epoch": 12}]}')

# elapsed_time = 164.4495289325714
# num_observations = 25
# num_configs = 9
_model_params.append('{"noise_variance": 0.000499818813466635, "kernel_alpha": 0.23713455205952677, "kernel_mean_lam": 45.09309080404886, "kernel_gamma": 0.9458802138724483, "kernel_kernelx_inv_bw0": 0.047224584489824036, "kernel_kernelx_inv_bw1": 5.4925599066759005, "kernel_kernelx_inv_bw2": 0.0013332553417099433, "kernel_kernelx_inv_bw3": 0.0065960016396222725, "kernel_kernelx_inv_bw4": 0.020133444037227638, "kernel_kernelx_inv_bw5": 10.716957151297681, "kernel_kernelx_inv_bw6": 0.003199027133464099, "kernel_kernelx_covariance_scale": 1.0755903136336264, "kernel_meanx_mean_value": 0.13675161355207244}')
_state.append('{"candidate_evaluations": [{"candidate": {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607}, "metrics": {"cost_metric": {"1": 15.282694101333618, "2": 32.215107917785645, "3": 48.62396502494812, "4": 62.46452808380127, "5": 77.36029410362244, "6": 91.85972809791565, "7": 107.89106607437134, "8": 122.8071219921112, "9": 138.6427869796753, "10": 155.3775441646576, "11": 171.86729407310486, "12": 186.69335007667542, "13": 204.0226011276245, "14": 219.01899409294128, "15": 234.25396490097046, "16": 249.26809406280518, "17": 264.3408589363098, "18": 280.09520292282104, "19": 296.0671851634979, "20": 319.6674599647522, "21": 337.2237899303436, "22": 354.650151014328, "23": 370.17603182792664, "24": 386.20747780799866, "25": 401.40604305267334, "26": 415.9903359413147, "27": 431.2174470424652, "28": 445.8420739173889, "29": 460.1389670372009, "30": 475.33422088623047, "31": 493.9087038040161, "32": 509.9310200214386, "33": 525.421621799469, "34": 540.1411230564117, "35": 554.5012362003326, "36": 569.0483500957489, "37": 583.1704621315002}, "active_metric": {"1": 0.41328783621035725, "2": 0.3706342834203131, "3": 0.351264552388599, "4": 0.33500602167804094, "5": 0.32707747892412686, "6": 0.3204536330790847, "7": 0.3140305098354075, "8": 0.2992773986350863, "9": 0.2865315134484143, "10": 0.27509032517061416, "11": 0.26525491770373344, "12": 0.2552187876354878, "13": 0.24989963869931753, "14": 0.24197109594540345, "15": 0.2352468887996788, "16": 0.22902448815736653, "17": 0.2233038940184665, "18": 0.2191890806904857, "19": 0.2133681252509032, "20": 0.21045764753111196, "21": 0.20503813729425935, "22": 0.20373344038538743, "23": 0.2004215174628663, "24": 0.1956041750301084, "25": 0.19299478121236446, "26": 0.19199116820553996, "27": 0.18908069048574871, "28": 0.18667201926936972, "29": 0.18526696105981533, "30": 0.18426334805299072, "31": 0.18406262545162588, "32": 0.18255720594138902, "33": 0.18145323163388194, "34": 0.18105178643115216, "35": 0.1785427539140907, "36": 0.1785427539140907, "37": 0.17723805700521877}}}, {"candidate": {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07}, "metrics": {"cost_metric": {"1": 22.334229230880737, "2": 47.57227420806885, "3": 68.38872504234314, "4": 91.17699599266052, "5": 114.11172008514404, "6": 137.2291030883789, "7": 161.9407799243927, "8": 186.01491808891296, "9": 210.73575019836426, "10": 233.72683000564575, "11": 256.1681180000305, "12": 279.3303301334381, "13": 311.3050580024719, "14": 336.1765010356903, "15": 362.16334199905396, "16": 385.8361783027649, "17": 408.6689350605011, "18": 430.98025918006897, "19": 453.24402713775635, "20": 474.99853110313416, "21": 501.653440952301, "22": 525.0653901100159, "23": 547.2747151851654, "24": 569.0442740917206, "25": 590.4288313388824}, "active_metric": {"1": 0.1725761217948718, "2": 0.15024038461538458, "3": 0.1353165064102564, "4": 0.13842147435897434, "5": 0.1303084935897436, "6": 0.12239583333333337, "7": 0.12089342948717952, "8": 0.11738782051282048, "9": 0.11909054487179482, "10": 0.11498397435897434, "11": 0.11618589743589747, "12": 0.11388221153846156, "13": 0.11087740384615385, "14": 0.11247996794871795, "15": 0.11227964743589747, "16": 0.11037660256410253, "17": 0.11237980769230771, "18": 0.10917467948717952, "19": 0.10506810897435892, "20": 0.10947516025641024, "21": 0.10627003205128205, "22": 0.10366586538461542, "23": 0.10356570512820518, "24": 0.10837339743589747, "25": 0.10406650641025639}}}, {"candidate": {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05}, "metrics": {"cost_metric": {"1": 23.648607969284058, "2": 50.783905029296875, "3": 94.63147592544556}, "active_metric": {"1": 0.16546618647458988, "2": 0.1404561824729892, "3": 0.1386554621848739}}}, {"candidate": {"n_units_1": 774, "n_units_2": 917, "batch_size": 29, "dropout_1": 0.7778923725289609, "dropout_2": 0.7413003050986398, "learning_rate": 6.472832341968678e-05, "wd": 0.0007744951242384949}, "metrics": {"cost_metric": {"1": 50.06452512741089}, "active_metric": {"1": 0.22955092221331197}}}, {"candidate": {"n_units_1": 91, "n_units_2": 459, "batch_size": 105, "dropout_1": 0.48639033141890325, "dropout_2": 0.21324913218446714, "learning_rate": 0.00013769715715418189, "wd": 0.02017249366944585}, "metrics": {"cost_metric": {"1": 13.569249868392944}, "active_metric": {"1": 0.3023558897243108}}}, {"candidate": {"n_units_1": 673, "n_units_2": 262, "batch_size": 78, "dropout_1": 0.9510740133913004, "dropout_2": 0.3263851441475057, "learning_rate": 0.009715536539110267, "wd": 0.0002984576239921338}, "metrics": {"cost_metric": {"1": 22.84876799583435}, "active_metric": {"1": 0.7708333333333334}}}, {"candidate": {"n_units_1": 29, "n_units_2": 584, "batch_size": 41, "dropout_1": 0.6843134351325143, "dropout_2": 0.20866634010808638, "learning_rate": 0.0007531586746050282, "wd": 8.542702035112525e-08}, "metrics": {"cost_metric": {"1": 16.058557987213135}, "active_metric": {"1": 0.4832881662149955}}}, {"candidate": {"n_units_1": 529, "n_units_2": 660, "batch_size": 30, "dropout_1": 0.5490868842845377, "dropout_2": 0.7920401266415242, "learning_rate": 2.101654064946448e-06, "wd": 9.568417280766271e-08}, "metrics": {"cost_metric": {"1": 33.76100707054138}, "active_metric": {"1": 0.47597597597597596}}}, {"candidate": {"n_units_1": 828, "n_units_2": 634, "batch_size": 123, "dropout_1": 0.4257740273137951, "dropout_2": 0.23113551626017848, "learning_rate": 6.860930897860399e-06, "wd": 8.101814548564974e-08}, "metrics": {"cost_metric": {"1": 27.756996154785156}, "active_metric": {"1": 0.33835190203753884}}}], "failed_candidates": [], "pending_candidates": [{"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07, "RESOURCE_ATTR_epoch": 7}, {"batch_size": 20, "dropout_1": 0.41104972550281815, "dropout_2": 0.7568250476630992, "learning_rate": 0.8320700517813453, "n_units_1": 931, "n_units_2": 527, "wd": 1.9853072351534312e-08, "RESOURCE_ATTR_epoch": 1}, {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607, "RESOURCE_ATTR_epoch": 11}]}')

# elapsed_time = 130.01123476028442
# num_observations = 19
# num_configs = 7
_model_params.append('{"noise_variance": 0.00024460591274526826, "kernel_alpha": 0.48004176199483545, "kernel_mean_lam": 49.99999999999999, "kernel_gamma": 0.734348408988702, "kernel_kernelx_inv_bw0": 0.024364546709466313, "kernel_kernelx_inv_bw1": 2.4581694176756876, "kernel_kernelx_inv_bw2": 0.00025864436451015127, "kernel_kernelx_inv_bw3": 0.00010000000000000009, "kernel_kernelx_inv_bw4": 0.0001075879781512719, "kernel_kernelx_inv_bw5": 7.628312984876062, "kernel_kernelx_inv_bw6": 0.0009132144114327569, "kernel_kernelx_covariance_scale": 1.1661945422479474, "kernel_meanx_mean_value": 0.06308935787869324}')
_state.append('{"candidate_evaluations": [{"candidate": {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607}, "metrics": {"cost_metric": {"1": 15.282694101333618, "2": 32.215107917785645, "3": 48.62396502494812, "4": 62.46452808380127, "5": 77.36029410362244, "6": 91.85972809791565, "7": 107.89106607437134, "8": 122.8071219921112, "9": 138.6427869796753, "10": 155.3775441646576, "11": 171.86729407310486, "12": 186.69335007667542, "13": 204.0226011276245, "14": 219.01899409294128, "15": 234.25396490097046, "16": 249.26809406280518, "17": 264.3408589363098, "18": 280.09520292282104, "19": 296.0671851634979, "20": 319.6674599647522, "21": 337.2237899303436, "22": 354.650151014328, "23": 370.17603182792664, "24": 386.20747780799866, "25": 401.40604305267334, "26": 415.9903359413147, "27": 431.2174470424652, "28": 445.8420739173889, "29": 460.1389670372009, "30": 475.33422088623047, "31": 493.9087038040161, "32": 509.9310200214386, "33": 525.421621799469, "34": 540.1411230564117, "35": 554.5012362003326, "36": 569.0483500957489, "37": 583.1704621315002}, "active_metric": {"1": 0.41328783621035725, "2": 0.3706342834203131, "3": 0.351264552388599, "4": 0.33500602167804094, "5": 0.32707747892412686, "6": 0.3204536330790847, "7": 0.3140305098354075, "8": 0.2992773986350863, "9": 0.2865315134484143, "10": 0.27509032517061416, "11": 0.26525491770373344, "12": 0.2552187876354878, "13": 0.24989963869931753, "14": 0.24197109594540345, "15": 0.2352468887996788, "16": 0.22902448815736653, "17": 0.2233038940184665, "18": 0.2191890806904857, "19": 0.2133681252509032, "20": 0.21045764753111196, "21": 0.20503813729425935, "22": 0.20373344038538743, "23": 0.2004215174628663, "24": 0.1956041750301084, "25": 0.19299478121236446, "26": 0.19199116820553996, "27": 0.18908069048574871, "28": 0.18667201926936972, "29": 0.18526696105981533, "30": 0.18426334805299072, "31": 0.18406262545162588, "32": 0.18255720594138902, "33": 0.18145323163388194, "34": 0.18105178643115216, "35": 0.1785427539140907, "36": 0.1785427539140907, "37": 0.17723805700521877}}}, {"candidate": {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07}, "metrics": {"cost_metric": {"1": 22.334229230880737, "2": 47.57227420806885, "3": 68.38872504234314, "4": 91.17699599266052, "5": 114.11172008514404, "6": 137.2291030883789, "7": 161.9407799243927, "8": 186.01491808891296, "9": 210.73575019836426, "10": 233.72683000564575, "11": 256.1681180000305, "12": 279.3303301334381, "13": 311.3050580024719, "14": 336.1765010356903, "15": 362.16334199905396, "16": 385.8361783027649, "17": 408.6689350605011, "18": 430.98025918006897, "19": 453.24402713775635, "20": 474.99853110313416, "21": 501.653440952301, "22": 525.0653901100159, "23": 547.2747151851654, "24": 569.0442740917206, "25": 590.4288313388824}, "active_metric": {"1": 0.1725761217948718, "2": 0.15024038461538458, "3": 0.1353165064102564, "4": 0.13842147435897434, "5": 0.1303084935897436, "6": 0.12239583333333337, "7": 0.12089342948717952, "8": 0.11738782051282048, "9": 0.11909054487179482, "10": 0.11498397435897434, "11": 0.11618589743589747, "12": 0.11388221153846156, "13": 0.11087740384615385, "14": 0.11247996794871795, "15": 0.11227964743589747, "16": 0.11037660256410253, "17": 0.11237980769230771, "18": 0.10917467948717952, "19": 0.10506810897435892, "20": 0.10947516025641024, "21": 0.10627003205128205, "22": 0.10366586538461542, "23": 0.10356570512820518, "24": 0.10837339743589747, "25": 0.10406650641025639}}}, {"candidate": {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05}, "metrics": {"cost_metric": {"1": 23.648607969284058, "2": 50.783905029296875, "3": 94.63147592544556}, "active_metric": {"1": 0.16546618647458988, "2": 0.1404561824729892, "3": 0.1386554621848739}}}, {"candidate": {"n_units_1": 774, "n_units_2": 917, "batch_size": 29, "dropout_1": 0.7778923725289609, "dropout_2": 0.7413003050986398, "learning_rate": 6.472832341968678e-05, "wd": 0.0007744951242384949}, "metrics": {"cost_metric": {"1": 50.06452512741089}, "active_metric": {"1": 0.22955092221331197}}}, {"candidate": {"n_units_1": 91, "n_units_2": 459, "batch_size": 105, "dropout_1": 0.48639033141890325, "dropout_2": 0.21324913218446714, "learning_rate": 0.00013769715715418189, "wd": 0.02017249366944585}, "metrics": {"cost_metric": {"1": 13.569249868392944}, "active_metric": {"1": 0.3023558897243108}}}, {"candidate": {"n_units_1": 673, "n_units_2": 262, "batch_size": 78, "dropout_1": 0.9510740133913004, "dropout_2": 0.3263851441475057, "learning_rate": 0.009715536539110267, "wd": 0.0002984576239921338}, "metrics": {"cost_metric": {"1": 22.84876799583435}, "active_metric": {"1": 0.7708333333333334}}}, {"candidate": {"n_units_1": 29, "n_units_2": 584, "batch_size": 41, "dropout_1": 0.6843134351325143, "dropout_2": 0.20866634010808638, "learning_rate": 0.0007531586746050282, "wd": 8.542702035112525e-08}, "metrics": {"cost_metric": {"1": 16.058557987213135}, "active_metric": {"1": 0.4832881662149955}}}], "failed_candidates": [], "pending_candidates": [{"batch_size": 30, "dropout_1": 0.5490868842845377, "dropout_2": 0.7920401266415242, "learning_rate": 2.101654064946448e-06, "n_units_1": 529, "n_units_2": 660, "wd": 9.568417280766271e-08, "RESOURCE_ATTR_epoch": 1}, {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607, "RESOURCE_ATTR_epoch": 8}, {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07, "RESOURCE_ATTR_epoch": 6}]}')

# elapsed_time = 60.27482986450195
# num_observations = 8
# num_configs = 4
_model_params.append('{"noise_variance": 0.0010000000000000002, "kernel_alpha": 1.0, "kernel_mean_lam": 0.5, "kernel_gamma": 0.5, "kernel_kernelx_inv_bw0": 1.0, "kernel_kernelx_inv_bw1": 1.0, "kernel_kernelx_inv_bw2": 1.0, "kernel_kernelx_inv_bw3": 1.0, "kernel_kernelx_inv_bw4": 1.0, "kernel_kernelx_inv_bw5": 1.0, "kernel_kernelx_inv_bw6": 1.0, "kernel_kernelx_covariance_scale": 1.0, "kernel_meanx_mean_value": 0.0}')
_state.append('{"candidate_evaluations": [{"candidate": {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607}, "metrics": {"cost_metric": {"1": 15.282694101333618, "2": 32.215107917785645, "3": 48.62396502494812, "4": 62.46452808380127, "5": 77.36029410362244, "6": 91.85972809791565, "7": 107.89106607437134, "8": 122.8071219921112, "9": 138.6427869796753, "10": 155.3775441646576, "11": 171.86729407310486, "12": 186.69335007667542, "13": 204.0226011276245, "14": 219.01899409294128, "15": 234.25396490097046, "16": 249.26809406280518, "17": 264.3408589363098, "18": 280.09520292282104, "19": 296.0671851634979, "20": 319.6674599647522, "21": 337.2237899303436, "22": 354.650151014328, "23": 370.17603182792664, "24": 386.20747780799866, "25": 401.40604305267334, "26": 415.9903359413147, "27": 431.2174470424652, "28": 445.8420739173889, "29": 460.1389670372009, "30": 475.33422088623047, "31": 493.9087038040161, "32": 509.9310200214386, "33": 525.421621799469, "34": 540.1411230564117, "35": 554.5012362003326, "36": 569.0483500957489, "37": 583.1704621315002}, "active_metric": {"1": 0.41328783621035725, "2": 0.3706342834203131, "3": 0.351264552388599, "4": 0.33500602167804094, "5": 0.32707747892412686, "6": 0.3204536330790847, "7": 0.3140305098354075, "8": 0.2992773986350863, "9": 0.2865315134484143, "10": 0.27509032517061416, "11": 0.26525491770373344, "12": 0.2552187876354878, "13": 0.24989963869931753, "14": 0.24197109594540345, "15": 0.2352468887996788, "16": 0.22902448815736653, "17": 0.2233038940184665, "18": 0.2191890806904857, "19": 0.2133681252509032, "20": 0.21045764753111196, "21": 0.20503813729425935, "22": 0.20373344038538743, "23": 0.2004215174628663, "24": 0.1956041750301084, "25": 0.19299478121236446, "26": 0.19199116820553996, "27": 0.18908069048574871, "28": 0.18667201926936972, "29": 0.18526696105981533, "30": 0.18426334805299072, "31": 0.18406262545162588, "32": 0.18255720594138902, "33": 0.18145323163388194, "34": 0.18105178643115216, "35": 0.1785427539140907, "36": 0.1785427539140907, "37": 0.17723805700521877}}}, {"candidate": {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07}, "metrics": {"cost_metric": {"1": 22.334229230880737, "2": 47.57227420806885, "3": 68.38872504234314, "4": 91.17699599266052, "5": 114.11172008514404, "6": 137.2291030883789, "7": 161.9407799243927, "8": 186.01491808891296, "9": 210.73575019836426, "10": 233.72683000564575, "11": 256.1681180000305, "12": 279.3303301334381, "13": 311.3050580024719, "14": 336.1765010356903, "15": 362.16334199905396, "16": 385.8361783027649, "17": 408.6689350605011, "18": 430.98025918006897, "19": 453.24402713775635, "20": 474.99853110313416, "21": 501.653440952301, "22": 525.0653901100159, "23": 547.2747151851654, "24": 569.0442740917206, "25": 590.4288313388824}, "active_metric": {"1": 0.1725761217948718, "2": 0.15024038461538458, "3": 0.1353165064102564, "4": 0.13842147435897434, "5": 0.1303084935897436, "6": 0.12239583333333337, "7": 0.12089342948717952, "8": 0.11738782051282048, "9": 0.11909054487179482, "10": 0.11498397435897434, "11": 0.11618589743589747, "12": 0.11388221153846156, "13": 0.11087740384615385, "14": 0.11247996794871795, "15": 0.11227964743589747, "16": 0.11037660256410253, "17": 0.11237980769230771, "18": 0.10917467948717952, "19": 0.10506810897435892, "20": 0.10947516025641024, "21": 0.10627003205128205, "22": 0.10366586538461542, "23": 0.10356570512820518, "24": 0.10837339743589747, "25": 0.10406650641025639}}}, {"candidate": {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05}, "metrics": {"cost_metric": {"1": 23.648607969284058, "2": 50.783905029296875, "3": 94.63147592544556}, "active_metric": {"1": 0.16546618647458988, "2": 0.1404561824729892, "3": 0.1386554621848739}}}, {"candidate": {"n_units_1": 774, "n_units_2": 917, "batch_size": 29, "dropout_1": 0.7778923725289609, "dropout_2": 0.7413003050986398, "learning_rate": 6.472832341968678e-05, "wd": 0.0007744951242384949}, "metrics": {"cost_metric": {"1": 50.06452512741089}, "active_metric": {"1": 0.22955092221331197}}}], "failed_candidates": [], "pending_candidates": [{"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07, "RESOURCE_ATTR_epoch": 3}, {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05, "RESOURCE_ATTR_epoch": 3}, {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607, "RESOURCE_ATTR_epoch": 4}]}')

# elapsed_time = 30.158570051193237
# num_observations = 2
# num_configs = 2
_model_params.append('{"noise_variance": 0.0010000000000000002, "kernel_alpha": 1.0, "kernel_mean_lam": 0.5, "kernel_gamma": 0.5, "kernel_kernelx_inv_bw0": 1.0, "kernel_kernelx_inv_bw1": 1.0, "kernel_kernelx_inv_bw2": 1.0, "kernel_kernelx_inv_bw3": 1.0, "kernel_kernelx_inv_bw4": 1.0, "kernel_kernelx_inv_bw5": 1.0, "kernel_kernelx_inv_bw6": 1.0, "kernel_kernelx_covariance_scale": 1.0, "kernel_meanx_mean_value": 0.0}')
_state.append('{"candidate_evaluations": [{"candidate": {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607}, "metrics": {"cost_metric": {"1": 15.282694101333618, "2": 32.215107917785645, "3": 48.62396502494812, "4": 62.46452808380127, "5": 77.36029410362244, "6": 91.85972809791565, "7": 107.89106607437134, "8": 122.8071219921112, "9": 138.6427869796753, "10": 155.3775441646576, "11": 171.86729407310486, "12": 186.69335007667542, "13": 204.0226011276245, "14": 219.01899409294128, "15": 234.25396490097046, "16": 249.26809406280518, "17": 264.3408589363098, "18": 280.09520292282104, "19": 296.0671851634979, "20": 319.6674599647522, "21": 337.2237899303436, "22": 354.650151014328, "23": 370.17603182792664, "24": 386.20747780799866, "25": 401.40604305267334, "26": 415.9903359413147, "27": 431.2174470424652, "28": 445.8420739173889, "29": 460.1389670372009, "30": 475.33422088623047, "31": 493.9087038040161, "32": 509.9310200214386, "33": 525.421621799469, "34": 540.1411230564117, "35": 554.5012362003326, "36": 569.0483500957489, "37": 583.1704621315002}, "active_metric": {"1": 0.41328783621035725, "2": 0.3706342834203131, "3": 0.351264552388599, "4": 0.33500602167804094, "5": 0.32707747892412686, "6": 0.3204536330790847, "7": 0.3140305098354075, "8": 0.2992773986350863, "9": 0.2865315134484143, "10": 0.27509032517061416, "11": 0.26525491770373344, "12": 0.2552187876354878, "13": 0.24989963869931753, "14": 0.24197109594540345, "15": 0.2352468887996788, "16": 0.22902448815736653, "17": 0.2233038940184665, "18": 0.2191890806904857, "19": 0.2133681252509032, "20": 0.21045764753111196, "21": 0.20503813729425935, "22": 0.20373344038538743, "23": 0.2004215174628663, "24": 0.1956041750301084, "25": 0.19299478121236446, "26": 0.19199116820553996, "27": 0.18908069048574871, "28": 0.18667201926936972, "29": 0.18526696105981533, "30": 0.18426334805299072, "31": 0.18406262545162588, "32": 0.18255720594138902, "33": 0.18145323163388194, "34": 0.18105178643115216, "35": 0.1785427539140907, "36": 0.1785427539140907, "37": 0.17723805700521877}}}, {"candidate": {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07}, "metrics": {"cost_metric": {"1": 22.334229230880737, "2": 47.57227420806885, "3": 68.38872504234314, "4": 91.17699599266052, "5": 114.11172008514404, "6": 137.2291030883789, "7": 161.9407799243927, "8": 186.01491808891296, "9": 210.73575019836426, "10": 233.72683000564575, "11": 256.1681180000305, "12": 279.3303301334381, "13": 311.3050580024719, "14": 336.1765010356903, "15": 362.16334199905396, "16": 385.8361783027649, "17": 408.6689350605011, "18": 430.98025918006897, "19": 453.24402713775635, "20": 474.99853110313416, "21": 501.653440952301, "22": 525.0653901100159, "23": 547.2747151851654, "24": 569.0442740917206, "25": 590.4288313388824}, "active_metric": {"1": 0.1725761217948718, "2": 0.15024038461538458, "3": 0.1353165064102564, "4": 0.13842147435897434, "5": 0.1303084935897436, "6": 0.12239583333333337, "7": 0.12089342948717952, "8": 0.11738782051282048, "9": 0.11909054487179482, "10": 0.11498397435897434, "11": 0.11618589743589747, "12": 0.11388221153846156, "13": 0.11087740384615385, "14": 0.11247996794871795, "15": 0.11227964743589747, "16": 0.11037660256410253, "17": 0.11237980769230771, "18": 0.10917467948717952, "19": 0.10506810897435892, "20": 0.10947516025641024, "21": 0.10627003205128205, "22": 0.10366586538461542, "23": 0.10356570512820518, "24": 0.10837339743589747, "25": 0.10406650641025639}}}], "failed_candidates": [], "pending_candidates": [{"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05, "RESOURCE_ATTR_epoch": 1}, {"n_units_1": 774, "n_units_2": 917, "batch_size": 29, "dropout_1": 0.7778923725289609, "dropout_2": 0.7413003050986398, "learning_rate": 6.472832341968678e-05, "wd": 0.0007744951242384949, "RESOURCE_ATTR_epoch": 1}, {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607, "RESOURCE_ATTR_epoch": 2}, {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07, "RESOURCE_ATTR_epoch": 2}]}')

# elapsed_time = 80.36403489112854
# num_observations = 11
# num_configs = 5
_model_params.append('{"noise_variance": 0.0010000000000000002, "kernel_alpha": 1.0, "kernel_mean_lam": 0.5, "kernel_gamma": 0.5, "kernel_kernelx_inv_bw0": 1.0, "kernel_kernelx_inv_bw1": 1.0, "kernel_kernelx_inv_bw2": 1.0, "kernel_kernelx_inv_bw3": 1.0, "kernel_kernelx_inv_bw4": 1.0, "kernel_kernelx_inv_bw5": 1.0, "kernel_kernelx_inv_bw6": 1.0, "kernel_kernelx_covariance_scale": 1.0, "kernel_meanx_mean_value": 0.0}')
_state.append('{"candidate_evaluations": [{"candidate": {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607}, "metrics": {"cost_metric": {"1": 15.282694101333618, "2": 32.215107917785645, "3": 48.62396502494812, "4": 62.46452808380127, "5": 77.36029410362244, "6": 91.85972809791565, "7": 107.89106607437134, "8": 122.8071219921112, "9": 138.6427869796753, "10": 155.3775441646576, "11": 171.86729407310486, "12": 186.69335007667542, "13": 204.0226011276245, "14": 219.01899409294128, "15": 234.25396490097046, "16": 249.26809406280518, "17": 264.3408589363098, "18": 280.09520292282104, "19": 296.0671851634979, "20": 319.6674599647522, "21": 337.2237899303436, "22": 354.650151014328, "23": 370.17603182792664, "24": 386.20747780799866, "25": 401.40604305267334, "26": 415.9903359413147, "27": 431.2174470424652, "28": 445.8420739173889, "29": 460.1389670372009, "30": 475.33422088623047, "31": 493.9087038040161, "32": 509.9310200214386, "33": 525.421621799469, "34": 540.1411230564117, "35": 554.5012362003326, "36": 569.0483500957489, "37": 583.1704621315002}, "active_metric": {"1": 0.41328783621035725, "2": 0.3706342834203131, "3": 0.351264552388599, "4": 0.33500602167804094, "5": 0.32707747892412686, "6": 0.3204536330790847, "7": 0.3140305098354075, "8": 0.2992773986350863, "9": 0.2865315134484143, "10": 0.27509032517061416, "11": 0.26525491770373344, "12": 0.2552187876354878, "13": 0.24989963869931753, "14": 0.24197109594540345, "15": 0.2352468887996788, "16": 0.22902448815736653, "17": 0.2233038940184665, "18": 0.2191890806904857, "19": 0.2133681252509032, "20": 0.21045764753111196, "21": 0.20503813729425935, "22": 0.20373344038538743, "23": 0.2004215174628663, "24": 0.1956041750301084, "25": 0.19299478121236446, "26": 0.19199116820553996, "27": 0.18908069048574871, "28": 0.18667201926936972, "29": 0.18526696105981533, "30": 0.18426334805299072, "31": 0.18406262545162588, "32": 0.18255720594138902, "33": 0.18145323163388194, "34": 0.18105178643115216, "35": 0.1785427539140907, "36": 0.1785427539140907, "37": 0.17723805700521877}}}, {"candidate": {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07}, "metrics": {"cost_metric": {"1": 22.334229230880737, "2": 47.57227420806885, "3": 68.38872504234314, "4": 91.17699599266052, "5": 114.11172008514404, "6": 137.2291030883789, "7": 161.9407799243927, "8": 186.01491808891296, "9": 210.73575019836426, "10": 233.72683000564575, "11": 256.1681180000305, "12": 279.3303301334381, "13": 311.3050580024719, "14": 336.1765010356903, "15": 362.16334199905396, "16": 385.8361783027649, "17": 408.6689350605011, "18": 430.98025918006897, "19": 453.24402713775635, "20": 474.99853110313416, "21": 501.653440952301, "22": 525.0653901100159, "23": 547.2747151851654, "24": 569.0442740917206, "25": 590.4288313388824}, "active_metric": {"1": 0.1725761217948718, "2": 0.15024038461538458, "3": 0.1353165064102564, "4": 0.13842147435897434, "5": 0.1303084935897436, "6": 0.12239583333333337, "7": 0.12089342948717952, "8": 0.11738782051282048, "9": 0.11909054487179482, "10": 0.11498397435897434, "11": 0.11618589743589747, "12": 0.11388221153846156, "13": 0.11087740384615385, "14": 0.11247996794871795, "15": 0.11227964743589747, "16": 0.11037660256410253, "17": 0.11237980769230771, "18": 0.10917467948717952, "19": 0.10506810897435892, "20": 0.10947516025641024, "21": 0.10627003205128205, "22": 0.10366586538461542, "23": 0.10356570512820518, "24": 0.10837339743589747, "25": 0.10406650641025639}}}, {"candidate": {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05}, "metrics": {"cost_metric": {"1": 23.648607969284058, "2": 50.783905029296875, "3": 94.63147592544556}, "active_metric": {"1": 0.16546618647458988, "2": 0.1404561824729892, "3": 0.1386554621848739}}}, {"candidate": {"n_units_1": 774, "n_units_2": 917, "batch_size": 29, "dropout_1": 0.7778923725289609, "dropout_2": 0.7413003050986398, "learning_rate": 6.472832341968678e-05, "wd": 0.0007744951242384949}, "metrics": {"cost_metric": {"1": 50.06452512741089}, "active_metric": {"1": 0.22955092221331197}}}, {"candidate": {"n_units_1": 91, "n_units_2": 459, "batch_size": 105, "dropout_1": 0.48639033141890325, "dropout_2": 0.21324913218446714, "learning_rate": 0.00013769715715418189, "wd": 0.02017249366944585}, "metrics": {"cost_metric": {"1": 13.569249868392944}, "active_metric": {"1": 0.3023558897243108}}}], "failed_candidates": [], "pending_candidates": [{"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05, "RESOURCE_ATTR_epoch": 3}, {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607, "RESOURCE_ATTR_epoch": 5}, {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07, "RESOURCE_ATTR_epoch": 4}]}')

# elapsed_time = 299.2465078830719
# num_observations = 45
# num_configs = 12
_model_params.append('{"noise_variance": 0.0003423634954194395, "kernel_alpha": 0.27529357970772367, "kernel_mean_lam": 20.70369146606114, "kernel_gamma": 1.0, "kernel_kernelx_inv_bw0": 0.00021426786615588259, "kernel_kernelx_inv_bw1": 1.949848307645357, "kernel_kernelx_inv_bw2": 0.0007825688851177687, "kernel_kernelx_inv_bw3": 3.4671378069531356, "kernel_kernelx_inv_bw4": 1.090384354169918, "kernel_kernelx_inv_bw5": 0.0028908649203899387, "kernel_kernelx_inv_bw6": 0.00011473931202983017, "kernel_kernelx_covariance_scale": 1.5931693693573752, "kernel_meanx_mean_value": 0.8391456738845063}')
_state.append('{"candidate_evaluations": [{"candidate": {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607}, "metrics": {"cost_metric": {"1": 15.282694101333618, "2": 32.215107917785645, "3": 48.62396502494812, "4": 62.46452808380127, "5": 77.36029410362244, "6": 91.85972809791565, "7": 107.89106607437134, "8": 122.8071219921112, "9": 138.6427869796753, "10": 155.3775441646576, "11": 171.86729407310486, "12": 186.69335007667542, "13": 204.0226011276245, "14": 219.01899409294128, "15": 234.25396490097046, "16": 249.26809406280518, "17": 264.3408589363098, "18": 280.09520292282104, "19": 296.0671851634979, "20": 319.6674599647522, "21": 337.2237899303436, "22": 354.650151014328, "23": 370.17603182792664, "24": 386.20747780799866, "25": 401.40604305267334, "26": 415.9903359413147, "27": 431.2174470424652, "28": 445.8420739173889, "29": 460.1389670372009, "30": 475.33422088623047, "31": 493.9087038040161, "32": 509.9310200214386, "33": 525.421621799469, "34": 540.1411230564117, "35": 554.5012362003326, "36": 569.0483500957489, "37": 583.1704621315002}, "active_metric": {"1": 0.41328783621035725, "2": 0.3706342834203131, "3": 0.351264552388599, "4": 0.33500602167804094, "5": 0.32707747892412686, "6": 0.3204536330790847, "7": 0.3140305098354075, "8": 0.2992773986350863, "9": 0.2865315134484143, "10": 0.27509032517061416, "11": 0.26525491770373344, "12": 0.2552187876354878, "13": 0.24989963869931753, "14": 0.24197109594540345, "15": 0.2352468887996788, "16": 0.22902448815736653, "17": 0.2233038940184665, "18": 0.2191890806904857, "19": 0.2133681252509032, "20": 0.21045764753111196, "21": 0.20503813729425935, "22": 0.20373344038538743, "23": 0.2004215174628663, "24": 0.1956041750301084, "25": 0.19299478121236446, "26": 0.19199116820553996, "27": 0.18908069048574871, "28": 0.18667201926936972, "29": 0.18526696105981533, "30": 0.18426334805299072, "31": 0.18406262545162588, "32": 0.18255720594138902, "33": 0.18145323163388194, "34": 0.18105178643115216, "35": 0.1785427539140907, "36": 0.1785427539140907, "37": 0.17723805700521877}}}, {"candidate": {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07}, "metrics": {"cost_metric": {"1": 22.334229230880737, "2": 47.57227420806885, "3": 68.38872504234314, "4": 91.17699599266052, "5": 114.11172008514404, "6": 137.2291030883789, "7": 161.9407799243927, "8": 186.01491808891296, "9": 210.73575019836426, "10": 233.72683000564575, "11": 256.1681180000305, "12": 279.3303301334381, "13": 311.3050580024719, "14": 336.1765010356903, "15": 362.16334199905396, "16": 385.8361783027649, "17": 408.6689350605011, "18": 430.98025918006897, "19": 453.24402713775635, "20": 474.99853110313416, "21": 501.653440952301, "22": 525.0653901100159, "23": 547.2747151851654, "24": 569.0442740917206, "25": 590.4288313388824}, "active_metric": {"1": 0.1725761217948718, "2": 0.15024038461538458, "3": 0.1353165064102564, "4": 0.13842147435897434, "5": 0.1303084935897436, "6": 0.12239583333333337, "7": 0.12089342948717952, "8": 0.11738782051282048, "9": 0.11909054487179482, "10": 0.11498397435897434, "11": 0.11618589743589747, "12": 0.11388221153846156, "13": 0.11087740384615385, "14": 0.11247996794871795, "15": 0.11227964743589747, "16": 0.11037660256410253, "17": 0.11237980769230771, "18": 0.10917467948717952, "19": 0.10506810897435892, "20": 0.10947516025641024, "21": 0.10627003205128205, "22": 0.10366586538461542, "23": 0.10356570512820518, "24": 0.10837339743589747, "25": 0.10406650641025639}}}, {"candidate": {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05}, "metrics": {"cost_metric": {"1": 23.648607969284058, "2": 50.783905029296875, "3": 94.63147592544556}, "active_metric": {"1": 0.16546618647458988, "2": 0.1404561824729892, "3": 0.1386554621848739}}}, {"candidate": {"n_units_1": 774, "n_units_2": 917, "batch_size": 29, "dropout_1": 0.7778923725289609, "dropout_2": 0.7413003050986398, "learning_rate": 6.472832341968678e-05, "wd": 0.0007744951242384949}, "metrics": {"cost_metric": {"1": 50.06452512741089}, "active_metric": {"1": 0.22955092221331197}}}, {"candidate": {"n_units_1": 91, "n_units_2": 459, "batch_size": 105, "dropout_1": 0.48639033141890325, "dropout_2": 0.21324913218446714, "learning_rate": 0.00013769715715418189, "wd": 0.02017249366944585}, "metrics": {"cost_metric": {"1": 13.569249868392944}, "active_metric": {"1": 0.3023558897243108}}}, {"candidate": {"n_units_1": 673, "n_units_2": 262, "batch_size": 78, "dropout_1": 0.9510740133913004, "dropout_2": 0.3263851441475057, "learning_rate": 0.009715536539110267, "wd": 0.0002984576239921338}, "metrics": {"cost_metric": {"1": 22.84876799583435}, "active_metric": {"1": 0.7708333333333334}}}, {"candidate": {"n_units_1": 29, "n_units_2": 584, "batch_size": 41, "dropout_1": 0.6843134351325143, "dropout_2": 0.20866634010808638, "learning_rate": 0.0007531586746050282, "wd": 8.542702035112525e-08}, "metrics": {"cost_metric": {"1": 16.058557987213135}, "active_metric": {"1": 0.4832881662149955}}}, {"candidate": {"n_units_1": 529, "n_units_2": 660, "batch_size": 30, "dropout_1": 0.5490868842845377, "dropout_2": 0.7920401266415242, "learning_rate": 2.101654064946448e-06, "wd": 9.568417280766271e-08}, "metrics": {"cost_metric": {"1": 33.76100707054138}, "active_metric": {"1": 0.47597597597597596}}}, {"candidate": {"n_units_1": 828, "n_units_2": 634, "batch_size": 123, "dropout_1": 0.4257740273137951, "dropout_2": 0.23113551626017848, "learning_rate": 6.860930897860399e-06, "wd": 8.101814548564974e-08}, "metrics": {"cost_metric": {"1": 27.756996154785156}, "active_metric": {"1": 0.33835190203753884}}}, {"candidate": {"n_units_1": 635, "n_units_2": 538, "batch_size": 69, "dropout_1": 0.45999796301051993, "dropout_2": 0.1659629767457713, "learning_rate": 0.00045244429498637726, "wd": 7.074070723442468e-06}, "metrics": {"cost_metric": {"1": 26.10046887397766, "2": 53.824394941329956, "3": 80.07230305671692, "4": 106.25571393966675, "5": 133.94080090522766, "6": 170.4485969543457, "7": 200.93623304367065, "8": 229.56330180168152, "9": 257.11693382263184, "10": 286.6725299358368, "11": 319.0151889324188, "12": 354.14807295799255, "13": 386.9317948818207, "14": 419.8007171154022}, "active_metric": {"1": 0.14744363929146542, "2": 0.13586956521739135, "3": 0.12983091787439616, "4": 0.12661030595813205, "5": 0.1279186795491143, "6": 0.12328904991948475, "7": 0.11674718196457323, "8": 0.1126207729468599, "9": 0.11292270531400961, "10": 0.11513687600644118, "11": 0.11584138486312401, "12": 0.11101046698872785, "13": 0.1090982286634461, "14": 0.10738727858293073}}}, {"candidate": {"n_units_1": 931, "n_units_2": 527, "batch_size": 20, "dropout_1": 0.41104972550281815, "dropout_2": 0.7568250476630992, "learning_rate": 0.8320700517813453, "wd": 1.9853072351534312e-08}, "metrics": {"cost_metric": {"1": 117.45865964889526}, "active_metric": {"1": 0.9045}}}, {"candidate": {"n_units_1": 576, "n_units_2": 265, "batch_size": 68, "dropout_1": 0.3230142899473453, "dropout_2": 0.3493653937208789, "learning_rate": 0.0004443989879229176, "wd": 0.012741750252890758}, "metrics": {"cost_metric": {"1": 23.032225131988525, "2": 53.855369091033936, "3": 78.89936113357544}, "active_metric": {"1": 0.18497398959583833, "2": 0.1910764305722289, "3": 0.17987194877951185}}}], "failed_candidates": [], "pending_candidates": [{"n_units_1": 635, "n_units_2": 538, "batch_size": 69, "dropout_1": 0.45999796301051993, "dropout_2": 0.1659629767457713, "learning_rate": 0.00045244429498637726, "wd": 7.074070723442468e-06, "RESOURCE_ATTR_epoch": 5}, {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07, "RESOURCE_ATTR_epoch": 13}, {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607, "RESOURCE_ATTR_epoch": 19}, {"n_units_1": 576, "n_units_2": 265, "batch_size": 68, "dropout_1": 0.3230142899473453, "dropout_2": 0.3493653937208789, "learning_rate": 0.0004443989879229176, "wd": 0.012741750252890758, "RESOURCE_ATTR_epoch": 2}]}')

# elapsed_time = 107.72016072273254
# num_observations = 16
# num_configs = 6
_model_params.append('{"noise_variance": 0.0007127447108752395, "kernel_alpha": 0.6611830364595244, "kernel_mean_lam": 49.99999999999999, "kernel_gamma": 1.0, "kernel_kernelx_inv_bw0": 0.00404913469299303, "kernel_kernelx_inv_bw1": 0.016052742254326176, "kernel_kernelx_inv_bw2": 0.0006873813980510872, "kernel_kernelx_inv_bw3": 0.0018486468658638656, "kernel_kernelx_inv_bw4": 2.2411401206695025, "kernel_kernelx_inv_bw5": 3.6329202178079703, "kernel_kernelx_inv_bw6": 0.0006680897104882892, "kernel_kernelx_covariance_scale": 0.7202707931819882, "kernel_meanx_mean_value": -0.25886685846400814}')
_state.append('{"candidate_evaluations": [{"candidate": {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607}, "metrics": {"cost_metric": {"1": 15.282694101333618, "2": 32.215107917785645, "3": 48.62396502494812, "4": 62.46452808380127, "5": 77.36029410362244, "6": 91.85972809791565, "7": 107.89106607437134, "8": 122.8071219921112, "9": 138.6427869796753, "10": 155.3775441646576, "11": 171.86729407310486, "12": 186.69335007667542, "13": 204.0226011276245, "14": 219.01899409294128, "15": 234.25396490097046, "16": 249.26809406280518, "17": 264.3408589363098, "18": 280.09520292282104, "19": 296.0671851634979, "20": 319.6674599647522, "21": 337.2237899303436, "22": 354.650151014328, "23": 370.17603182792664, "24": 386.20747780799866, "25": 401.40604305267334, "26": 415.9903359413147, "27": 431.2174470424652, "28": 445.8420739173889, "29": 460.1389670372009, "30": 475.33422088623047, "31": 493.9087038040161, "32": 509.9310200214386, "33": 525.421621799469, "34": 540.1411230564117, "35": 554.5012362003326, "36": 569.0483500957489, "37": 583.1704621315002}, "active_metric": {"1": 0.41328783621035725, "2": 0.3706342834203131, "3": 0.351264552388599, "4": 0.33500602167804094, "5": 0.32707747892412686, "6": 0.3204536330790847, "7": 0.3140305098354075, "8": 0.2992773986350863, "9": 0.2865315134484143, "10": 0.27509032517061416, "11": 0.26525491770373344, "12": 0.2552187876354878, "13": 0.24989963869931753, "14": 0.24197109594540345, "15": 0.2352468887996788, "16": 0.22902448815736653, "17": 0.2233038940184665, "18": 0.2191890806904857, "19": 0.2133681252509032, "20": 0.21045764753111196, "21": 0.20503813729425935, "22": 0.20373344038538743, "23": 0.2004215174628663, "24": 0.1956041750301084, "25": 0.19299478121236446, "26": 0.19199116820553996, "27": 0.18908069048574871, "28": 0.18667201926936972, "29": 0.18526696105981533, "30": 0.18426334805299072, "31": 0.18406262545162588, "32": 0.18255720594138902, "33": 0.18145323163388194, "34": 0.18105178643115216, "35": 0.1785427539140907, "36": 0.1785427539140907, "37": 0.17723805700521877}}}, {"candidate": {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07}, "metrics": {"cost_metric": {"1": 22.334229230880737, "2": 47.57227420806885, "3": 68.38872504234314, "4": 91.17699599266052, "5": 114.11172008514404, "6": 137.2291030883789, "7": 161.9407799243927, "8": 186.01491808891296, "9": 210.73575019836426, "10": 233.72683000564575, "11": 256.1681180000305, "12": 279.3303301334381, "13": 311.3050580024719, "14": 336.1765010356903, "15": 362.16334199905396, "16": 385.8361783027649, "17": 408.6689350605011, "18": 430.98025918006897, "19": 453.24402713775635, "20": 474.99853110313416, "21": 501.653440952301, "22": 525.0653901100159, "23": 547.2747151851654, "24": 569.0442740917206, "25": 590.4288313388824}, "active_metric": {"1": 0.1725761217948718, "2": 0.15024038461538458, "3": 0.1353165064102564, "4": 0.13842147435897434, "5": 0.1303084935897436, "6": 0.12239583333333337, "7": 0.12089342948717952, "8": 0.11738782051282048, "9": 0.11909054487179482, "10": 0.11498397435897434, "11": 0.11618589743589747, "12": 0.11388221153846156, "13": 0.11087740384615385, "14": 0.11247996794871795, "15": 0.11227964743589747, "16": 0.11037660256410253, "17": 0.11237980769230771, "18": 0.10917467948717952, "19": 0.10506810897435892, "20": 0.10947516025641024, "21": 0.10627003205128205, "22": 0.10366586538461542, "23": 0.10356570512820518, "24": 0.10837339743589747, "25": 0.10406650641025639}}}, {"candidate": {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05}, "metrics": {"cost_metric": {"1": 23.648607969284058, "2": 50.783905029296875, "3": 94.63147592544556}, "active_metric": {"1": 0.16546618647458988, "2": 0.1404561824729892, "3": 0.1386554621848739}}}, {"candidate": {"n_units_1": 774, "n_units_2": 917, "batch_size": 29, "dropout_1": 0.7778923725289609, "dropout_2": 0.7413003050986398, "learning_rate": 6.472832341968678e-05, "wd": 0.0007744951242384949}, "metrics": {"cost_metric": {"1": 50.06452512741089}, "active_metric": {"1": 0.22955092221331197}}}, {"candidate": {"n_units_1": 91, "n_units_2": 459, "batch_size": 105, "dropout_1": 0.48639033141890325, "dropout_2": 0.21324913218446714, "learning_rate": 0.00013769715715418189, "wd": 0.02017249366944585}, "metrics": {"cost_metric": {"1": 13.569249868392944}, "active_metric": {"1": 0.3023558897243108}}}, {"candidate": {"n_units_1": 673, "n_units_2": 262, "batch_size": 78, "dropout_1": 0.9510740133913004, "dropout_2": 0.3263851441475057, "learning_rate": 0.009715536539110267, "wd": 0.0002984576239921338}, "metrics": {"cost_metric": {"1": 22.84876799583435}, "active_metric": {"1": 0.7708333333333334}}}], "failed_candidates": [], "pending_candidates": [{"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07, "RESOURCE_ATTR_epoch": 5}, {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607, "RESOURCE_ATTR_epoch": 7}, {"batch_size": 30, "dropout_1": 0.5490868842845377, "dropout_2": 0.7920401266415242, "learning_rate": 2.101654064946448e-06, "n_units_1": 529, "n_units_2": 660, "wd": 9.568417280766271e-08, "RESOURCE_ATTR_epoch": 1}]}')

# elapsed_time = 196.5048840045929
# num_observations = 30
# num_configs = 10
_model_params.append('{"noise_variance": 0.0013634444543287733, "kernel_alpha": 0.33215948811021134, "kernel_mean_lam": 15.077221388452632, "kernel_gamma": 1.0, "kernel_kernelx_inv_bw0": 0.0025264221960287574, "kernel_kernelx_inv_bw1": 4.586734084695334, "kernel_kernelx_inv_bw2": 0.003154109460037928, "kernel_kernelx_inv_bw3": 0.000894766660821803, "kernel_kernelx_inv_bw4": 0.005214180628382712, "kernel_kernelx_inv_bw5": 10.093642962950414, "kernel_kernelx_inv_bw6": 0.016153252870882064, "kernel_kernelx_covariance_scale": 1.2311611100489817, "kernel_meanx_mean_value": 0.09552280164740308}')
_state.append('{"candidate_evaluations": [{"candidate": {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607}, "metrics": {"cost_metric": {"1": 15.282694101333618, "2": 32.215107917785645, "3": 48.62396502494812, "4": 62.46452808380127, "5": 77.36029410362244, "6": 91.85972809791565, "7": 107.89106607437134, "8": 122.8071219921112, "9": 138.6427869796753, "10": 155.3775441646576, "11": 171.86729407310486, "12": 186.69335007667542, "13": 204.0226011276245, "14": 219.01899409294128, "15": 234.25396490097046, "16": 249.26809406280518, "17": 264.3408589363098, "18": 280.09520292282104, "19": 296.0671851634979, "20": 319.6674599647522, "21": 337.2237899303436, "22": 354.650151014328, "23": 370.17603182792664, "24": 386.20747780799866, "25": 401.40604305267334, "26": 415.9903359413147, "27": 431.2174470424652, "28": 445.8420739173889, "29": 460.1389670372009, "30": 475.33422088623047, "31": 493.9087038040161, "32": 509.9310200214386, "33": 525.421621799469, "34": 540.1411230564117, "35": 554.5012362003326, "36": 569.0483500957489, "37": 583.1704621315002}, "active_metric": {"1": 0.41328783621035725, "2": 0.3706342834203131, "3": 0.351264552388599, "4": 0.33500602167804094, "5": 0.32707747892412686, "6": 0.3204536330790847, "7": 0.3140305098354075, "8": 0.2992773986350863, "9": 0.2865315134484143, "10": 0.27509032517061416, "11": 0.26525491770373344, "12": 0.2552187876354878, "13": 0.24989963869931753, "14": 0.24197109594540345, "15": 0.2352468887996788, "16": 0.22902448815736653, "17": 0.2233038940184665, "18": 0.2191890806904857, "19": 0.2133681252509032, "20": 0.21045764753111196, "21": 0.20503813729425935, "22": 0.20373344038538743, "23": 0.2004215174628663, "24": 0.1956041750301084, "25": 0.19299478121236446, "26": 0.19199116820553996, "27": 0.18908069048574871, "28": 0.18667201926936972, "29": 0.18526696105981533, "30": 0.18426334805299072, "31": 0.18406262545162588, "32": 0.18255720594138902, "33": 0.18145323163388194, "34": 0.18105178643115216, "35": 0.1785427539140907, "36": 0.1785427539140907, "37": 0.17723805700521877}}}, {"candidate": {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07}, "metrics": {"cost_metric": {"1": 22.334229230880737, "2": 47.57227420806885, "3": 68.38872504234314, "4": 91.17699599266052, "5": 114.11172008514404, "6": 137.2291030883789, "7": 161.9407799243927, "8": 186.01491808891296, "9": 210.73575019836426, "10": 233.72683000564575, "11": 256.1681180000305, "12": 279.3303301334381, "13": 311.3050580024719, "14": 336.1765010356903, "15": 362.16334199905396, "16": 385.8361783027649, "17": 408.6689350605011, "18": 430.98025918006897, "19": 453.24402713775635, "20": 474.99853110313416, "21": 501.653440952301, "22": 525.0653901100159, "23": 547.2747151851654, "24": 569.0442740917206, "25": 590.4288313388824}, "active_metric": {"1": 0.1725761217948718, "2": 0.15024038461538458, "3": 0.1353165064102564, "4": 0.13842147435897434, "5": 0.1303084935897436, "6": 0.12239583333333337, "7": 0.12089342948717952, "8": 0.11738782051282048, "9": 0.11909054487179482, "10": 0.11498397435897434, "11": 0.11618589743589747, "12": 0.11388221153846156, "13": 0.11087740384615385, "14": 0.11247996794871795, "15": 0.11227964743589747, "16": 0.11037660256410253, "17": 0.11237980769230771, "18": 0.10917467948717952, "19": 0.10506810897435892, "20": 0.10947516025641024, "21": 0.10627003205128205, "22": 0.10366586538461542, "23": 0.10356570512820518, "24": 0.10837339743589747, "25": 0.10406650641025639}}}, {"candidate": {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05}, "metrics": {"cost_metric": {"1": 23.648607969284058, "2": 50.783905029296875, "3": 94.63147592544556}, "active_metric": {"1": 0.16546618647458988, "2": 0.1404561824729892, "3": 0.1386554621848739}}}, {"candidate": {"n_units_1": 774, "n_units_2": 917, "batch_size": 29, "dropout_1": 0.7778923725289609, "dropout_2": 0.7413003050986398, "learning_rate": 6.472832341968678e-05, "wd": 0.0007744951242384949}, "metrics": {"cost_metric": {"1": 50.06452512741089}, "active_metric": {"1": 0.22955092221331197}}}, {"candidate": {"n_units_1": 91, "n_units_2": 459, "batch_size": 105, "dropout_1": 0.48639033141890325, "dropout_2": 0.21324913218446714, "learning_rate": 0.00013769715715418189, "wd": 0.02017249366944585}, "metrics": {"cost_metric": {"1": 13.569249868392944}, "active_metric": {"1": 0.3023558897243108}}}, {"candidate": {"n_units_1": 673, "n_units_2": 262, "batch_size": 78, "dropout_1": 0.9510740133913004, "dropout_2": 0.3263851441475057, "learning_rate": 0.009715536539110267, "wd": 0.0002984576239921338}, "metrics": {"cost_metric": {"1": 22.84876799583435}, "active_metric": {"1": 0.7708333333333334}}}, {"candidate": {"n_units_1": 29, "n_units_2": 584, "batch_size": 41, "dropout_1": 0.6843134351325143, "dropout_2": 0.20866634010808638, "learning_rate": 0.0007531586746050282, "wd": 8.542702035112525e-08}, "metrics": {"cost_metric": {"1": 16.058557987213135}, "active_metric": {"1": 0.4832881662149955}}}, {"candidate": {"n_units_1": 529, "n_units_2": 660, "batch_size": 30, "dropout_1": 0.5490868842845377, "dropout_2": 0.7920401266415242, "learning_rate": 2.101654064946448e-06, "wd": 9.568417280766271e-08}, "metrics": {"cost_metric": {"1": 33.76100707054138}, "active_metric": {"1": 0.47597597597597596}}}, {"candidate": {"n_units_1": 828, "n_units_2": 634, "batch_size": 123, "dropout_1": 0.4257740273137951, "dropout_2": 0.23113551626017848, "learning_rate": 6.860930897860399e-06, "wd": 8.101814548564974e-08}, "metrics": {"cost_metric": {"1": 27.756996154785156}, "active_metric": {"1": 0.33835190203753884}}}, {"candidate": {"n_units_1": 635, "n_units_2": 538, "batch_size": 69, "dropout_1": 0.45999796301051993, "dropout_2": 0.1659629767457713, "learning_rate": 0.00045244429498637726, "wd": 7.074070723442468e-06}, "metrics": {"cost_metric": {"1": 26.10046887397766, "2": 53.824394941329956, "3": 80.07230305671692, "4": 106.25571393966675, "5": 133.94080090522766, "6": 170.4485969543457, "7": 200.93623304367065, "8": 229.56330180168152, "9": 257.11693382263184, "10": 286.6725299358368, "11": 319.0151889324188, "12": 354.14807295799255, "13": 386.9317948818207, "14": 419.8007171154022}, "active_metric": {"1": 0.14744363929146542, "2": 0.13586956521739135, "3": 0.12983091787439616, "4": 0.12661030595813205, "5": 0.1279186795491143, "6": 0.12328904991948475, "7": 0.11674718196457323, "8": 0.1126207729468599, "9": 0.11292270531400961, "10": 0.11513687600644118, "11": 0.11584138486312401, "12": 0.11101046698872785, "13": 0.1090982286634461, "14": 0.10738727858293073}}}], "failed_candidates": [], "pending_candidates": [{"batch_size": 20, "dropout_1": 0.41104972550281815, "dropout_2": 0.7568250476630992, "learning_rate": 0.8320700517813453, "n_units_1": 931, "n_units_2": 527, "wd": 1.9853072351534312e-08, "RESOURCE_ATTR_epoch": 1}, {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07, "RESOURCE_ATTR_epoch": 9}, {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607, "RESOURCE_ATTR_epoch": 13}, {"n_units_1": 635, "n_units_2": 538, "batch_size": 69, "dropout_1": 0.45999796301051993, "dropout_2": 0.1659629767457713, "learning_rate": 0.00045244429498637726, "wd": 7.074070723442468e-06, "RESOURCE_ATTR_epoch": 2}]}')


@pytest.mark.parametrize(
    "_model_params, _state", zip(_model_params, _state))
def test_compare_gp_model_gped_model(_model_params, _state):
    config_space = {
        'n_units_1': randint(4, 1024),
        'n_units_2': randint(4, 1024),
        'batch_size': randint(8, 128),
        'dropout_1': uniform(0, 0.99),
        'dropout_2': uniform(0, 0.99),
        'learning_rate': loguniform(1e-6, 1),
        'wd': loguniform(1e-8, 1),
        'epochs': 81,
    }

    model_params = json.loads(_model_params)
    gp_objs = build_gp_model_factory(config_space, model_params)
    configspace_ext = gp_objs['configspace_ext']
    gp_model_factory = gp_objs['model_factory']
    gped_model_factory = build_gped_model_factory(
        config_space, model_params)['model_factory']

    state = decode_state_from_old_encoding(
        enc_state=json.loads(_state), hp_ranges=configspace_ext.hp_ranges_ext)
    if state.pending_evaluations:
        # Remove pending evaluations
        state = TuningJobState(
            hp_ranges=state.hp_ranges,
            config_for_trial=state.config_for_trial,
            trials_evaluations=state.trials_evaluations,
            failed_trials=state.failed_trials)

    # We compare the two by computing the learning criterion for either
    gp_model = gp_model_factory.model(state, fit_params=False)
    gp_critval = gp_model.posterior_states[0].neg_log_likelihood().item()
    gped_model = gped_model_factory.model(state, fit_params=False)
    gped_critval = gped_model.posterior_states[0].neg_log_likelihood()
    np.testing.assert_allclose(gp_critval, gped_critval, rtol=1e-5)


@pytest.mark.parametrize(
    "_model_params, _state", zip(_model_params, _state))
def test_compare_gped_likelihood_oldnew(_model_params, _state):
    config_space = {
        'n_units_1': randint(4, 1024),
        'n_units_2': randint(4, 1024),
        'batch_size': randint(8, 128),
        'dropout_1': uniform(0, 0.99),
        'dropout_2': uniform(0, 0.99),
        'learning_rate': loguniform(1e-6, 1),
        'wd': loguniform(1e-8, 1),
        'epochs': 81,
    }

    gped_model_factory = []  # new, old
    model_params = json.loads(_model_params)
    kwargs = dict(no_fantasizing=True)
    gped_objs = build_gped_model_factory(
        config_space, model_params, **kwargs)
    configspace_ext = gped_objs['configspace_ext']
    gped_model_factory.append(gped_objs['model_factory'])
    gped_model_factory.append(build_gped_model_factory(
        config_space, model_params, use_new_code=False,
        **kwargs)['model_factory'])
    state = decode_state_from_old_encoding(
        enc_state=json.loads(_state), hp_ranges=configspace_ext.hp_ranges_ext)

    # Compare likelihoods
    likelihood = [
        factory.model(
            state,
            fit_params=False).posterior_states[0].poster_state['likelihood']
        for factory in gped_model_factory]
    for name, value in likelihood[0].items():
        if name != 'num_data':
            np.testing.assert_almost_equal(value, likelihood[1][name])


@pytest.mark.parametrize(
    "_model_params, _state", zip(_model_params, _state))
def test_compare_gped_likelihood_fantasizing_oldnew(_model_params, _state):
    config_space = {
        'n_units_1': randint(4, 1024),
        'n_units_2': randint(4, 1024),
        'batch_size': randint(8, 128),
        'dropout_1': uniform(0, 0.99),
        'dropout_2': uniform(0, 0.99),
        'learning_rate': loguniform(1e-6, 1),
        'wd': loguniform(1e-8, 1),
        'epochs': 81,
    }
    num_fantasy_samples = 10

    gped_model_factory = []  # new, old
    model_params = json.loads(_model_params)
    kwargs = dict(
        num_fantasy_samples=num_fantasy_samples,
        no_fantasizing=False)
    gped_objs = build_gped_model_factory(
        config_space, model_params, **kwargs)
    configspace_ext = gped_objs['configspace_ext']
    gped_model_factory.append(gped_objs['model_factory'])
    gped_model_factory.append(build_gped_model_factory(
        config_space, model_params, use_new_code=False,
        **kwargs)['model_factory'])
    state = decode_state_from_old_encoding(
        enc_state=json.loads(_state), hp_ranges=configspace_ext.hp_ranges_ext)

    # Compare likelihoods
    # We need to force them to use the same fantasy samples
    gped_model1 = gped_model_factory[0].model(state, fit_params=False)
    likelihood = [gped_model1.posterior_states[0].poster_state['likelihood']]
    gped_model2 = gped_model_factory[1].model_for_fantasy_samples(
        state, fantasy_samples=gped_model1.fantasy_samples)
    likelihood.append(
        gped_model2.posterior_states[0].poster_state['likelihood'])
    for name, value in likelihood[0].items():
        if name != 'num_data':
            np.testing.assert_almost_equal(value, likelihood[1][name])
