# Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License").
# You may not use this file except in compliance with the License.
# A copy of the License is located at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# or in the "license" file accompanying this file. This file is distributed
# on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
# express or implied. See the License for the specific language governing
# permissions and limitations under the License.
from typing import Dict
import json
import numpy as np
import pytest

from syne_tune.optimizer.schedulers.searchers.gp_searcher_factory import \
    gp_multifidelity_searcher_defaults, gp_multifidelity_searcher_factory
from syne_tune.optimizer.schedulers.searchers.utils.default_arguments \
    import check_and_merge_defaults
from syne_tune.optimizer.schedulers.searchers.gp_searcher_utils import \
    decode_state_from_old_encoding
from syne_tune.search_space import randint, uniform, loguniform


def _common_kwargs(configspace: Dict) -> Dict:
    return {
        'configspace': configspace,
        'max_epochs': configspace['epochs'],
        'metric': 'accuracy',
        'resource_attr': 'epoch',
        'scheduler': 'hyperband_stopping',
        'scheduler_mode': 'max',
        'debug_log': False,
        'normalize_targets': True,
    }


def build_gpiss_model_factory(
        configspace: Dict, model_params: Dict, **kwargs):
    kwargs = dict(
        _common_kwargs(configspace),
        model='gp_issm',
        issm_gamma_one=False,
        **kwargs)
    _kwargs = check_and_merge_defaults(
        kwargs, *gp_multifidelity_searcher_defaults(),
        dict_name='search_options')
    kwargs_int = gp_multifidelity_searcher_factory(**_kwargs)
    # Need to convert `model_params`
    kwargs_int['model_factory'].set_params(model_params)
    return kwargs_int


# We ran launch_sample_searcher_states.py to sample the searcher states
# used here, which runs MOBSTER (hyperband_stopping, bayesopt) with the
# mlp_fashionmnist_benchmark

_model_params = []
_state = []


_model_params.append('{"noise_variance": 0.008381548138906916, "kernel_inv_bw0": 0.004177002691678498, "kernel_inv_bw1": 0.000402494802013946, "kernel_inv_bw2": 0.00036005844016162423, "kernel_inv_bw3": 4.278552430496177, "kernel_inv_bw4": 0.38190450370225937, "kernel_inv_bw5": 0.0001674608736118065, "kernel_inv_bw6": 0.5371572608999335, "kernel_covariance_scale": 1.0487725555603677, "mean_mean_value": -0.37162308332346305, "issm_gamma": 0.0010000000000000002, "issm_alpha": -0.18364130320022903, "issm_beta": 1.1069304811899965}')
_state.append('{"candidate_evaluations": [{"candidate": {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607}, "metrics": {"cost_metric": {"1": 12.25258493423462, "2": 24.305160999298096, "3": 44.05741477012634, "4": 62.029183864593506, "5": 81.38737893104553, "6": 99.16185593605042, "7": 118.72888779640198, "8": 133.45333671569824, "9": 148.23734402656555, "10": 166.52369689941406, "11": 194.99460196495056, "12": 215.73117184638977, "13": 235.3977439403534, "14": 253.71279788017273, "15": 267.6743288040161, "16": 281.8612160682678, "17": 296.0602250099182, "18": 310.0040330886841, "19": 324.75612902641296, "20": 344.674284696579, "21": 360.0983910560608, "22": 375.9487638473511, "23": 395.81145191192627, "24": 411.6494069099426, "25": 426.79202795028687, "26": 448.74489879608154, "27": 464.90988278388977, "28": 480.28413486480713, "29": 494.5631868839264, "30": 510.31515073776245, "31": 527.6290948390961, "32": 542.7905468940735, "33": 558.1524910926819, "34": 572.6776859760284, "35": 588.3533399105072}, "active_metric": {"1": 0.4978924126856684, "2": 0.3896025692492975, "3": 0.3546768366118025, "4": 0.33289843436370936, "5": 0.3259735046166198, "6": 0.30971497390606184, "7": 0.29626655961461257, "8": 0.2863307908470494, "9": 0.2753914090726616, "10": 0.26455238859895625, "11": 0.25491770373344036, "12": 0.2485949417904456, "13": 0.24678843837816133, "14": 0.23996386993175434, "15": 0.2332396627860297, "16": 0.23143315937374553, "17": 0.22390606182256123, "18": 0.22350461661983134, "19": 0.2195905258932156, "20": 0.217482938578884, "21": 0.21176234443998398, "22": 0.2106583701324769, "23": 0.20764753111200318, "24": 0.2052388598956243, "25": 0.20102368526696102, "26": 0.1963067041348856, "27": 0.1949016459253312, "28": 0.19409875551987155, "29": 0.1904857486953031, "30": 0.18747490967482938, "31": 0.18516659975913285, "32": 0.18446407065435566, "33": 0.18115214773183463, "34": 0.18004817342432755, "35": 0.17844239261340822}}}, {"candidate": {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05}, "metrics": {"cost_metric": {"1": 18.50484609603882, "2": 45.980664014816284, "3": 96.58328175544739, "4": 148.91678476333618, "5": 229.4046459197998, "6": 300.6477417945862, "7": 383.9124698638916, "8": 473.07399010658264, "9": 555.2527649402618}, "active_metric": {"1": 0.15496198479391754, "2": 0.1519607843137255, "3": 0.13645458183273307, "4": 0.14435774309723892, "5": 0.13225290116046418, "6": 0.1253501400560224, "7": 0.12735094037615047, "8": 0.13265306122448983, "9": 0.12104841936774713}}}, {"candidate": {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07}, "metrics": {"cost_metric": {"1": 18.432047128677368}, "active_metric": {"1": 0.16786858974358976}}}, {"candidate": {"n_units_1": 91, "n_units_2": 459, "batch_size": 105, "dropout_1": 0.48639033141890325, "dropout_2": 0.21324913218446714, "learning_rate": 0.00013769715715418189, "wd": 0.02017249366944585}, "metrics": {"cost_metric": {"1": 17.439072132110596}, "active_metric": {"1": 0.3006516290726817}}}, {"candidate": {"n_units_1": 774, "n_units_2": 917, "batch_size": 29, "dropout_1": 0.7778923725289609, "dropout_2": 0.7413003050986398, "learning_rate": 6.472832341968678e-05, "wd": 0.0007744951242384949}, "metrics": {"cost_metric": {"1": 44.07283306121826}, "active_metric": {"1": 0.23085404971932644}}}, {"candidate": {"n_units_1": 673, "n_units_2": 262, "batch_size": 78, "dropout_1": 0.9510740133913004, "dropout_2": 0.3263851441475057, "learning_rate": 0.009715536539110267, "wd": 0.0002984576239921338}, "metrics": {"cost_metric": {"1": 29.966220140457153}, "active_metric": {"1": 0.6366185897435898}}}, {"candidate": {"n_units_1": 672, "n_units_2": 820, "batch_size": 108, "dropout_1": 0.6443283647430158, "dropout_2": 0.8194904484310889, "learning_rate": 9.196365243521935e-05, "wd": 0.002536625472111785}, "metrics": {"cost_metric": {"1": 32.458306074142456}, "active_metric": {"1": 0.24486714975845414}}}, {"candidate": {"n_units_1": 688, "n_units_2": 597, "batch_size": 123, "dropout_1": 0.7829512576762913, "dropout_2": 0.2834197685256876, "learning_rate": 0.1784738929251937, "wd": 4.489784182359429e-08}, "metrics": {"cost_metric": {"1": 30.61675500869751}, "active_metric": {"1": 0.8976211984342066}}}, {"candidate": {"n_units_1": 501, "n_units_2": 601, "batch_size": 34, "dropout_1": 0.7410256603874262, "dropout_2": 0.046625361151571336, "learning_rate": 0.07937041160202492, "wd": 8.340962845965557e-07}, "metrics": {"cost_metric": {"1": 38.63721990585327}, "active_metric": {"1": 0.904561824729892}}}, {"candidate": {"n_units_1": 1024, "n_units_2": 1002, "batch_size": 116, "dropout_1": 0.0366350257842321, "dropout_2": 0.6883751950302733, "learning_rate": 0.0003133897834907133, "wd": 1.1611672813117278e-08}, "metrics": {"cost_metric": {"1": 44.914865016937256, "2": 95.70968389511108, "3": 134.2296760082245, "4": 167.57774996757507, "5": 205.92636585235596, "6": 242.58663892745972, "7": 283.2623338699341, "8": 326.5033459663391, "9": 360.76255893707275, "10": 398.4191679954529, "11": 434.4982190132141}, "active_metric": {"1": 0.15477145148356053, "2": 0.13502405773857262, "3": 0.1245990376904571, "4": 0.12840817963111473, "5": 0.12219326383319973, "6": 0.11778267842822776, "7": 0.1133720930232558, "8": 0.11166800320769843, "9": 0.10976343223736973, "10": 0.1067562149157979, "11": 0.10555332798716921}}}, {"candidate": {"n_units_1": 1024, "n_units_2": 707, "batch_size": 89, "dropout_1": 0.19654676887125966, "dropout_2": 0.8682666451901773, "learning_rate": 0.00031134631996358774, "wd": 1e-08}, "metrics": {"cost_metric": {"1": 50.98606991767883, "2": 92.16159892082214, "3": 123.69115900993347, "4": 154.21311402320862, "5": 192.06259202957153, "6": 227.72296023368835, "7": 263.6407790184021, "8": 304.7051441669464, "9": 336.6670799255371, "10": 372.1759181022644, "11": 405.45882201194763}, "active_metric": {"1": 0.1700441412520064, "2": 0.1407504012841091, "3": 0.1359349919743178, "4": 0.131922150882825, "5": 0.1220906902086677, "6": 0.1213884430176565, "7": 0.1182784911717496, "8": 0.120284911717496, "9": 0.1221910112359551, "10": 0.1134630818619583, "11": 0.1151685393258427}}}], "failed_candidates": [], "pending_candidates": [{"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05, "RESOURCE_ATTR_epoch": 10}, {"n_units_1": 1024, "n_units_2": 707, "batch_size": 89, "dropout_1": 0.19654676887125966, "dropout_2": 0.8682666451901773, "learning_rate": 0.00031134631996358774, "wd": 1e-08, "RESOURCE_ATTR_epoch": 12}, {"n_units_1": 1024, "n_units_2": 1002, "batch_size": 116, "dropout_1": 0.0366350257842321, "dropout_2": 0.6883751950302733, "learning_rate": 0.0003133897834907133, "wd": 1.1611672813117278e-08, "RESOURCE_ATTR_epoch": 12}, {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607, "RESOURCE_ATTR_epoch": 36}]}')
# elapsed_time = 595.700856924057
# num_observations = 73
# num_configs = 11

_model_params.append('{"noise_variance": 0.008381548138906916, "kernel_inv_bw0": 0.004177002691678498, "kernel_inv_bw1": 0.000402494802013946, "kernel_inv_bw2": 0.00036005844016162423, "kernel_inv_bw3": 4.278552430496177, "kernel_inv_bw4": 0.38190450370225937, "kernel_inv_bw5": 0.0001674608736118065, "kernel_inv_bw6": 0.5371572608999335, "kernel_covariance_scale": 1.0487725555603677, "mean_mean_value": -0.37162308332346305, "issm_gamma": 0.0010000000000000002, "issm_alpha": -0.18364130320022903, "issm_beta": 1.1069304811899965}')
_state.append('{"candidate_evaluations": [{"candidate": {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607}, "metrics": {"cost_metric": {"1": 12.25258493423462, "2": 24.305160999298096, "3": 44.05741477012634, "4": 62.029183864593506, "5": 81.38737893104553, "6": 99.16185593605042, "7": 118.72888779640198, "8": 133.45333671569824, "9": 148.23734402656555, "10": 166.52369689941406, "11": 194.99460196495056, "12": 215.73117184638977, "13": 235.3977439403534, "14": 253.71279788017273, "15": 267.6743288040161, "16": 281.8612160682678, "17": 296.0602250099182, "18": 310.0040330886841, "19": 324.75612902641296, "20": 344.674284696579, "21": 360.0983910560608, "22": 375.9487638473511, "23": 395.81145191192627, "24": 411.6494069099426, "25": 426.79202795028687, "26": 448.74489879608154, "27": 464.90988278388977, "28": 480.28413486480713, "29": 494.5631868839264, "30": 510.31515073776245, "31": 527.6290948390961, "32": 542.7905468940735, "33": 558.1524910926819, "34": 572.6776859760284, "35": 588.3533399105072}, "active_metric": {"1": 0.4978924126856684, "2": 0.3896025692492975, "3": 0.3546768366118025, "4": 0.33289843436370936, "5": 0.3259735046166198, "6": 0.30971497390606184, "7": 0.29626655961461257, "8": 0.2863307908470494, "9": 0.2753914090726616, "10": 0.26455238859895625, "11": 0.25491770373344036, "12": 0.2485949417904456, "13": 0.24678843837816133, "14": 0.23996386993175434, "15": 0.2332396627860297, "16": 0.23143315937374553, "17": 0.22390606182256123, "18": 0.22350461661983134, "19": 0.2195905258932156, "20": 0.217482938578884, "21": 0.21176234443998398, "22": 0.2106583701324769, "23": 0.20764753111200318, "24": 0.2052388598956243, "25": 0.20102368526696102, "26": 0.1963067041348856, "27": 0.1949016459253312, "28": 0.19409875551987155, "29": 0.1904857486953031, "30": 0.18747490967482938, "31": 0.18516659975913285, "32": 0.18446407065435566, "33": 0.18115214773183463, "34": 0.18004817342432755, "35": 0.17844239261340822}}}, {"candidate": {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05}, "metrics": {"cost_metric": {"1": 18.50484609603882, "2": 45.980664014816284, "3": 96.58328175544739, "4": 148.91678476333618, "5": 229.4046459197998, "6": 300.6477417945862, "7": 383.9124698638916, "8": 473.07399010658264, "9": 555.2527649402618}, "active_metric": {"1": 0.15496198479391754, "2": 0.1519607843137255, "3": 0.13645458183273307, "4": 0.14435774309723892, "5": 0.13225290116046418, "6": 0.1253501400560224, "7": 0.12735094037615047, "8": 0.13265306122448983, "9": 0.12104841936774713}}}, {"candidate": {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07}, "metrics": {"cost_metric": {"1": 18.432047128677368}, "active_metric": {"1": 0.16786858974358976}}}, {"candidate": {"n_units_1": 91, "n_units_2": 459, "batch_size": 105, "dropout_1": 0.48639033141890325, "dropout_2": 0.21324913218446714, "learning_rate": 0.00013769715715418189, "wd": 0.02017249366944585}, "metrics": {"cost_metric": {"1": 17.439072132110596}, "active_metric": {"1": 0.3006516290726817}}}, {"candidate": {"n_units_1": 774, "n_units_2": 917, "batch_size": 29, "dropout_1": 0.7778923725289609, "dropout_2": 0.7413003050986398, "learning_rate": 6.472832341968678e-05, "wd": 0.0007744951242384949}, "metrics": {"cost_metric": {"1": 44.07283306121826}, "active_metric": {"1": 0.23085404971932644}}}, {"candidate": {"n_units_1": 673, "n_units_2": 262, "batch_size": 78, "dropout_1": 0.9510740133913004, "dropout_2": 0.3263851441475057, "learning_rate": 0.009715536539110267, "wd": 0.0002984576239921338}, "metrics": {"cost_metric": {"1": 29.966220140457153}, "active_metric": {"1": 0.6366185897435898}}}, {"candidate": {"n_units_1": 672, "n_units_2": 820, "batch_size": 108, "dropout_1": 0.6443283647430158, "dropout_2": 0.8194904484310889, "learning_rate": 9.196365243521935e-05, "wd": 0.002536625472111785}, "metrics": {"cost_metric": {"1": 32.458306074142456}, "active_metric": {"1": 0.24486714975845414}}}, {"candidate": {"n_units_1": 688, "n_units_2": 597, "batch_size": 123, "dropout_1": 0.7829512576762913, "dropout_2": 0.2834197685256876, "learning_rate": 0.1784738929251937, "wd": 4.489784182359429e-08}, "metrics": {"cost_metric": {"1": 30.61675500869751}, "active_metric": {"1": 0.8976211984342066}}}, {"candidate": {"n_units_1": 501, "n_units_2": 601, "batch_size": 34, "dropout_1": 0.7410256603874262, "dropout_2": 0.046625361151571336, "learning_rate": 0.07937041160202492, "wd": 8.340962845965557e-07}, "metrics": {"cost_metric": {"1": 38.63721990585327}, "active_metric": {"1": 0.904561824729892}}}, {"candidate": {"n_units_1": 1024, "n_units_2": 1002, "batch_size": 116, "dropout_1": 0.0366350257842321, "dropout_2": 0.6883751950302733, "learning_rate": 0.0003133897834907133, "wd": 1.1611672813117278e-08}, "metrics": {"cost_metric": {"1": 44.914865016937256, "2": 95.70968389511108, "3": 134.2296760082245, "4": 167.57774996757507, "5": 205.92636585235596, "6": 242.58663892745972, "7": 283.2623338699341, "8": 326.5033459663391, "9": 360.76255893707275, "10": 398.4191679954529, "11": 434.4982190132141}, "active_metric": {"1": 0.15477145148356053, "2": 0.13502405773857262, "3": 0.1245990376904571, "4": 0.12840817963111473, "5": 0.12219326383319973, "6": 0.11778267842822776, "7": 0.1133720930232558, "8": 0.11166800320769843, "9": 0.10976343223736973, "10": 0.1067562149157979, "11": 0.10555332798716921}}}, {"candidate": {"n_units_1": 1024, "n_units_2": 707, "batch_size": 89, "dropout_1": 0.19654676887125966, "dropout_2": 0.8682666451901773, "learning_rate": 0.00031134631996358774, "wd": 1e-08}, "metrics": {"cost_metric": {"1": 50.98606991767883, "2": 92.16159892082214, "3": 123.69115900993347, "4": 154.21311402320862, "5": 192.06259202957153, "6": 227.72296023368835, "7": 263.6407790184021, "8": 304.7051441669464, "9": 336.6670799255371, "10": 372.1759181022644, "11": 405.45882201194763}, "active_metric": {"1": 0.1700441412520064, "2": 0.1407504012841091, "3": 0.1359349919743178, "4": 0.131922150882825, "5": 0.1220906902086677, "6": 0.1213884430176565, "7": 0.1182784911717496, "8": 0.120284911717496, "9": 0.1221910112359551, "10": 0.1134630818619583, "11": 0.1151685393258427}}}], "failed_candidates": [], "pending_candidates": [{"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05, "RESOURCE_ATTR_epoch": 9}, {"n_units_1": 1024, "n_units_2": 707, "batch_size": 89, "dropout_1": 0.19654676887125966, "dropout_2": 0.8682666451901773, "learning_rate": 0.00031134631996358774, "wd": 1e-08, "RESOURCE_ATTR_epoch": 10}, {"n_units_1": 1024, "n_units_2": 1002, "batch_size": 116, "dropout_1": 0.0366350257842321, "dropout_2": 0.6883751950302733, "learning_rate": 0.0003133897834907133, "wd": 1.1611672813117278e-08, "RESOURCE_ATTR_epoch": 10}, {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607, "RESOURCE_ATTR_epoch": 31}]}')
# elapsed_time = 520.2518529891968
# num_observations = 63
# num_configs = 11

_model_params.append('{"noise_variance": 0.008381548138906916, "kernel_inv_bw0": 0.004177002691678498, "kernel_inv_bw1": 0.000402494802013946, "kernel_inv_bw2": 0.00036005844016162423, "kernel_inv_bw3": 4.278552430496177, "kernel_inv_bw4": 0.38190450370225937, "kernel_inv_bw5": 0.0001674608736118065, "kernel_inv_bw6": 0.5371572608999335, "kernel_covariance_scale": 1.0487725555603677, "mean_mean_value": -0.37162308332346305, "issm_gamma": 0.0010000000000000002, "issm_alpha": -0.18364130320022903, "issm_beta": 1.1069304811899965}')
_state.append('{"candidate_evaluations": [{"candidate": {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607}, "metrics": {"cost_metric": {"1": 12.25258493423462, "2": 24.305160999298096, "3": 44.05741477012634, "4": 62.029183864593506, "5": 81.38737893104553, "6": 99.16185593605042, "7": 118.72888779640198, "8": 133.45333671569824, "9": 148.23734402656555, "10": 166.52369689941406, "11": 194.99460196495056, "12": 215.73117184638977, "13": 235.3977439403534, "14": 253.71279788017273, "15": 267.6743288040161, "16": 281.8612160682678, "17": 296.0602250099182, "18": 310.0040330886841, "19": 324.75612902641296, "20": 344.674284696579, "21": 360.0983910560608, "22": 375.9487638473511, "23": 395.81145191192627, "24": 411.6494069099426, "25": 426.79202795028687, "26": 448.74489879608154, "27": 464.90988278388977, "28": 480.28413486480713, "29": 494.5631868839264, "30": 510.31515073776245, "31": 527.6290948390961, "32": 542.7905468940735, "33": 558.1524910926819, "34": 572.6776859760284, "35": 588.3533399105072}, "active_metric": {"1": 0.4978924126856684, "2": 0.3896025692492975, "3": 0.3546768366118025, "4": 0.33289843436370936, "5": 0.3259735046166198, "6": 0.30971497390606184, "7": 0.29626655961461257, "8": 0.2863307908470494, "9": 0.2753914090726616, "10": 0.26455238859895625, "11": 0.25491770373344036, "12": 0.2485949417904456, "13": 0.24678843837816133, "14": 0.23996386993175434, "15": 0.2332396627860297, "16": 0.23143315937374553, "17": 0.22390606182256123, "18": 0.22350461661983134, "19": 0.2195905258932156, "20": 0.217482938578884, "21": 0.21176234443998398, "22": 0.2106583701324769, "23": 0.20764753111200318, "24": 0.2052388598956243, "25": 0.20102368526696102, "26": 0.1963067041348856, "27": 0.1949016459253312, "28": 0.19409875551987155, "29": 0.1904857486953031, "30": 0.18747490967482938, "31": 0.18516659975913285, "32": 0.18446407065435566, "33": 0.18115214773183463, "34": 0.18004817342432755, "35": 0.17844239261340822}}}, {"candidate": {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05}, "metrics": {"cost_metric": {"1": 18.50484609603882, "2": 45.980664014816284, "3": 96.58328175544739, "4": 148.91678476333618, "5": 229.4046459197998, "6": 300.6477417945862, "7": 383.9124698638916, "8": 473.07399010658264, "9": 555.2527649402618}, "active_metric": {"1": 0.15496198479391754, "2": 0.1519607843137255, "3": 0.13645458183273307, "4": 0.14435774309723892, "5": 0.13225290116046418, "6": 0.1253501400560224, "7": 0.12735094037615047, "8": 0.13265306122448983, "9": 0.12104841936774713}}}, {"candidate": {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07}, "metrics": {"cost_metric": {"1": 18.432047128677368}, "active_metric": {"1": 0.16786858974358976}}}, {"candidate": {"n_units_1": 91, "n_units_2": 459, "batch_size": 105, "dropout_1": 0.48639033141890325, "dropout_2": 0.21324913218446714, "learning_rate": 0.00013769715715418189, "wd": 0.02017249366944585}, "metrics": {"cost_metric": {"1": 17.439072132110596}, "active_metric": {"1": 0.3006516290726817}}}, {"candidate": {"n_units_1": 774, "n_units_2": 917, "batch_size": 29, "dropout_1": 0.7778923725289609, "dropout_2": 0.7413003050986398, "learning_rate": 6.472832341968678e-05, "wd": 0.0007744951242384949}, "metrics": {"cost_metric": {"1": 44.07283306121826}, "active_metric": {"1": 0.23085404971932644}}}, {"candidate": {"n_units_1": 673, "n_units_2": 262, "batch_size": 78, "dropout_1": 0.9510740133913004, "dropout_2": 0.3263851441475057, "learning_rate": 0.009715536539110267, "wd": 0.0002984576239921338}, "metrics": {"cost_metric": {"1": 29.966220140457153}, "active_metric": {"1": 0.6366185897435898}}}, {"candidate": {"n_units_1": 672, "n_units_2": 820, "batch_size": 108, "dropout_1": 0.6443283647430158, "dropout_2": 0.8194904484310889, "learning_rate": 9.196365243521935e-05, "wd": 0.002536625472111785}, "metrics": {"cost_metric": {"1": 32.458306074142456}, "active_metric": {"1": 0.24486714975845414}}}, {"candidate": {"n_units_1": 688, "n_units_2": 597, "batch_size": 123, "dropout_1": 0.7829512576762913, "dropout_2": 0.2834197685256876, "learning_rate": 0.1784738929251937, "wd": 4.489784182359429e-08}, "metrics": {"cost_metric": {"1": 30.61675500869751}, "active_metric": {"1": 0.8976211984342066}}}, {"candidate": {"n_units_1": 501, "n_units_2": 601, "batch_size": 34, "dropout_1": 0.7410256603874262, "dropout_2": 0.046625361151571336, "learning_rate": 0.07937041160202492, "wd": 8.340962845965557e-07}, "metrics": {"cost_metric": {"1": 38.63721990585327}, "active_metric": {"1": 0.904561824729892}}}, {"candidate": {"n_units_1": 1024, "n_units_2": 1002, "batch_size": 116, "dropout_1": 0.0366350257842321, "dropout_2": 0.6883751950302733, "learning_rate": 0.0003133897834907133, "wd": 1.1611672813117278e-08}, "metrics": {"cost_metric": {"1": 44.914865016937256, "2": 95.70968389511108, "3": 134.2296760082245, "4": 167.57774996757507, "5": 205.92636585235596, "6": 242.58663892745972, "7": 283.2623338699341, "8": 326.5033459663391, "9": 360.76255893707275, "10": 398.4191679954529, "11": 434.4982190132141}, "active_metric": {"1": 0.15477145148356053, "2": 0.13502405773857262, "3": 0.1245990376904571, "4": 0.12840817963111473, "5": 0.12219326383319973, "6": 0.11778267842822776, "7": 0.1133720930232558, "8": 0.11166800320769843, "9": 0.10976343223736973, "10": 0.1067562149157979, "11": 0.10555332798716921}}}, {"candidate": {"n_units_1": 1024, "n_units_2": 707, "batch_size": 89, "dropout_1": 0.19654676887125966, "dropout_2": 0.8682666451901773, "learning_rate": 0.00031134631996358774, "wd": 1e-08}, "metrics": {"cost_metric": {"1": 50.98606991767883, "2": 92.16159892082214, "3": 123.69115900993347, "4": 154.21311402320862, "5": 192.06259202957153, "6": 227.72296023368835, "7": 263.6407790184021, "8": 304.7051441669464, "9": 336.6670799255371, "10": 372.1759181022644, "11": 405.45882201194763}, "active_metric": {"1": 0.1700441412520064, "2": 0.1407504012841091, "3": 0.1359349919743178, "4": 0.131922150882825, "5": 0.1220906902086677, "6": 0.1213884430176565, "7": 0.1182784911717496, "8": 0.120284911717496, "9": 0.1221910112359551, "10": 0.1134630818619583, "11": 0.1151685393258427}}}], "failed_candidates": [], "pending_candidates": [{"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05, "RESOURCE_ATTR_epoch": 8}, {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607, "RESOURCE_ATTR_epoch": 27}, {"n_units_1": 1024, "n_units_2": 1002, "batch_size": 116, "dropout_1": 0.0366350257842321, "dropout_2": 0.6883751950302733, "learning_rate": 0.0003133897834907133, "wd": 1.1611672813117278e-08, "RESOURCE_ATTR_epoch": 9}, {"n_units_1": 1024, "n_units_2": 707, "batch_size": 89, "dropout_1": 0.19654676887125966, "dropout_2": 0.8682666451901773, "learning_rate": 0.00031134631996358774, "wd": 1e-08, "RESOURCE_ATTR_epoch": 9}]}')
# elapsed_time = 469.9041178226471
# num_observations = 56
# num_configs = 11

_model_params.append('{"noise_variance": 0.008381548138906916, "kernel_inv_bw0": 0.004177002691678498, "kernel_inv_bw1": 0.000402494802013946, "kernel_inv_bw2": 0.00036005844016162423, "kernel_inv_bw3": 4.278552430496177, "kernel_inv_bw4": 0.38190450370225937, "kernel_inv_bw5": 0.0001674608736118065, "kernel_inv_bw6": 0.5371572608999335, "kernel_covariance_scale": 1.0487725555603677, "mean_mean_value": -0.37162308332346305, "issm_gamma": 0.0010000000000000002, "issm_alpha": -0.18364130320022903, "issm_beta": 1.1069304811899965}')
_state.append('{"candidate_evaluations": [{"candidate": {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607}, "metrics": {"cost_metric": {"1": 12.25258493423462, "2": 24.305160999298096, "3": 44.05741477012634, "4": 62.029183864593506, "5": 81.38737893104553, "6": 99.16185593605042, "7": 118.72888779640198, "8": 133.45333671569824, "9": 148.23734402656555, "10": 166.52369689941406, "11": 194.99460196495056, "12": 215.73117184638977, "13": 235.3977439403534, "14": 253.71279788017273, "15": 267.6743288040161, "16": 281.8612160682678, "17": 296.0602250099182, "18": 310.0040330886841, "19": 324.75612902641296, "20": 344.674284696579, "21": 360.0983910560608, "22": 375.9487638473511, "23": 395.81145191192627, "24": 411.6494069099426, "25": 426.79202795028687, "26": 448.74489879608154, "27": 464.90988278388977, "28": 480.28413486480713, "29": 494.5631868839264, "30": 510.31515073776245, "31": 527.6290948390961, "32": 542.7905468940735, "33": 558.1524910926819, "34": 572.6776859760284, "35": 588.3533399105072}, "active_metric": {"1": 0.4978924126856684, "2": 0.3896025692492975, "3": 0.3546768366118025, "4": 0.33289843436370936, "5": 0.3259735046166198, "6": 0.30971497390606184, "7": 0.29626655961461257, "8": 0.2863307908470494, "9": 0.2753914090726616, "10": 0.26455238859895625, "11": 0.25491770373344036, "12": 0.2485949417904456, "13": 0.24678843837816133, "14": 0.23996386993175434, "15": 0.2332396627860297, "16": 0.23143315937374553, "17": 0.22390606182256123, "18": 0.22350461661983134, "19": 0.2195905258932156, "20": 0.217482938578884, "21": 0.21176234443998398, "22": 0.2106583701324769, "23": 0.20764753111200318, "24": 0.2052388598956243, "25": 0.20102368526696102, "26": 0.1963067041348856, "27": 0.1949016459253312, "28": 0.19409875551987155, "29": 0.1904857486953031, "30": 0.18747490967482938, "31": 0.18516659975913285, "32": 0.18446407065435566, "33": 0.18115214773183463, "34": 0.18004817342432755, "35": 0.17844239261340822}}}, {"candidate": {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05}, "metrics": {"cost_metric": {"1": 18.50484609603882, "2": 45.980664014816284, "3": 96.58328175544739, "4": 148.91678476333618, "5": 229.4046459197998, "6": 300.6477417945862, "7": 383.9124698638916, "8": 473.07399010658264, "9": 555.2527649402618}, "active_metric": {"1": 0.15496198479391754, "2": 0.1519607843137255, "3": 0.13645458183273307, "4": 0.14435774309723892, "5": 0.13225290116046418, "6": 0.1253501400560224, "7": 0.12735094037615047, "8": 0.13265306122448983, "9": 0.12104841936774713}}}, {"candidate": {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07}, "metrics": {"cost_metric": {"1": 18.432047128677368}, "active_metric": {"1": 0.16786858974358976}}}, {"candidate": {"n_units_1": 91, "n_units_2": 459, "batch_size": 105, "dropout_1": 0.48639033141890325, "dropout_2": 0.21324913218446714, "learning_rate": 0.00013769715715418189, "wd": 0.02017249366944585}, "metrics": {"cost_metric": {"1": 17.439072132110596}, "active_metric": {"1": 0.3006516290726817}}}, {"candidate": {"n_units_1": 774, "n_units_2": 917, "batch_size": 29, "dropout_1": 0.7778923725289609, "dropout_2": 0.7413003050986398, "learning_rate": 6.472832341968678e-05, "wd": 0.0007744951242384949}, "metrics": {"cost_metric": {"1": 44.07283306121826}, "active_metric": {"1": 0.23085404971932644}}}, {"candidate": {"n_units_1": 673, "n_units_2": 262, "batch_size": 78, "dropout_1": 0.9510740133913004, "dropout_2": 0.3263851441475057, "learning_rate": 0.009715536539110267, "wd": 0.0002984576239921338}, "metrics": {"cost_metric": {"1": 29.966220140457153}, "active_metric": {"1": 0.6366185897435898}}}, {"candidate": {"n_units_1": 672, "n_units_2": 820, "batch_size": 108, "dropout_1": 0.6443283647430158, "dropout_2": 0.8194904484310889, "learning_rate": 9.196365243521935e-05, "wd": 0.002536625472111785}, "metrics": {"cost_metric": {"1": 32.458306074142456}, "active_metric": {"1": 0.24486714975845414}}}, {"candidate": {"n_units_1": 688, "n_units_2": 597, "batch_size": 123, "dropout_1": 0.7829512576762913, "dropout_2": 0.2834197685256876, "learning_rate": 0.1784738929251937, "wd": 4.489784182359429e-08}, "metrics": {"cost_metric": {"1": 30.61675500869751}, "active_metric": {"1": 0.8976211984342066}}}, {"candidate": {"n_units_1": 501, "n_units_2": 601, "batch_size": 34, "dropout_1": 0.7410256603874262, "dropout_2": 0.046625361151571336, "learning_rate": 0.07937041160202492, "wd": 8.340962845965557e-07}, "metrics": {"cost_metric": {"1": 38.63721990585327}, "active_metric": {"1": 0.904561824729892}}}, {"candidate": {"n_units_1": 1024, "n_units_2": 1002, "batch_size": 116, "dropout_1": 0.0366350257842321, "dropout_2": 0.6883751950302733, "learning_rate": 0.0003133897834907133, "wd": 1.1611672813117278e-08}, "metrics": {"cost_metric": {"1": 44.914865016937256, "2": 95.70968389511108, "3": 134.2296760082245, "4": 167.57774996757507, "5": 205.92636585235596, "6": 242.58663892745972, "7": 283.2623338699341, "8": 326.5033459663391, "9": 360.76255893707275, "10": 398.4191679954529, "11": 434.4982190132141}, "active_metric": {"1": 0.15477145148356053, "2": 0.13502405773857262, "3": 0.1245990376904571, "4": 0.12840817963111473, "5": 0.12219326383319973, "6": 0.11778267842822776, "7": 0.1133720930232558, "8": 0.11166800320769843, "9": 0.10976343223736973, "10": 0.1067562149157979, "11": 0.10555332798716921}}}, {"candidate": {"n_units_1": 1024, "n_units_2": 707, "batch_size": 89, "dropout_1": 0.19654676887125966, "dropout_2": 0.8682666451901773, "learning_rate": 0.00031134631996358774, "wd": 1e-08}, "metrics": {"cost_metric": {"1": 50.98606991767883, "2": 92.16159892082214, "3": 123.69115900993347, "4": 154.21311402320862, "5": 192.06259202957153, "6": 227.72296023368835, "7": 263.6407790184021, "8": 304.7051441669464, "9": 336.6670799255371, "10": 372.1759181022644, "11": 405.45882201194763}, "active_metric": {"1": 0.1700441412520064, "2": 0.1407504012841091, "3": 0.1359349919743178, "4": 0.131922150882825, "5": 0.1220906902086677, "6": 0.1213884430176565, "7": 0.1182784911717496, "8": 0.120284911717496, "9": 0.1221910112359551, "10": 0.1134630818619583, "11": 0.1151685393258427}}}], "failed_candidates": [], "pending_candidates": [{"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05, "RESOURCE_ATTR_epoch": 7}, {"n_units_1": 1024, "n_units_2": 707, "batch_size": 89, "dropout_1": 0.19654676887125966, "dropout_2": 0.8682666451901773, "learning_rate": 0.00031134631996358774, "wd": 1e-08, "RESOURCE_ATTR_epoch": 5}, {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607, "RESOURCE_ATTR_epoch": 20}, {"n_units_1": 1024, "n_units_2": 1002, "batch_size": 116, "dropout_1": 0.0366350257842321, "dropout_2": 0.6883751950302733, "learning_rate": 0.0003133897834907133, "wd": 1.1611672813117278e-08, "RESOURCE_ATTR_epoch": 6}]}')
# elapsed_time = 349.2686309814453
# num_observations = 41
# num_configs = 11

_model_params.append('{"noise_variance": 0.008381548138906916, "kernel_inv_bw0": 0.004177002691678498, "kernel_inv_bw1": 0.000402494802013946, "kernel_inv_bw2": 0.00036005844016162423, "kernel_inv_bw3": 4.278552430496177, "kernel_inv_bw4": 0.38190450370225937, "kernel_inv_bw5": 0.0001674608736118065, "kernel_inv_bw6": 0.5371572608999335, "kernel_covariance_scale": 1.0487725555603677, "mean_mean_value": -0.37162308332346305, "issm_gamma": 0.0010000000000000002, "issm_alpha": -0.18364130320022903, "issm_beta": 1.1069304811899965}')
_state.append('{"candidate_evaluations": [{"candidate": {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607}, "metrics": {"cost_metric": {"1": 12.25258493423462, "2": 24.305160999298096, "3": 44.05741477012634, "4": 62.029183864593506, "5": 81.38737893104553, "6": 99.16185593605042, "7": 118.72888779640198, "8": 133.45333671569824, "9": 148.23734402656555, "10": 166.52369689941406, "11": 194.99460196495056, "12": 215.73117184638977, "13": 235.3977439403534, "14": 253.71279788017273, "15": 267.6743288040161, "16": 281.8612160682678, "17": 296.0602250099182, "18": 310.0040330886841, "19": 324.75612902641296, "20": 344.674284696579, "21": 360.0983910560608, "22": 375.9487638473511, "23": 395.81145191192627, "24": 411.6494069099426, "25": 426.79202795028687, "26": 448.74489879608154, "27": 464.90988278388977, "28": 480.28413486480713, "29": 494.5631868839264, "30": 510.31515073776245, "31": 527.6290948390961, "32": 542.7905468940735, "33": 558.1524910926819, "34": 572.6776859760284, "35": 588.3533399105072}, "active_metric": {"1": 0.4978924126856684, "2": 0.3896025692492975, "3": 0.3546768366118025, "4": 0.33289843436370936, "5": 0.3259735046166198, "6": 0.30971497390606184, "7": 0.29626655961461257, "8": 0.2863307908470494, "9": 0.2753914090726616, "10": 0.26455238859895625, "11": 0.25491770373344036, "12": 0.2485949417904456, "13": 0.24678843837816133, "14": 0.23996386993175434, "15": 0.2332396627860297, "16": 0.23143315937374553, "17": 0.22390606182256123, "18": 0.22350461661983134, "19": 0.2195905258932156, "20": 0.217482938578884, "21": 0.21176234443998398, "22": 0.2106583701324769, "23": 0.20764753111200318, "24": 0.2052388598956243, "25": 0.20102368526696102, "26": 0.1963067041348856, "27": 0.1949016459253312, "28": 0.19409875551987155, "29": 0.1904857486953031, "30": 0.18747490967482938, "31": 0.18516659975913285, "32": 0.18446407065435566, "33": 0.18115214773183463, "34": 0.18004817342432755, "35": 0.17844239261340822}}}, {"candidate": {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05}, "metrics": {"cost_metric": {"1": 18.50484609603882, "2": 45.980664014816284, "3": 96.58328175544739, "4": 148.91678476333618, "5": 229.4046459197998, "6": 300.6477417945862, "7": 383.9124698638916, "8": 473.07399010658264, "9": 555.2527649402618}, "active_metric": {"1": 0.15496198479391754, "2": 0.1519607843137255, "3": 0.13645458183273307, "4": 0.14435774309723892, "5": 0.13225290116046418, "6": 0.1253501400560224, "7": 0.12735094037615047, "8": 0.13265306122448983, "9": 0.12104841936774713}}}, {"candidate": {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07}, "metrics": {"cost_metric": {"1": 18.432047128677368}, "active_metric": {"1": 0.16786858974358976}}}, {"candidate": {"n_units_1": 91, "n_units_2": 459, "batch_size": 105, "dropout_1": 0.48639033141890325, "dropout_2": 0.21324913218446714, "learning_rate": 0.00013769715715418189, "wd": 0.02017249366944585}, "metrics": {"cost_metric": {"1": 17.439072132110596}, "active_metric": {"1": 0.3006516290726817}}}, {"candidate": {"n_units_1": 774, "n_units_2": 917, "batch_size": 29, "dropout_1": 0.7778923725289609, "dropout_2": 0.7413003050986398, "learning_rate": 6.472832341968678e-05, "wd": 0.0007744951242384949}, "metrics": {"cost_metric": {"1": 44.07283306121826}, "active_metric": {"1": 0.23085404971932644}}}, {"candidate": {"n_units_1": 673, "n_units_2": 262, "batch_size": 78, "dropout_1": 0.9510740133913004, "dropout_2": 0.3263851441475057, "learning_rate": 0.009715536539110267, "wd": 0.0002984576239921338}, "metrics": {"cost_metric": {"1": 29.966220140457153}, "active_metric": {"1": 0.6366185897435898}}}, {"candidate": {"n_units_1": 672, "n_units_2": 820, "batch_size": 108, "dropout_1": 0.6443283647430158, "dropout_2": 0.8194904484310889, "learning_rate": 9.196365243521935e-05, "wd": 0.002536625472111785}, "metrics": {"cost_metric": {"1": 32.458306074142456}, "active_metric": {"1": 0.24486714975845414}}}, {"candidate": {"n_units_1": 688, "n_units_2": 597, "batch_size": 123, "dropout_1": 0.7829512576762913, "dropout_2": 0.2834197685256876, "learning_rate": 0.1784738929251937, "wd": 4.489784182359429e-08}, "metrics": {"cost_metric": {"1": 30.61675500869751}, "active_metric": {"1": 0.8976211984342066}}}, {"candidate": {"n_units_1": 501, "n_units_2": 601, "batch_size": 34, "dropout_1": 0.7410256603874262, "dropout_2": 0.046625361151571336, "learning_rate": 0.07937041160202492, "wd": 8.340962845965557e-07}, "metrics": {"cost_metric": {"1": 38.63721990585327}, "active_metric": {"1": 0.904561824729892}}}, {"candidate": {"n_units_1": 1024, "n_units_2": 1002, "batch_size": 116, "dropout_1": 0.0366350257842321, "dropout_2": 0.6883751950302733, "learning_rate": 0.0003133897834907133, "wd": 1.1611672813117278e-08}, "metrics": {"cost_metric": {"1": 44.914865016937256, "2": 95.70968389511108, "3": 134.2296760082245, "4": 167.57774996757507, "5": 205.92636585235596, "6": 242.58663892745972, "7": 283.2623338699341, "8": 326.5033459663391, "9": 360.76255893707275, "10": 398.4191679954529, "11": 434.4982190132141}, "active_metric": {"1": 0.15477145148356053, "2": 0.13502405773857262, "3": 0.1245990376904571, "4": 0.12840817963111473, "5": 0.12219326383319973, "6": 0.11778267842822776, "7": 0.1133720930232558, "8": 0.11166800320769843, "9": 0.10976343223736973, "10": 0.1067562149157979, "11": 0.10555332798716921}}}], "failed_candidates": [], "pending_candidates": [{"batch_size": 89, "dropout_1": 0.19654676887125966, "dropout_2": 0.8682666451901773, "learning_rate": 0.00031134631996358774, "n_units_1": 1024, "n_units_2": 707, "wd": 1e-08, "RESOURCE_ATTR_epoch": 1}, {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05, "RESOURCE_ATTR_epoch": 5}, {"n_units_1": 1024, "n_units_2": 1002, "batch_size": 116, "dropout_1": 0.0366350257842321, "dropout_2": 0.6883751950302733, "learning_rate": 0.0003133897834907133, "wd": 1.1611672813117278e-08, "RESOURCE_ATTR_epoch": 2}, {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607, "RESOURCE_ATTR_epoch": 12}]}')
# elapsed_time = 203.53759908676147
# num_observations = 23
# num_configs = 10

_model_params.append('{"noise_variance": 0.012624704488939506, "kernel_inv_bw0": 0.0026714958295617746, "kernel_inv_bw1": 0.002294225496133934, "kernel_inv_bw2": 0.0005810444910329019, "kernel_inv_bw3": 4.756569311119674, "kernel_inv_bw4": 0.41912704911412996, "kernel_inv_bw5": 0.007082508117597436, "kernel_inv_bw6": 0.6008226671164758, "kernel_covariance_scale": 1.2790537663629489, "mean_mean_value": -0.29754767463440174, "issm_gamma": 0.0010000000000000002, "issm_alpha": -0.20709875141786813, "issm_beta": 1.1564145320327957}')
_state.append('{"candidate_evaluations": [{"candidate": {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607}, "metrics": {"cost_metric": {"1": 12.25258493423462, "2": 24.305160999298096, "3": 44.05741477012634, "4": 62.029183864593506, "5": 81.38737893104553, "6": 99.16185593605042, "7": 118.72888779640198, "8": 133.45333671569824, "9": 148.23734402656555, "10": 166.52369689941406, "11": 194.99460196495056, "12": 215.73117184638977, "13": 235.3977439403534, "14": 253.71279788017273, "15": 267.6743288040161, "16": 281.8612160682678, "17": 296.0602250099182, "18": 310.0040330886841, "19": 324.75612902641296, "20": 344.674284696579, "21": 360.0983910560608, "22": 375.9487638473511, "23": 395.81145191192627, "24": 411.6494069099426, "25": 426.79202795028687, "26": 448.74489879608154, "27": 464.90988278388977, "28": 480.28413486480713, "29": 494.5631868839264, "30": 510.31515073776245, "31": 527.6290948390961, "32": 542.7905468940735, "33": 558.1524910926819, "34": 572.6776859760284, "35": 588.3533399105072}, "active_metric": {"1": 0.4978924126856684, "2": 0.3896025692492975, "3": 0.3546768366118025, "4": 0.33289843436370936, "5": 0.3259735046166198, "6": 0.30971497390606184, "7": 0.29626655961461257, "8": 0.2863307908470494, "9": 0.2753914090726616, "10": 0.26455238859895625, "11": 0.25491770373344036, "12": 0.2485949417904456, "13": 0.24678843837816133, "14": 0.23996386993175434, "15": 0.2332396627860297, "16": 0.23143315937374553, "17": 0.22390606182256123, "18": 0.22350461661983134, "19": 0.2195905258932156, "20": 0.217482938578884, "21": 0.21176234443998398, "22": 0.2106583701324769, "23": 0.20764753111200318, "24": 0.2052388598956243, "25": 0.20102368526696102, "26": 0.1963067041348856, "27": 0.1949016459253312, "28": 0.19409875551987155, "29": 0.1904857486953031, "30": 0.18747490967482938, "31": 0.18516659975913285, "32": 0.18446407065435566, "33": 0.18115214773183463, "34": 0.18004817342432755, "35": 0.17844239261340822}}}, {"candidate": {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05}, "metrics": {"cost_metric": {"1": 18.50484609603882, "2": 45.980664014816284, "3": 96.58328175544739, "4": 148.91678476333618, "5": 229.4046459197998, "6": 300.6477417945862, "7": 383.9124698638916, "8": 473.07399010658264, "9": 555.2527649402618}, "active_metric": {"1": 0.15496198479391754, "2": 0.1519607843137255, "3": 0.13645458183273307, "4": 0.14435774309723892, "5": 0.13225290116046418, "6": 0.1253501400560224, "7": 0.12735094037615047, "8": 0.13265306122448983, "9": 0.12104841936774713}}}, {"candidate": {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07}, "metrics": {"cost_metric": {"1": 18.432047128677368}, "active_metric": {"1": 0.16786858974358976}}}, {"candidate": {"n_units_1": 91, "n_units_2": 459, "batch_size": 105, "dropout_1": 0.48639033141890325, "dropout_2": 0.21324913218446714, "learning_rate": 0.00013769715715418189, "wd": 0.02017249366944585}, "metrics": {"cost_metric": {"1": 17.439072132110596}, "active_metric": {"1": 0.3006516290726817}}}, {"candidate": {"n_units_1": 774, "n_units_2": 917, "batch_size": 29, "dropout_1": 0.7778923725289609, "dropout_2": 0.7413003050986398, "learning_rate": 6.472832341968678e-05, "wd": 0.0007744951242384949}, "metrics": {"cost_metric": {"1": 44.07283306121826}, "active_metric": {"1": 0.23085404971932644}}}, {"candidate": {"n_units_1": 673, "n_units_2": 262, "batch_size": 78, "dropout_1": 0.9510740133913004, "dropout_2": 0.3263851441475057, "learning_rate": 0.009715536539110267, "wd": 0.0002984576239921338}, "metrics": {"cost_metric": {"1": 29.966220140457153}, "active_metric": {"1": 0.6366185897435898}}}, {"candidate": {"n_units_1": 672, "n_units_2": 820, "batch_size": 108, "dropout_1": 0.6443283647430158, "dropout_2": 0.8194904484310889, "learning_rate": 9.196365243521935e-05, "wd": 0.002536625472111785}, "metrics": {"cost_metric": {"1": 32.458306074142456}, "active_metric": {"1": 0.24486714975845414}}}, {"candidate": {"n_units_1": 688, "n_units_2": 597, "batch_size": 123, "dropout_1": 0.7829512576762913, "dropout_2": 0.2834197685256876, "learning_rate": 0.1784738929251937, "wd": 4.489784182359429e-08}, "metrics": {"cost_metric": {"1": 30.61675500869751}, "active_metric": {"1": 0.8976211984342066}}}], "failed_candidates": [], "pending_candidates": [{"batch_size": 34, "dropout_1": 0.7410256603874262, "dropout_2": 0.046625361151571336, "learning_rate": 0.07937041160202492, "n_units_1": 501, "n_units_2": 601, "wd": 8.340962845965557e-07, "RESOURCE_ATTR_epoch": 1}, {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05, "RESOURCE_ATTR_epoch": 4}, {"batch_size": 116, "dropout_1": 0.0366350257842321, "dropout_2": 0.6883751950302733, "learning_rate": 0.0003133897834907133, "n_units_1": 1024, "n_units_2": 1002, "wd": 1.1611672813117278e-08, "RESOURCE_ATTR_epoch": 1}, {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607, "RESOURCE_ATTR_epoch": 9}]}')
# elapsed_time = 141.0116560459137
# num_observations = 17
# num_configs = 8

_model_params.append('{"noise_variance": 0.02443305886195063, "kernel_inv_bw0": 0.01410539584512635, "kernel_inv_bw1": 1.4106734173901074, "kernel_inv_bw2": 0.002912772873874073, "kernel_inv_bw3": 0.00010000000000000009, "kernel_inv_bw4": 0.001289783525647755, "kernel_inv_bw5": 6.274402643366595, "kernel_inv_bw6": 0.014263119266637505, "kernel_covariance_scale": 1.0004606474604771, "mean_mean_value": -1.0965610760358047, "issm_gamma": 0.0010000000000000002, "issm_alpha": -0.6394260638653898, "issm_beta": 0.8896093870386187}')
_state.append('{"candidate_evaluations": [{"candidate": {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607}, "metrics": {"cost_metric": {"1": 12.25258493423462, "2": 24.305160999298096, "3": 44.05741477012634, "4": 62.029183864593506, "5": 81.38737893104553, "6": 99.16185593605042, "7": 118.72888779640198, "8": 133.45333671569824, "9": 148.23734402656555, "10": 166.52369689941406, "11": 194.99460196495056, "12": 215.73117184638977, "13": 235.3977439403534, "14": 253.71279788017273, "15": 267.6743288040161, "16": 281.8612160682678, "17": 296.0602250099182, "18": 310.0040330886841, "19": 324.75612902641296, "20": 344.674284696579, "21": 360.0983910560608, "22": 375.9487638473511, "23": 395.81145191192627, "24": 411.6494069099426, "25": 426.79202795028687, "26": 448.74489879608154, "27": 464.90988278388977, "28": 480.28413486480713, "29": 494.5631868839264, "30": 510.31515073776245, "31": 527.6290948390961, "32": 542.7905468940735, "33": 558.1524910926819, "34": 572.6776859760284, "35": 588.3533399105072}, "active_metric": {"1": 0.4978924126856684, "2": 0.3896025692492975, "3": 0.3546768366118025, "4": 0.33289843436370936, "5": 0.3259735046166198, "6": 0.30971497390606184, "7": 0.29626655961461257, "8": 0.2863307908470494, "9": 0.2753914090726616, "10": 0.26455238859895625, "11": 0.25491770373344036, "12": 0.2485949417904456, "13": 0.24678843837816133, "14": 0.23996386993175434, "15": 0.2332396627860297, "16": 0.23143315937374553, "17": 0.22390606182256123, "18": 0.22350461661983134, "19": 0.2195905258932156, "20": 0.217482938578884, "21": 0.21176234443998398, "22": 0.2106583701324769, "23": 0.20764753111200318, "24": 0.2052388598956243, "25": 0.20102368526696102, "26": 0.1963067041348856, "27": 0.1949016459253312, "28": 0.19409875551987155, "29": 0.1904857486953031, "30": 0.18747490967482938, "31": 0.18516659975913285, "32": 0.18446407065435566, "33": 0.18115214773183463, "34": 0.18004817342432755, "35": 0.17844239261340822}}}, {"candidate": {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05}, "metrics": {"cost_metric": {"1": 18.50484609603882, "2": 45.980664014816284, "3": 96.58328175544739, "4": 148.91678476333618, "5": 229.4046459197998, "6": 300.6477417945862, "7": 383.9124698638916, "8": 473.07399010658264, "9": 555.2527649402618}, "active_metric": {"1": 0.15496198479391754, "2": 0.1519607843137255, "3": 0.13645458183273307, "4": 0.14435774309723892, "5": 0.13225290116046418, "6": 0.1253501400560224, "7": 0.12735094037615047, "8": 0.13265306122448983, "9": 0.12104841936774713}}}, {"candidate": {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07}, "metrics": {"cost_metric": {"1": 18.432047128677368}, "active_metric": {"1": 0.16786858974358976}}}, {"candidate": {"n_units_1": 91, "n_units_2": 459, "batch_size": 105, "dropout_1": 0.48639033141890325, "dropout_2": 0.21324913218446714, "learning_rate": 0.00013769715715418189, "wd": 0.02017249366944585}, "metrics": {"cost_metric": {"1": 17.439072132110596}, "active_metric": {"1": 0.3006516290726817}}}, {"candidate": {"n_units_1": 774, "n_units_2": 917, "batch_size": 29, "dropout_1": 0.7778923725289609, "dropout_2": 0.7413003050986398, "learning_rate": 6.472832341968678e-05, "wd": 0.0007744951242384949}, "metrics": {"cost_metric": {"1": 44.07283306121826}, "active_metric": {"1": 0.23085404971932644}}}, {"candidate": {"n_units_1": 673, "n_units_2": 262, "batch_size": 78, "dropout_1": 0.9510740133913004, "dropout_2": 0.3263851441475057, "learning_rate": 0.009715536539110267, "wd": 0.0002984576239921338}, "metrics": {"cost_metric": {"1": 29.966220140457153}, "active_metric": {"1": 0.6366185897435898}}}, {"candidate": {"n_units_1": 672, "n_units_2": 820, "batch_size": 108, "dropout_1": 0.6443283647430158, "dropout_2": 0.8194904484310889, "learning_rate": 9.196365243521935e-05, "wd": 0.002536625472111785}, "metrics": {"cost_metric": {"1": 32.458306074142456}, "active_metric": {"1": 0.24486714975845414}}}], "failed_candidates": [], "pending_candidates": [{"batch_size": 123, "dropout_1": 0.7829512576762913, "dropout_2": 0.2834197685256876, "learning_rate": 0.1784738929251937, "n_units_1": 688, "n_units_2": 597, "wd": 4.489784182359429e-08, "RESOURCE_ATTR_epoch": 1}, {"batch_size": 34, "dropout_1": 0.7410256603874262, "dropout_2": 0.046625361151571336, "learning_rate": 0.07937041160202492, "n_units_1": 501, "n_units_2": 601, "wd": 8.340962845965557e-07, "RESOURCE_ATTR_epoch": 1}, {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05, "RESOURCE_ATTR_epoch": 4}, {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607, "RESOURCE_ATTR_epoch": 7}]}')
# elapsed_time = 108.29521012306213
# num_observations = 14
# num_configs = 7

_model_params.append('{"noise_variance": 0.02433360380308369, "kernel_inv_bw0": 0.033230128902756034, "kernel_inv_bw1": 1.3832161502574647, "kernel_inv_bw2": 0.0010926642716173997, "kernel_inv_bw3": 0.0009913284444091315, "kernel_inv_bw4": 0.00037318250862594773, "kernel_inv_bw5": 7.150355629993121, "kernel_inv_bw6": 0.005367219098449991, "kernel_covariance_scale": 0.9234243919759128, "mean_mean_value": -1.0950448515295788, "issm_gamma": 0.0010000000000000002, "issm_alpha": -0.6000883418698378, "issm_beta": 0.814092090699343}')
_state.append('{"candidate_evaluations": [{"candidate": {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607}, "metrics": {"cost_metric": {"1": 12.25258493423462, "2": 24.305160999298096, "3": 44.05741477012634, "4": 62.029183864593506, "5": 81.38737893104553, "6": 99.16185593605042, "7": 118.72888779640198, "8": 133.45333671569824, "9": 148.23734402656555, "10": 166.52369689941406, "11": 194.99460196495056, "12": 215.73117184638977, "13": 235.3977439403534, "14": 253.71279788017273, "15": 267.6743288040161, "16": 281.8612160682678, "17": 296.0602250099182, "18": 310.0040330886841, "19": 324.75612902641296, "20": 344.674284696579, "21": 360.0983910560608, "22": 375.9487638473511, "23": 395.81145191192627, "24": 411.6494069099426, "25": 426.79202795028687, "26": 448.74489879608154, "27": 464.90988278388977, "28": 480.28413486480713, "29": 494.5631868839264, "30": 510.31515073776245, "31": 527.6290948390961, "32": 542.7905468940735, "33": 558.1524910926819, "34": 572.6776859760284, "35": 588.3533399105072}, "active_metric": {"1": 0.4978924126856684, "2": 0.3896025692492975, "3": 0.3546768366118025, "4": 0.33289843436370936, "5": 0.3259735046166198, "6": 0.30971497390606184, "7": 0.29626655961461257, "8": 0.2863307908470494, "9": 0.2753914090726616, "10": 0.26455238859895625, "11": 0.25491770373344036, "12": 0.2485949417904456, "13": 0.24678843837816133, "14": 0.23996386993175434, "15": 0.2332396627860297, "16": 0.23143315937374553, "17": 0.22390606182256123, "18": 0.22350461661983134, "19": 0.2195905258932156, "20": 0.217482938578884, "21": 0.21176234443998398, "22": 0.2106583701324769, "23": 0.20764753111200318, "24": 0.2052388598956243, "25": 0.20102368526696102, "26": 0.1963067041348856, "27": 0.1949016459253312, "28": 0.19409875551987155, "29": 0.1904857486953031, "30": 0.18747490967482938, "31": 0.18516659975913285, "32": 0.18446407065435566, "33": 0.18115214773183463, "34": 0.18004817342432755, "35": 0.17844239261340822}}}, {"candidate": {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05}, "metrics": {"cost_metric": {"1": 18.50484609603882, "2": 45.980664014816284, "3": 96.58328175544739, "4": 148.91678476333618, "5": 229.4046459197998, "6": 300.6477417945862, "7": 383.9124698638916, "8": 473.07399010658264, "9": 555.2527649402618}, "active_metric": {"1": 0.15496198479391754, "2": 0.1519607843137255, "3": 0.13645458183273307, "4": 0.14435774309723892, "5": 0.13225290116046418, "6": 0.1253501400560224, "7": 0.12735094037615047, "8": 0.13265306122448983, "9": 0.12104841936774713}}}, {"candidate": {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07}, "metrics": {"cost_metric": {"1": 18.432047128677368}, "active_metric": {"1": 0.16786858974358976}}}, {"candidate": {"n_units_1": 91, "n_units_2": 459, "batch_size": 105, "dropout_1": 0.48639033141890325, "dropout_2": 0.21324913218446714, "learning_rate": 0.00013769715715418189, "wd": 0.02017249366944585}, "metrics": {"cost_metric": {"1": 17.439072132110596}, "active_metric": {"1": 0.3006516290726817}}}, {"candidate": {"n_units_1": 774, "n_units_2": 917, "batch_size": 29, "dropout_1": 0.7778923725289609, "dropout_2": 0.7413003050986398, "learning_rate": 6.472832341968678e-05, "wd": 0.0007744951242384949}, "metrics": {"cost_metric": {"1": 44.07283306121826}, "active_metric": {"1": 0.23085404971932644}}}, {"candidate": {"n_units_1": 673, "n_units_2": 262, "batch_size": 78, "dropout_1": 0.9510740133913004, "dropout_2": 0.3263851441475057, "learning_rate": 0.009715536539110267, "wd": 0.0002984576239921338}, "metrics": {"cost_metric": {"1": 29.966220140457153}, "active_metric": {"1": 0.6366185897435898}}}], "failed_candidates": [], "pending_candidates": [{"batch_size": 108, "dropout_1": 0.6443283647430158, "dropout_2": 0.8194904484310889, "learning_rate": 9.196365243521935e-05, "n_units_1": 672, "n_units_2": 820, "wd": 0.002536625472111785, "RESOURCE_ATTR_epoch": 1}, {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05, "RESOURCE_ATTR_epoch": 3}, {"batch_size": 123, "dropout_1": 0.7829512576762913, "dropout_2": 0.2834197685256876, "learning_rate": 0.1784738929251937, "n_units_1": 688, "n_units_2": 597, "wd": 4.489784182359429e-08, "RESOURCE_ATTR_epoch": 1}, {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607, "RESOURCE_ATTR_epoch": 6}]}')
# elapsed_time = 91.06228709220886
# num_observations = 11
# num_configs = 6

_model_params.append('{"noise_variance": 4.023868196955162e-08, "kernel_inv_bw0": 0.0499484523829736, "kernel_inv_bw1": 0.5041477744353572, "kernel_inv_bw2": 0.045440426051123285, "kernel_inv_bw3": 3.509634264819305, "kernel_inv_bw4": 1.8117976798318889, "kernel_inv_bw5": 16.29050792588867, "kernel_inv_bw6": 0.011845890028904541, "kernel_covariance_scale": 2.72277711886595, "mean_mean_value": -1.412204593314323, "issm_gamma": 0.0010000000000000002, "issm_alpha": -1.7778733115941194, "issm_beta": 1.226405864173305}')
_state.append('{"candidate_evaluations": [{"candidate": {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607}, "metrics": {"cost_metric": {"1": 12.25258493423462, "2": 24.305160999298096, "3": 44.05741477012634, "4": 62.029183864593506, "5": 81.38737893104553, "6": 99.16185593605042, "7": 118.72888779640198, "8": 133.45333671569824, "9": 148.23734402656555, "10": 166.52369689941406, "11": 194.99460196495056, "12": 215.73117184638977, "13": 235.3977439403534, "14": 253.71279788017273, "15": 267.6743288040161, "16": 281.8612160682678, "17": 296.0602250099182, "18": 310.0040330886841, "19": 324.75612902641296, "20": 344.674284696579, "21": 360.0983910560608, "22": 375.9487638473511, "23": 395.81145191192627, "24": 411.6494069099426, "25": 426.79202795028687, "26": 448.74489879608154, "27": 464.90988278388977, "28": 480.28413486480713, "29": 494.5631868839264, "30": 510.31515073776245, "31": 527.6290948390961, "32": 542.7905468940735, "33": 558.1524910926819, "34": 572.6776859760284, "35": 588.3533399105072}, "active_metric": {"1": 0.4978924126856684, "2": 0.3896025692492975, "3": 0.3546768366118025, "4": 0.33289843436370936, "5": 0.3259735046166198, "6": 0.30971497390606184, "7": 0.29626655961461257, "8": 0.2863307908470494, "9": 0.2753914090726616, "10": 0.26455238859895625, "11": 0.25491770373344036, "12": 0.2485949417904456, "13": 0.24678843837816133, "14": 0.23996386993175434, "15": 0.2332396627860297, "16": 0.23143315937374553, "17": 0.22390606182256123, "18": 0.22350461661983134, "19": 0.2195905258932156, "20": 0.217482938578884, "21": 0.21176234443998398, "22": 0.2106583701324769, "23": 0.20764753111200318, "24": 0.2052388598956243, "25": 0.20102368526696102, "26": 0.1963067041348856, "27": 0.1949016459253312, "28": 0.19409875551987155, "29": 0.1904857486953031, "30": 0.18747490967482938, "31": 0.18516659975913285, "32": 0.18446407065435566, "33": 0.18115214773183463, "34": 0.18004817342432755, "35": 0.17844239261340822}}}, {"candidate": {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05}, "metrics": {"cost_metric": {"1": 18.50484609603882, "2": 45.980664014816284, "3": 96.58328175544739, "4": 148.91678476333618, "5": 229.4046459197998, "6": 300.6477417945862, "7": 383.9124698638916, "8": 473.07399010658264, "9": 555.2527649402618}, "active_metric": {"1": 0.15496198479391754, "2": 0.1519607843137255, "3": 0.13645458183273307, "4": 0.14435774309723892, "5": 0.13225290116046418, "6": 0.1253501400560224, "7": 0.12735094037615047, "8": 0.13265306122448983, "9": 0.12104841936774713}}}, {"candidate": {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07}, "metrics": {"cost_metric": {"1": 18.432047128677368}, "active_metric": {"1": 0.16786858974358976}}}, {"candidate": {"n_units_1": 91, "n_units_2": 459, "batch_size": 105, "dropout_1": 0.48639033141890325, "dropout_2": 0.21324913218446714, "learning_rate": 0.00013769715715418189, "wd": 0.02017249366944585}, "metrics": {"cost_metric": {"1": 17.439072132110596}, "active_metric": {"1": 0.3006516290726817}}}, {"candidate": {"n_units_1": 774, "n_units_2": 917, "batch_size": 29, "dropout_1": 0.7778923725289609, "dropout_2": 0.7413003050986398, "learning_rate": 6.472832341968678e-05, "wd": 0.0007744951242384949}, "metrics": {"cost_metric": {"1": 44.07283306121826}, "active_metric": {"1": 0.23085404971932644}}}], "failed_candidates": [], "pending_candidates": [{"n_units_1": 673, "n_units_2": 262, "batch_size": 78, "dropout_1": 0.9510740133913004, "dropout_2": 0.3263851441475057, "learning_rate": 0.009715536539110267, "wd": 0.0002984576239921338, "RESOURCE_ATTR_epoch": 1}, {"batch_size": 108, "dropout_1": 0.6443283647430158, "dropout_2": 0.8194904484310889, "learning_rate": 9.196365243521935e-05, "n_units_1": 672, "n_units_2": 820, "wd": 0.002536625472111785, "RESOURCE_ATTR_epoch": 1}, {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05, "RESOURCE_ATTR_epoch": 3}, {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607, "RESOURCE_ATTR_epoch": 4}]}')
# elapsed_time = 55.957242012023926
# num_observations = 8
# num_configs = 5

_model_params.append('{"noise_variance": 0.0010000000000000002, "kernel_inv_bw0": 1.0, "kernel_inv_bw1": 1.0, "kernel_inv_bw2": 1.0, "kernel_inv_bw3": 1.0, "kernel_inv_bw4": 1.0, "kernel_inv_bw5": 1.0, "kernel_inv_bw6": 1.0, "kernel_covariance_scale": 1.0, "mean_mean_value": 0.0, "issm_gamma": 1.0, "issm_alpha": -0.5, "issm_beta": 0.0}')
_state.append('{"candidate_evaluations": [{"candidate": {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607}, "metrics": {"cost_metric": {"1": 12.25258493423462, "2": 24.305160999298096, "3": 44.05741477012634, "4": 62.029183864593506, "5": 81.38737893104553, "6": 99.16185593605042, "7": 118.72888779640198, "8": 133.45333671569824, "9": 148.23734402656555, "10": 166.52369689941406, "11": 194.99460196495056, "12": 215.73117184638977, "13": 235.3977439403534, "14": 253.71279788017273, "15": 267.6743288040161, "16": 281.8612160682678, "17": 296.0602250099182, "18": 310.0040330886841, "19": 324.75612902641296, "20": 344.674284696579, "21": 360.0983910560608, "22": 375.9487638473511, "23": 395.81145191192627, "24": 411.6494069099426, "25": 426.79202795028687, "26": 448.74489879608154, "27": 464.90988278388977, "28": 480.28413486480713, "29": 494.5631868839264, "30": 510.31515073776245, "31": 527.6290948390961, "32": 542.7905468940735, "33": 558.1524910926819, "34": 572.6776859760284, "35": 588.3533399105072}, "active_metric": {"1": 0.4978924126856684, "2": 0.3896025692492975, "3": 0.3546768366118025, "4": 0.33289843436370936, "5": 0.3259735046166198, "6": 0.30971497390606184, "7": 0.29626655961461257, "8": 0.2863307908470494, "9": 0.2753914090726616, "10": 0.26455238859895625, "11": 0.25491770373344036, "12": 0.2485949417904456, "13": 0.24678843837816133, "14": 0.23996386993175434, "15": 0.2332396627860297, "16": 0.23143315937374553, "17": 0.22390606182256123, "18": 0.22350461661983134, "19": 0.2195905258932156, "20": 0.217482938578884, "21": 0.21176234443998398, "22": 0.2106583701324769, "23": 0.20764753111200318, "24": 0.2052388598956243, "25": 0.20102368526696102, "26": 0.1963067041348856, "27": 0.1949016459253312, "28": 0.19409875551987155, "29": 0.1904857486953031, "30": 0.18747490967482938, "31": 0.18516659975913285, "32": 0.18446407065435566, "33": 0.18115214773183463, "34": 0.18004817342432755, "35": 0.17844239261340822}}}, {"candidate": {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05}, "metrics": {"cost_metric": {"1": 18.50484609603882, "2": 45.980664014816284, "3": 96.58328175544739, "4": 148.91678476333618, "5": 229.4046459197998, "6": 300.6477417945862, "7": 383.9124698638916, "8": 473.07399010658264, "9": 555.2527649402618}, "active_metric": {"1": 0.15496198479391754, "2": 0.1519607843137255, "3": 0.13645458183273307, "4": 0.14435774309723892, "5": 0.13225290116046418, "6": 0.1253501400560224, "7": 0.12735094037615047, "8": 0.13265306122448983, "9": 0.12104841936774713}}}, {"candidate": {"n_units_1": 347, "n_units_2": 566, "batch_size": 48, "dropout_1": 0.40991313560097764, "dropout_2": 0.1486640484580416, "learning_rate": 0.0001521657976426163, "wd": 2.46706548222209e-07}, "metrics": {"cost_metric": {"1": 18.432047128677368}, "active_metric": {"1": 0.16786858974358976}}}, {"candidate": {"n_units_1": 91, "n_units_2": 459, "batch_size": 105, "dropout_1": 0.48639033141890325, "dropout_2": 0.21324913218446714, "learning_rate": 0.00013769715715418189, "wd": 0.02017249366944585}, "metrics": {"cost_metric": {"1": 17.439072132110596}, "active_metric": {"1": 0.3006516290726817}}}], "failed_candidates": [], "pending_candidates": [{"n_units_1": 774, "n_units_2": 917, "batch_size": 29, "dropout_1": 0.7778923725289609, "dropout_2": 0.7413003050986398, "learning_rate": 6.472832341968678e-05, "wd": 0.0007744951242384949, "RESOURCE_ATTR_epoch": 1}, {"n_units_1": 514, "n_units_2": 514, "batch_size": 68, "dropout_1": 0.495, "dropout_2": 0.495, "learning_rate": 0.0010000000000000002, "wd": 9.999999999999991e-05, "RESOURCE_ATTR_epoch": 2}, {"n_units_1": 38, "n_units_2": 187, "batch_size": 53, "dropout_1": 0.36209963448394383, "dropout_2": 0.09749003575393035, "learning_rate": 1.180123718822517e-05, "wd": 0.00011948182727147607, "RESOURCE_ATTR_epoch": 3}]}')
# elapsed_time = 50.29185605049133
# num_observations = 5
# num_configs = 4


@pytest.mark.parametrize(
    "_model_params, _state", zip(_model_params, _state))
def test_compare_gpiss_likelihood_oldnew(_model_params, _state):
    config_space = {
        'n_units_1': randint(4, 1024),
        'n_units_2': randint(4, 1024),
        'batch_size': randint(8, 128),
        'dropout_1': uniform(0, 0.99),
        'dropout_2': uniform(0, 0.99),
        'learning_rate': loguniform(1e-6, 1),
        'wd': loguniform(1e-8, 1),
        'epochs': 81,
    }

    gpiss_model_factory = []  # new, old
    model_params = json.loads(_model_params)
    kwargs = dict(no_fantasizing=True)
    gpiss_objs = build_gpiss_model_factory(
        config_space, model_params, **kwargs)
    configspace_ext = gpiss_objs['configspace_ext']
    gpiss_model_factory.append(gpiss_objs['model_factory'])
    gpiss_model_factory.append(build_gpiss_model_factory(
        config_space, model_params, use_new_code=False,
        **kwargs)['model_factory'])
    state = decode_state_from_old_encoding(
        enc_state=json.loads(_state), hp_ranges=configspace_ext.hp_ranges_ext)

    # Compare likelihoods
    likelihood = [
        factory.model(
            state,
            fit_params=False).posterior_states[0].poster_state['likelihood']
        for factory in gpiss_model_factory]
    for name, value in likelihood[0].items():
        if name != 'num_data':
            np.testing.assert_almost_equal(value, likelihood[1][name])


@pytest.mark.parametrize(
    "_model_params, _state", zip(_model_params, _state))
def test_compare_gpiss_likelihood_fantasizing_oldnew(_model_params, _state):
    config_space = {
        'n_units_1': randint(4, 1024),
        'n_units_2': randint(4, 1024),
        'batch_size': randint(8, 128),
        'dropout_1': uniform(0, 0.99),
        'dropout_2': uniform(0, 0.99),
        'learning_rate': loguniform(1e-6, 1),
        'wd': loguniform(1e-8, 1),
        'epochs': 81,
    }
    num_fantasy_samples = 10

    gpiss_model_factory = []  # new, old
    model_params = json.loads(_model_params)
    kwargs = dict(
        num_fantasy_samples=num_fantasy_samples,
        no_fantasizing=False)
    gpiss_objs = build_gpiss_model_factory(
        config_space, model_params, **kwargs)
    configspace_ext = gpiss_objs['configspace_ext']
    gpiss_model_factory.append(gpiss_objs['model_factory'])
    gpiss_model_factory.append(build_gpiss_model_factory(
        config_space, model_params, use_new_code=False,
        **kwargs)['model_factory'])
    state = decode_state_from_old_encoding(
        enc_state=json.loads(_state), hp_ranges=configspace_ext.hp_ranges_ext)

    # Compare likelihoods
    # We need to force them to use the same fantasy samples
    gpiss_model1 = gpiss_model_factory[0].model(state, fit_params=False)
    likelihood = [gpiss_model1.posterior_states[0].poster_state['likelihood']]
    gpiss_model2 = gpiss_model_factory[1].model_for_fantasy_samples(
        state, fantasy_samples=gpiss_model1.fantasy_samples)
    likelihood.append(
        gpiss_model2.posterior_states[0].poster_state['likelihood'])
    for name, value in likelihood[0].items():
        if name != 'num_data':
            np.testing.assert_almost_equal(value, likelihood[1][name])
