import torch
import torch.nn as nn
import torch.nn.functional as F
import math


class ConvLayer(nn.Module):
    def __init__(self, c_in):
        super(ConvLayer, self).__init__()
        self.downConv = nn.Conv1d(in_channels=c_in,
                                  out_channels=c_in,
                                  kernel_size=3,
                                  padding=2,
                                  padding_mode='circular')
        self.norm = nn.BatchNorm1d(c_in)
        self.activation = nn.ELU()
        self.maxPool = nn.MaxPool1d(kernel_size=3, stride=2, padding=1)

    def forward(self, x):
        x = self.downConv(x.permute(0, 2, 1))
        x = self.norm(x)
        x = self.activation(x)
        x = self.maxPool(x)
        x = x.transpose(1, 2)
        return x


class EncoderLayer(nn.Module):
    def __init__(self, attention, d_model, d_ff=None, dropout=0.1, activation="relu"):
        super(EncoderLayer, self).__init__()
        d_ff = d_ff or 4 * d_model
        
        self.attention = attention
        self.conv1 = nn.Conv1d(in_channels=d_model, out_channels=d_ff, kernel_size=1)
        self.conv2 = nn.Conv1d(in_channels=d_ff, out_channels=d_model, kernel_size=1)

        self.conv3 = nn.Conv1d(d_model, d_model, kernel_size=1)
        self.conv4 = nn.Conv1d(d_model, d_model, kernel_size=1)
        self.conv5 = nn.Conv1d(3 * d_model, d_model, kernel_size=1)
        self.conv6 = nn.Conv1d(3 * d_model, d_model, kernel_size=1)
        
        self.norm1 = nn.LayerNorm(d_model)
        self.norm2 = nn.LayerNorm(d_model)

        self.conv_1 = nn.Conv1d(d_model, d_model, kernel_size=1, padding=0)
        self.conv_2 = nn.Conv1d(d_model, d_model, kernel_size=3, padding=1)
        self.conv_3 = nn.Conv1d(d_model, d_model, kernel_size=5, padding=2)
        self.fc = nn.Conv1d(d_model, d_model, kernel_size=1)

        self.dropout = nn.Dropout(dropout)
        self.activation = F.relu if activation == "relu" else F.gelu

    def forward(self, s, t, attn_mask=None, tau=None, delta=None):

        new_st, attn = self.attention(
            s, s, s,
            attn_mask=attn_mask,
            tau=tau, delta=delta
        )

        s = s - self.dropout(new_st)
        s_ln = s = self.norm1(s)
        s_ln = self.dropout(self.activation(self.conv1(s_ln.transpose(-1, 1))))
        s_ln = self.dropout(self.conv2(s_ln).transpose(-1, 1))

        s = (s - s_ln).transpose(-1, 1)
        s_gate = F.sigmoid(self.conv3(s))
        out_s = s_gate * self.conv4(s)

        weights = self.conv_1(new_st.transpose(-1, 1)) + self.conv_2(new_st.transpose(-1, 1)) + self.conv_3(new_st.transpose(-1, 1))
        weights = F.sigmoid(self.fc(weights).transpose(-1, 1))
        t = weights * t

        t = torch.cat([new_st, s_ln, t], dim=-1)
        t = t.transpose(-1, 1)
        t_gate = F.sigmoid(self.conv5(t))
        out_t = t_gate * self.conv6(t)

        return self.norm2(out_s.transpose(-1, 1)), out_t.transpose(-1, 1), attn



class Encoder(nn.Module):
    def __init__(self, attn_layers, conv_layers=None, norm_layer=None):
        super(Encoder, self).__init__()
        self.attn_layers = nn.ModuleList(attn_layers)
        self.conv_layers = nn.ModuleList(conv_layers) if conv_layers is not None else None
        
        self.norm = norm_layer

    def forward(self, s, t, attn_mask=None, tau=None, delta=None):

        attns = []

        for attn_layer in self.attn_layers:
            s, t, attn = attn_layer(s, t, attn_mask=attn_mask, tau=tau, delta=delta)
            attns.append(attn)

        output = s + t

        return output, attns


class DecoderLayer(nn.Module):
    def __init__(self, self_attention, cross_attention, d_model, d_ff=None,
                 dropout=0.1, activation="relu"):
        super(DecoderLayer, self).__init__()
        d_ff = d_ff or 4 * d_model
        self.self_attention = self_attention
        self.cross_attention = cross_attention
        self.conv1 = nn.Conv1d(in_channels=d_model, out_channels=d_ff, kernel_size=1)
        self.conv2 = nn.Conv1d(in_channels=d_ff, out_channels=d_model, kernel_size=1)
        self.norm1 = nn.LayerNorm(d_model)
        self.norm2 = nn.LayerNorm(d_model)
        self.norm3 = nn.LayerNorm(d_model)
        self.dropout = nn.Dropout(dropout)
        self.activation = F.relu if activation == "relu" else F.gelu

    def forward(self, x, cross, x_mask=None, cross_mask=None, tau=None, delta=None):
        x = x + self.dropout(self.self_attention(
            x, x, x,
            attn_mask=x_mask,
            tau=tau, delta=None
        )[0])
        x = self.norm1(x)

        x = x + self.dropout(self.cross_attention(
            x, cross, cross,
            attn_mask=cross_mask,
            tau=tau, delta=delta
        )[0])

        y = x = self.norm2(x)
        y = self.dropout(self.activation(self.conv1(y.transpose(-1, 1))))
        y = self.dropout(self.conv2(y).transpose(-1, 1))

        return self.norm3(x + y)


class Decoder(nn.Module):
    def __init__(self, layers, norm_layer=None, projection=None):
        super(Decoder, self).__init__()
        self.layers = nn.ModuleList(layers)
        self.norm = norm_layer
        self.projection = projection

    def forward(self, x, cross, x_mask=None, cross_mask=None, tau=None, delta=None):
        for layer in self.layers:
            x = layer(x, cross, x_mask=x_mask, cross_mask=cross_mask, tau=tau, delta=delta)

        if self.norm is not None:
            x = self.norm(x)

        if self.projection is not None:
            x = self.projection(x)
        return x
