"""
Simplified logging utilities for MoE Knowledge Editing
"""

import logging
import sys
from pathlib import Path
from datetime import datetime


# Simple logger setup
_logger = None


def _setup_logger():
    """Setup a simple logger"""
    global _logger
    if _logger is not None:
        return _logger

    _logger = logging.getLogger('moe_edit')
    _logger.setLevel(logging.INFO)

    # Avoid duplicate handlers
    if _logger.handlers:
        return _logger

    # Console handler
    console_handler = logging.StreamHandler(sys.stdout)
    console_handler.setLevel(logging.INFO)
    console_formatter = logging.Formatter('%(levelname)s - %(message)s')
    console_handler.setFormatter(console_formatter)
    _logger.addHandler(console_handler)

    # Optional file handler
    try:
        log_dir = Path("logs")
        log_dir.mkdir(exist_ok=True)
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        file_handler = logging.FileHandler(log_dir / f"moe_edit_{timestamp}.log")
        file_handler.setLevel(logging.DEBUG)
        file_formatter = logging.Formatter(
            '%(asctime)s - %(levelname)s - %(message)s'
        )
        file_handler.setFormatter(file_formatter)
        _logger.addHandler(file_handler)
    except Exception:
        # If file logging fails, continue with console only
        pass

    return _logger


def get_logger():
    """Get the logger instance"""
    return _setup_logger()





# Convenience functions for different log levels
def debug(msg: str, *args, **kwargs):
    """Log debug message"""
    get_logger().debug(msg, *args, **kwargs)


def info(msg: str, *args, **kwargs):
    """Log info message"""
    get_logger().info(msg, *args, **kwargs)


def warning(msg: str, *args, **kwargs):
    """Log warning message"""
    get_logger().warning(msg, *args, **kwargs)


def error(msg: str, *args, **kwargs):
    """Log error message"""
    get_logger().error(msg, *args, **kwargs)


def get_tqdm_kwargs(**override_kwargs):
    """Get standardized tqdm configuration to avoid ANSI escape sequences in logs"""
    default_kwargs = {
        "disable": False,
        "ascii": True,      # Use ASCII characters instead of Unicode
        "ncols": 80,        # Fixed width to avoid terminal detection issues
        "dynamic_ncols": False,  # Disable dynamic width detection
        "leave": True,      # Keep progress bar after completion
        "file": None,       # Use default output (stdout)
    }

    # Override with any provided kwargs
    default_kwargs.update(override_kwargs)
    return default_kwargs

