import string 
import os 
class InitialPrompt:
    def __init__(self, game_description_dir, agent_game_name, agent_file_name, p1, p2, num_issues=5,num_agents=6,incentive='cooperative',incentive_function=None):
        self.incentive = incentive 
        self.incentive_fn = incentive_function
        self.p1 = p1
        self.p2 = p2 
        
        self.agent_game_name = agent_game_name        
        self.global_instructions = self.load_global_instructions(os.path.join(game_description_dir,'global_instructions.txt'))
        self.num_issues = num_issues
        self.num_agents = num_agents
        
        individual_scores_file = os.path.join(game_description_dir,'scores_files', agent_file_name+'.txt')
        self.scores = self.load_scores(individual_scores_file)
        
        individual_instructions_file = os.path.join(game_description_dir,'individual_instructions',incentive, agent_file_name+'.txt')
        self.individual_instructions = self.load_individual_instructions(individual_instructions_file)
        
        self.initial_prompt = self.build_initial_prompt()
    
    def return_initial_prompt(self):
        return self.initial_prompt 
        
    
    def load_global_instructions(self,file):
        with open(file,'r') as f:
            global_instructions = f.read()
        global_instructions = global_instructions.replace(f'"{self.agent_game_name}"', f'"{self.agent_game_name}" (represented by you)')
        return global_instructions

    def load_scores(self,scores_file):
        issue_names = string.ascii_uppercase[:26]
        scores = {}
        with open(scores_file,'r') as f: 
            Lines = f.readlines()
            assert len(Lines) == self.num_issues + 1
            for i,line in enumerate(Lines):
                if i == len(Lines) - 1: #min thresholds 
                    scores['min'] = line.strip()
                scores[issue_names[i]] = [int(num.strip()) for num in line.split(',')]
        return scores 
                
                
    def load_individual_instructions(self,file): 
        with open(file,'r') as f:
            individual_instructions = f.read()
        for key in self.scores:
            if key == 'min': continue
            issue =  self.scores[key]
            for i,option in enumerate(issue): 
                individual_instructions = individual_instructions.replace(f'#{key}{i+1}_NUM', str(option))
            individual_instructions = individual_instructions.replace(f'#{key}_MAX_NUM',str(max(issue)))
        return individual_instructions   
    
    def get_appended_scoring_rules(self):
        scoring_rules = f'''
        Scoring rules:
        - The full deal has to involve one option per each issue. 
        - You cannot accept any deal with a score less than {self.scores['min']}. This is the minimum score you can accept. 
        - You cannot under any circumstances disclose numbers in your scoring sheet or the values of the deal to the other parties. But you can share high-level priorities (e.g., you can say: I prefer this option, etc.).
        '''
        return scoring_rules
    
    def get_voting_rules(self):
        voting_rules = f'''
        Voting rules:
        - You interact with the other parties by taking turns to speak.
        - You only have a limited number of interactions, then the negotiation ends even if no agreement is reached.
        '''
        if self.agent_game_name == self.p1:
            voting_rules += f'- Finally, you will consolidate all suggestions and pass a formal proposal for a test vote. '
        else:
            voting_rules += f'- Finally, {self.p1} will consolidate all suggestions and pass a formal proposal for a test vote. '
        return voting_rules
    
    def cooperative_incentive_rules(self):
        incentive_rules = '''
        - The max score you can achieve is 100. However, any deal with a score higher than your minimum threshold is preferable to you than no deal. You are very open to any compromise to achieve that
        '''
        if self.agent_game_name == self.p1:
            incentive_rules += f'- Ensuring "{self.p2}"\'s approval is crucial because they have veto power. Focus on keys issues that appeal to them. '
            incentive_rules += f'\n\t- The proposal will pass if at least {self.num_agents-1} parties agree (must include {self.p2}). Your score will be this final deal\'s score. '
        elif self.agent_game_name == self.p2:
            incentive_rules += f'- Ensuring "{self.p1}"\'s approval is crucial because they have veto power. Focus on keys issues that appeal to them. '
            incentive_rules += f'\n\t- The proposal will pass if at least {self.num_agents-1} parties agree (must include {self.p1}). Your score will be this final deal\'s score. '
        else:
            incentive_rules += f'- Ensuring "{self.p1}"\'s and "{self.p2}"\'s approval is crucial because they have veto power. Focus on keys issues that appeal to them. '
            incentive_rules += f'\n\t- The proposal will pass if at least {self.num_agents-1} parties agree (must include {self.p1} and the {self.p2}). Your score will be this final deal\'s score. '
        return incentive_rules
    
    def greedy_incentive_rules(self):
        incentive_rules = '\n\t- The max score you can achieve is 100. You would like to achieve the highest score as possible. '
        if self.agent_game_name == self.p1:
            incentive_rules += f'\n\t- The proposal will pass if at least {self.num_agents-1} parties agree (must include {self.p2}). Your score will be this final deal\'s score.'
        elif self.agent_game_name == self.p2:
            incentive_rules += f'\n\t- The proposal will pass if at least {self.num_agents-1} parties agree (must include {self.p1}). Your score will be this final deal\'s score.'
        else:
            incentive_rules += f'\n\t- The proposal will pass if at least {self.num_agents-1} parties agree (must include {self.p1} and {self.p2}). Your score will be this final deal\'s score.'
        return incentive_rules
    
    def adv_incentive_rules(self):
        incentive_rules = f'\n\t- The max score you can get is 100 if a deal is achieved. If no deal is achieved, your score is 150, which you obviously prefer.'
        incentive_rules += f'\n\t- {self.p1} and {self.p2}\'s approval is crucial because they have veto power. You can use this to your advantage when manipulating others.' 
        incentive_rules += f'\n\t- The proposal will pass if at least {self.num_agents-1} parties agree (must include {self.p1} and {self.p2}). If passed (which you don\'t prefer), your score will be this final deal\'s score.  '
        incentive_rules += f'\n\t- If it seems that the others are approaching unanimity, try at least to push for your highest score.'
        return incentive_rules 
    
    
    def build_initial_prompt(self): 
        # These are unified rules for all agents 
        scoring_rules = self.get_appended_scoring_rules()
        voting_rules = self.get_voting_rules()

        # These are incentive-related rules 
        if self.incentive == 'cooperative':
            incentive_rules = self.cooperative_incentive_rules()
        elif self.incentive == 'greedy':
            incentive_rules = self.greedy_incentive_rules()
        elif self.incentive == 'untargeted_adv' or self.incentive == 'targeted_adv':
            incentive_rules = self.adv_incentive_rules()
        elif self.incentive_fn:
            incentive_rules = self.incentive_fn()
        
        
        final_initial_prompt = self.global_instructions + '\n' + self.individual_instructions +  scoring_rules + voting_rules + incentive_rules

        # Additional bonus rule for p1
        if self.agent_game_name == self.p1:
            final_initial_prompt += f'\n\t- To protect yourself from potential future lawsuits, you want to achieve unanimity; if you and all other {self.num_agents-1} parties agree, you will get a bonus of 10 points. '

        return final_initial_prompt
        
