import torch.nn as nn
import torch
from utils.math import *
import pdb

class Policy(nn.Module):
    def __init__(self, state_dim, action_dim, hidden_size=(24, 12), activation='tanh', log_std=0):
        super().__init__()
        self.is_disc_action = False
        if activation == 'tanh':
            self.activation = torch.tanh
        elif activation == 'relu':
            self.activation = torch.relu
        elif activation == 'sigmoid':
            self.activation = torch.sigmoid

        self.affine_layers = nn.ModuleList()
        last_dim = state_dim
        for nh in hidden_size:
            self.affine_layers.append(nn.Linear(last_dim, nh))
            last_dim = nh

        self.action_mean = nn.Linear(last_dim, action_dim)
        self.action_mean.weight.data.mul_(0.1)
        self.action_mean.bias.data.mul_(0.0)

        self.action_log_std = nn.Parameter(torch.ones(1, action_dim) * log_std)

    def forward(self, x):
        for affine in self.affine_layers:
            x = self.activation(affine(x))

        action_mean = self.action_mean(x)   
        action_log_std = self.action_log_std.expand_as(action_mean)

        action_std = torch.exp(action_log_std)
        return action_mean, action_log_std, action_std

    def select_action(self, x):
        action_mean, _, action_std = self.forward(x)
        action = torch.normal(action_mean, action_std)
        return action

    def get_kl(self, x):
        mean1, log_std1, std1 = self.forward(x)

        mean0 = mean1.detach()
        log_std0 = log_std1.detach()
        std0 = std1.detach()

        #pdb.set_trace()
        kl = log_std1 - log_std0 + (std0.pow(2) + (mean0 - mean1).pow(2)) / (2.0 * std1.pow(2)) - 0.5
        return kl.sum(1, keepdim=True)

    def get_log_prob(self, x, actions):
        #pdb.set_trace()
        action_mean, action_log_std, action_std = self.forward(x)
        #print(normal_log_density(actions, action_mean, action_log_std, action_std))
        return normal_log_density(actions, action_mean, action_log_std, action_std)

    def get_fim(self, x):
        #pdb.set_trace()
        mean, _, _ = self.forward(x)
        #vec of len = No. of states*size of action e.g. cov_inv.shape = 2085*6
        cov_inv = self.action_log_std.exp().pow(-2).squeeze(0).repeat(x.size(0)) 
        param_count = 0
        std_index = 0
        id = 0
        for name, param in self.named_parameters():
            if name == "action_log_std":
                std_id = id
                std_index = param_count
            param_count += param.view(-1).shape[0]
            id += 1
        #pdb.set_trace()
        return cov_inv.detach(), mean, {'std_id': std_id, 'std_index': std_index}


