'''
run using $python -m models.defense.baselines.MART.train_MART >& log/3may_mart.txt& 
from src
'''


from __future__ import print_function
import os
import argparse
import pickle
import torch
import torch.nn as nn
import torch.nn.functional as F
import torchvision
import torch.optim as optim
from torchvision import datasets, transforms
from torch.utils.data import DataLoader
from torch.autograd import Variable


from models.defense.baselines.MART.wideresnet import *
from models.defense.baselines.MART.resnet import *
from models.defense.baselines.MART.mart import mart_loss
import numpy as np
import time

# os.environ["CUDA_VISIBLE_DEVICES"]="0"


def train(args, model, device, train_loader, optimizer, epoch):
    model.train()
    for batch_idx, (data, target) in enumerate(train_loader):
        data, target = data.to(device), target.to(device)

        optimizer.zero_grad()

        # calculate robust loss
        loss = mart_loss(model=model,
                           x_natural=data,
                           y=target,
                           optimizer=optimizer,
                           step_size=args.step_size,
                           epsilon=args.epsilon,
                           perturb_steps=args.num_steps,
                           beta=args.beta)
        loss.backward()
        optimizer.step()

        # print progress
        if batch_idx % args.log_interval == 0:
            print('Train Epoch: {} [{}/{} ({:.0f}%)]\tLoss: {:.6f}'.format(
                epoch, batch_idx * len(data), len(train_loader.dataset),
                       100. * batch_idx / len(train_loader), loss.item()))

def adjust_learning_rate(optimizer, epoch):
    """decrease the learning rate"""
    lr = args.lr
    if epoch >= 100:
        lr = args.lr * 0.001
    elif epoch >= 90:
        lr = args.lr * 0.01
    elif epoch >= 75:
        lr = args.lr * 0.1
    for param_group in optimizer.param_groups:
        param_group['lr'] = lr
        
def _pgd_whitebox(model,
                  X,
                  y, 
                  args):
    epsilon=args.epsilon
    num_steps=args.num_steps
    step_size=args.step_size
    out = model(X)
    err = (out.data.max(1)[1] != y.data).float().sum()
    X_pgd = Variable(X.data, requires_grad=True)

    random_noise = torch.FloatTensor(*X_pgd.shape).uniform_(-epsilon, epsilon).to(device)
    X_pgd = Variable(X_pgd.data + random_noise, requires_grad=True)

    for _ in range(num_steps):
        opt = optim.SGD([X_pgd], lr=1e-3)
        opt.zero_grad()

        with torch.enable_grad():
            loss = nn.CrossEntropyLoss()(model(X_pgd), y)
        loss.backward()
        eta = step_size * X_pgd.grad.data.sign()
        X_pgd = Variable(X_pgd.data + eta, requires_grad=True)
        eta = torch.clamp(X_pgd.data - X.data, -epsilon, epsilon)
        X_pgd = Variable(X.data + eta, requires_grad=True)
        X_pgd = Variable(torch.clamp(X_pgd, 0, 1.0), requires_grad=True)
    err_pgd = (model(X_pgd).data.max(1)[1] != y.data).float().sum()
    return err, err_pgd

def eval_adv_test_whitebox(model, device, test_loader, args):

    model.eval()
    robust_err_total = 0
    natural_err_total = 0

    for data, target in test_loader:
        data, target = data.to(device), target.to(device)
        # pgd attack
        X, y = Variable(data, requires_grad=True), Variable(target)
        err_natural, err_robust = _pgd_whitebox(model, X, y, args)
        robust_err_total += err_robust
        natural_err_total += err_natural
    print('natural_acc: ', 1 - natural_err_total / len(test_loader.dataset))
    print('robust_acc: ', 1- robust_err_total / len(test_loader.dataset))
    return 1 - natural_err_total / len(test_loader.dataset), 1- robust_err_total / len(test_loader.dataset)


def main():

    if args.data.lower() == 'cifar10':
        model = NN_CIFAR10().to(device)
    elif args.data.lower() == 'fmnist':
        model = NN_MNIST().to(device)
    else: 
        raise NotImplementedError
    print("Using model:")
    print(model)

    optimizer = optim.SGD(model.parameters(), lr=args.lr, momentum=args.momentum, weight_decay=args.weight_decay)   
    
    natural_acc = []
    robust_acc = []
    
    for epoch in range(1, args.epochs + 1):
        # adjust learning rate for SGD
        adjust_learning_rate(optimizer, epoch)
        
        start_time = time.time()

        # adversarial training
        train(args, model, device, train_loader, optimizer, epoch)


        print('================================================================')

        natural_err_total, robust_err_total = eval_adv_test_whitebox(model, device, test_loader, args)

        print('using time:', time.time()-start_time)
        
        natural_acc.append(natural_err_total.detach().cpu().numpy())
        robust_acc.append(robust_err_total.detach().cpu().numpy())
        print('================================================================')
        
        file_name = os.path.join(log_dir, 'train_stats.npy')
        np.save(file_name, np.stack((np.array(natural_acc), np.array(robust_acc))))        

        # save checkpoint
        if epoch % args.save_freq == 0:
            model_save_path = os.path.join(model_dir, 'model-nn-epoch{}.pt'.format(epoch))
            torch.save(model.state_dict(), model_save_path)
            torch.save(optimizer.state_dict(),
                       os.path.join(model_dir, 'opt-nn-checkpoint_epoch{}.tar'.format(epoch)))
            remote_save_dir=None
            model_final_save_path = os.path.join(model_dir, 'model-final.pt')
            subprocess.run(["cp", model_save_path, model_final_save_path])
            subprocess.run(["scp", "-r", model_save_path, remote_save_dir])
            subprocess.run(["scp", "-r", model_final_save_path, remote_save_dir])
            
if __name__ == '__main__':

    parser = argparse.ArgumentParser(description='PyTorch CIFAR MART Defense')
    parser.add_argument('-d','--data', type=str, default='CIFAR10')
    parser.add_argument('--batch-size', type=int, default=128, metavar='N',
                        help='input batch size for training (default: 128)')
    parser.add_argument('--test-batch-size', type=int, default=100, metavar='N',
                        help='input batch size for testing (default: 100)')
    # parser.add_argument('--epochs', type=int, default=120, metavar='N',
    #                     help='number of epochs to train')
    parser.add_argument('--weight-decay', '--wd', default=3.5e-3,
                        type=float, metavar='W')
    # parser.add_argument('--lr', type=float, default=0.01, metavar='LR',
                        # help='learning rate')
    parser.add_argument('--momentum', type=float, default=0.9, metavar='M',
                        help='SGD momentum')
    parser.add_argument('--no-cuda', action='store_true', default=False,
                        help='disables CUDA training')
    # parser.add_argument('--epsilon', default=0.031,
                        # help='perturbation')
    # parser.add_argument('--num_steps', default=20,
    #                     help='perturb number of steps')
    # parser.add_argument('--step_size', default=0.007,
    #                     help='perturb step size')
    # parser.add_argument('--beta', default=5.0,
                        # help='weight before kl (misclassified examples)')
    parser.add_argument('--seed', type=int, default=1, metavar='S',
                        help='random seed (default: 1)')
    parser.add_argument('--log-interval', type=int, default=100, metavar='N',
                        help='how many batches to wait before logging training status')
    # parser.add_argument('--model', default='MART',
                        # help='directory of model for saving checkpoint')
    parser.add_argument('--save-freq', '-s', default=1, type=int, metavar='N',
                        help='save frequency')

    # if __name__ == '__main__':
    args = parser.parse_args()

    if args.data.lower() == 'cifar10':
        args.epsilon = 0.031
        args.num_steps = 20
        args.step_size = 0.007
        args.lr = 0.01
        args.beta = 5.0
        args.epochs = 120
    elif args.data.lower() == 'fmnist':
        args.epsilon = 0.3
        args.num_steps = 40
        args.step_size = 0.01
        args.lr = 0.01
        args.beta = 1.0
        args.epochs = 100
    else:
        raise NotImplementedError

    import path
    import sys
    curr_path = path.Path(__file__).abspath()
    # folder_path = curr_path.parent
    # expts_path = path.Path(folder_path.parent+"/adversarial-learning")
    # sys.path.append(expts_path)
    # sys.path.append(expts_path+"/src")
    # sys.path.append(expts_path+"/src/models")
    # from classifier_base import Classifier
    sys.path.append("../../../../")
    from models.defense.nn_mnist import NN_MNIST
    from models.defense.nn_cifar10 import NN_CIFAR10

    import subprocess

    # settings
    # model_dir = args.model
    model_dir = f'saved_models/baselines/{args.data}/MART'
    if not os.path.exists(model_dir):
        os.makedirs(model_dir)
        
    log_dir = './log'
    if not os.path.exists(log_dir):
        os.makedirs(log_dir)
        
    use_cuda = not args.no_cuda and torch.cuda.is_available()
    torch.manual_seed(args.seed)
    device = torch.device("cuda" if use_cuda else "cpu")
    kwargs = {'num_workers': 5, 'pin_memory': True} if use_cuda else {}
    torch.backends.cudnn.benchmark = True

    # setup data loader
    if args.data == 'cifar10':
        transform_test = transforms.Compose([
            transforms.ToTensor(),
        ])
        testset = torchvision.datasets.CIFAR10(root='../data', train=False, download=True, transform=transform_test)
    elif args.data == 'fmnist':
        testset = torchvision.datasets.FashionMNIST('data', download=True, train=False, transform=transforms.Compose([transforms.ToTensor()]))
    else: 
        raise NotImplementedError

    test_loader = torch.utils.data.DataLoader(testset, batch_size=args.test_batch_size, shuffle=False)

    pkl_path = f'dataset{"_"+args.data if args.data!="cifar10" else ""}_split.pkl'
    with open(pkl_path, 'rb') as f:
        dat = pickle.load(f)
        print("Loading datasets")
        
    trainset = dat["train_ds"]
    print('Trainset size: ', len(trainset))
    train_loader = torch.utils.data.DataLoader(trainset, batch_size=args.batch_size, shuffle=False)
    valset = dat["val_ds"]
    print('Valset size: ', len(valset))
    val_loader = torch.utils.data.DataLoader(valset, batch_size=args.batch_size, shuffle=False)


    main()
