# =================================================================================================#
# Description: Plots the experimental results for the timings
# Author: Ryan Thompson
# =================================================================================================#

import Cairo, ColorSchemes, CSV, DataFrames, Fontconfig, Gadfly, Pipe, Statistics

# Load data
result = CSV.read("Results/timings.csv", DataFrames.DataFrame)

# Summarise results for plotting
result = Pipe.@pipe result |>
         DataFrames.stack(_, [:time]) |>
         DataFrames.groupby(_, [:n, :m, :p, :estimator, :loss, :variable]) |>
         DataFrames.combine(
        _, 
        :value => Statistics.mean => :mean, 
        :value => (x -> Statistics.mean(x) - Statistics.std(x) / sqrt(size(x, 1))) => :low,
        :value => (x -> Statistics.mean(x) + Statistics.std(x) / sqrt(size(x, 1))) => :high
         )

# Plot timings as a function of n
Gadfly.plot(
    DataFrames.filter(:p => x -> x == 50, result),
    x = :n,
    y = :mean,
    ymin = :low,
    ymax = :high,
    Gadfly.Geom.line, 
    Gadfly.Geom.point, 
    Gadfly.Geom.yerrorbar,
    Gadfly.Coord.cartesian(xmin = 1000),
    Gadfly.Guide.xlabel("Sample size"), 
    Gadfly.Guide.ylabel("Run time"),
    Gadfly.Theme(default_color = "black", plot_padding = [0Gadfly.mm])
    ) |> 
    Gadfly.PDF("Figures/timings_n.pdf", 4.5Gadfly.inch, 2.1Gadfly.inch)

# Plot timings as a function of p
Gadfly.plot(
    DataFrames.filter(:p => x -> x > 50, result),
    x = :p,
    y = :mean,
    ymin = :low,
    ymax = :high,
    Gadfly.Geom.line, 
    Gadfly.Geom.point, 
    Gadfly.Geom.yerrorbar,
    Gadfly.Coord.cartesian(xmin = 100),
    Gadfly.Guide.xlabel("Number of explanatory features"), 
    Gadfly.Guide.ylabel("Run time"),
    Gadfly.Theme(default_color = "black", plot_padding = [0Gadfly.mm])
    ) |> 
    Gadfly.PDF("Figures/timings_p.pdf", 4.5Gadfly.inch, 2.1Gadfly.inch)