import numpy as np
from itertools import chain, combinations


# Generate the power set of a given iterable
def powerset(iterable):
    """Yield all subsets of the given iterable (as tuples)."""
    s = list(iterable)
    return chain.from_iterable(combinations(s, r) for r in range(len(s) + 1))

# Exact minimization via exhaustive search


def exact_minimization(F, n):
    """
    Evaluate F on every subset of the ground set {0,...,n-1}
    and return the minimum function value found.
    """
    subsets = powerset(range(n))
    min_value = float('inf')
    for S in subsets:
        val = F(S)
        if val < min_value:
            min_value = val
    return min_value


# Experiment to measure minimization error for submodular functions
if __name__ == "__main__":
    # Configuration
    n = 2        # number of ground elements
    T = 2000     # number of algorithm steps / oracle calls

    # Run experiments
    results = []
    for i in range(100):
        # Generate a random submodular function (requires generate_random_submodular_function)
        F = generate_random_submodular_function(n)

        # Average algorithm output over multiple runs to reduce variance
        s = 0
        for j in range(100):
            s += F(submodular_minimization(F, n, T))
        avg_alg_value = s / 100

        # Compute error = algorithm value - exact minimum
        results.append(avg_alg_value - exact_minimization(F, n))

    # Compute summary statistics
    results = np.array(results)
    mean_value = np.mean(results)
    std_dev = np.std(results)

    # Print results
    print(f"n: {n}")
    print(f"T: {T}")
    print(f"mean error: {mean_value}")
    print(f"std deviation: {std_dev}")
