import numpy as np


def sigmoid_submodular_minimization(F, n, T, b):
    """
    Submodular function minimization with sigmoid transfer sunction.

    Args:
        F (callable): Submodular function.
        n (int): Number of elements in the ground set.
        T (int): Total number of iterations.
        b (float): Sigmoid transfer parameter.

    Returns:
        set: The solution set S_T.
    """

    # Sigmoid transfer function mapping real values to [-1, 1]
    def sigmoid(x):
        return 2 / (1 + np.exp(-x)) - 1

    # Initialization of parameters
    psi = 1 / (1 + np.exp(-1))
    Cb = np.abs((2 * psi - 1) / (24 * psi**2 * (1 - psi)**2))
    # Number of samples per comparison
    k = int(Cb**0.4 * T**0.2 / n**0.2)
    # Effective number of iterations
    Td = int(T**0.8 / (n**0.8 * Cb**0.4))
    eta = (Cb**0.2) * (n**0.4) / (T**0.4)            # Step size
    w = np.full(n, 0.5)                              # Initial weight vector
    w_history = np.zeros((Td, n))                    # Store weight vectors

    for t in range(Td):
        pi = np.argsort(-w)      # Sort indices in descending order of weights
        g_t = np.zeros(n)        # Gradient estimate

        for i in range(n):
            # Construct subsets B_i and B_{i-1}
            B_i = set(pi[:i+1])
            B_i_minus_1 = set(pi[:i]) if i > 0 else set()

            # Oracle feedback with sigmoid smoothing
            delta_F = F(B_i) - F(B_i_minus_1)
            p = np.clip(sigmoid(delta_F), -1, 1)

            # Repeated sampling (majority vote) to reduce noise
            kp, km = 0, 0
            for j in range(k):
                if np.random.random() < (1 + p) / 2:
                    kp += 1
                else:
                    km += 1

            # Gradient estimate for coordinate pi[i]
            g_t[pi[i]] = np.log((kp + 0.5) / (km + 0.5))

        # Update and project onto [0,1]^n
        w -= eta * g_t
        np.clip(w, 0, 1)  # projection (in-place)

        # Record current weights
        w_history[t] = w

    # Compute averaged solution
    w_bar = np.mean(w_history, axis=0)

    # Random threshold z ~ U[0,1]
    z = np.random.uniform(0, 1)

    # Construct final set S_T
    S_T = {i for i in range(n) if w_bar[i] >= z}

    return S_T
